/*********************************************************************
	VuKitten.cpp
	
	This file contains the implementation of a kitten as a BeOS
	view.
	
	Who		When		What
	GJW		971022	Original coding and commenting.
	GJW		971024	Added graphics, finished replication code,
					tweaked performance slightly.
*********************************************************************/

//	Class Declaration
#include "VuKitten.h"

//	Types
struct roSColor { 
	float m_Red; 
	float m_Green; 
	float m_Blue; 
	float m_Alpha; 
	float m_Hue; 
};

//	Constants
const double VuKitten::kSmallAngle = atan(1) / 2;
const bigtime_t VuKitten::kShortDelay = 200000;
const bigtime_t VuKitten::kLongDelay = 400000;
#define RO_COLOR_STRUCT_128_TYPE 'roCr'

/*********************************************************************
	Instantiate
	Args:	inArchive - the message containing an archived kitten
	Returns:	a kitten created as per the archive
	This routine is the public instantiation function for archived
	kittens. Given a message, we look to see if it contains an
	archived kitten. If it does, we call the archive constructor.
*********************************************************************/
VuKitten* VuKitten::Instantiate(BMessage* inArchive)
{
	if(!validate_instantiation(inArchive,"VuKitten")) return NULL;
	return new VuKitten(inArchive);
}

#pragma mark -

/*********************************************************************
	VuKitten
	Args:	<none>
	Returns:	<none>
	The default constructor builds a new kitten based on some generic
	parameters.
*********************************************************************/
VuKitten::VuKitten(BRect inRect)
	: BView(inRect,"neko",B_FOLLOW_ALL_SIDES,B_WILL_DRAW)
{
	status_t status;
	
	app_info theInfo;
	status = be_app->GetAppInfo(&theInfo);
	BFile theFile(&theInfo.ref,B_READ_ONLY);
	BResources theResources(&theFile);
	
	size_t theSize;
	char* theBuffer = (char*)theResources.FindResource(B_MESSAGE_TYPE,128,&theSize);
	BMessage theMessage;
	status = theMessage.Unflatten(theBuffer);
	free(theBuffer);
	
	mKittens = new BBitmap(&theMessage);

	InitKitten();
	AddChild(new BDragger(BRect(0,0,7,7),this));
}


/*********************************************************************
	VuKitten
	Args:	inArchive - a message containing an archived kitten
	Returns:	<none>
	The archive constructor builds a new kitten from data stored in
	an archive message.
*********************************************************************/
VuKitten::VuKitten(BMessage* inArchive)
	: BView(inArchive), mKittens(NULL)
{
	//	Extract the kitten images
	BMessage theKittens;
	if(inArchive->FindMessage("neko_kittens",&theKittens) == B_OK)
		mKittens = new BBitmap(&theKittens);
	
	//	Don't let replicated views resize
	SetResizingMode(B_FOLLOW_NONE);

	InitKitten();
}


/*********************************************************************
	~VuKitten
	Args:	<none>
	Returns:	<none>
	The destructor is a place-holder for now. In the complete version,
	it will release the image(s) of the kitten.
*********************************************************************/
VuKitten::~VuKitten(void)
{
	mRunning = false;
	status_t status = kill_thread(mThread);
	delete mKittens;
}

#pragma mark -

/*********************************************************************
	Archive
	Args:	inMessage - the message into which we archive
			inDeep - true if we should archive subordinate objects
	Returns:	a status code for the archiving operation
	This routine supports replication of neko by archiving the
	view and the utility objects in a message for storage or
	transmission.
*********************************************************************/
status_t VuKitten::Archive(BMessage* inMessage, bool inDeep) const
{
	status_t status = inherited::Archive(inMessage,inDeep);
	if(status == B_OK)
	{
		status = inMessage->AddString("add_on","application/x-vnd.Bastion-replicat");
		if(inDeep)
		{
			BMessage theKittens;
			if(mKittens->Archive(&theKittens,inDeep) == B_OK)
				inMessage->AddMessage("neko_kittens",&theKittens); 
		}
	}
	return status;
}


/*********************************************************************
	Draw
	Args:	inUpdate - the area that needs to be redrawn
	Returns:	<none>
	This routine draws the current image of the kitten.
*********************************************************************/
void VuKitten::Draw(BRect /*inUpdate*/)
{
	if(mKittens != NULL)
	{
		BRect theRect(0,0,31,31);
		theRect.OffsetBy(32*(mCatFace%8),32*(mCatFace/8));
		DrawBitmap(mKittens,theRect,mCatBox);
	}
	else FillEllipse(mCatBox);
}


/*********************************************************************
	MessageReceived
	Args:	inMessage - the message we got
	Returns:	<none>
	This routine captures about requests and displays a system modal
	(grumble) dialog with copyright info.
*********************************************************************/
void VuKitten::MessageReceived(BMessage* inMessage)
{
	switch(inMessage->what)
	{
		case B_ABOUT_REQUESTED:
			{
				BAlert* alert = new BAlert("","Replicat 1.0 by Greg Weston","So?"); 
				alert->Go(NULL);
			}
			break;
			
		case B_PASTE:
			DoPaste(inMessage);
			break;
			
		default:
			inherited::MessageReceived(inMessage);
			break;
	}
}


/*********************************************************************
	MouseMoved
	Args:	inPoint - the current location of the mouse
			inTransit - the type of movement noted
			inMessage - the message (if any) being dragged
	Returns:	<none>
	This routine handles mouse movement over the view.
*********************************************************************/
void VuKitten::MouseMoved(BPoint inPoint, uint32 inTransit, const BMessage* /*inMessage*/)
{
	switch(inTransit)
	{
		case B_EXITED_VIEW:
			mMouseOutside = true;
			break;
		
		default:
			CalcVector(inPoint);
			mMouseMoved = true;
			mMouseOutside = false;
			break;
	}
}

#pragma mark -

/*********************************************************************
*********************************************************************/
void VuKitten::CalcVector(BPoint inPoint)
{
	mMouseLoc = inPoint;
	if(mMouseLoc.y < 32)
	{
		mMouseLoc.y = 32;
		mOutsidePos = kPosAbove;
	}
	else if(mMouseLoc.y > Bounds().bottom)
	{
		mMouseLoc.y = Bounds().bottom;
		mOutsidePos = kPosBelow;
	}
	if(mMouseLoc.x < 16)
	{
		mMouseLoc.x = 16;
		mOutsidePos = kPosLeft;
	}
	else if(mMouseLoc.x > Bounds().right - 16)
	{
		mMouseLoc.x = Bounds().right - 16;
		mOutsidePos = kPosRight;
	}
	int dx = mMouseLoc.x - mCatLoc.x;
	int dy = mCatLoc.y - mMouseLoc.y;
	mDistance = sqrt(dx*dx+dy*dy);
	mTheta = atan2(dy,dx);
	mSleepTime = kShortDelay;
}


/*********************************************************************
*********************************************************************/
void VuKitten::DoPaste(BMessage* inMessage)
{
	int32 numBytes = sizeof(struct roSColor);

	roSColor* pColorData;
	if(inMessage->FindData("roColour",RO_COLOR_STRUCT_128_TYPE,(void**)&pColorData,&numBytes) == B_OK)
	{
		roSColor color = *pColorData;

		if(inMessage->WasDropped()) 
		{
			SetViewColor(color.m_Red*255,color.m_Green*255,color.m_Blue*255);
			Invalidate();
		}
	}
}


/*********************************************************************
	InitKitten
	Args:	<none>
	Returns:	<none>
	This routine sets up the aspects of the kitten that are
	irrelevant of the constructor used.
*********************************************************************/
void VuKitten::InitKitten(void)
{
	//	Respect the kitten's transparency
	SetDrawingMode(B_OP_OVER);
	
	mMouseMoved = false;
	mMouseOutside = true;
	mSleepTime = kShortDelay;
	mEdgeCount = 0;
	mScratchCount = 0;
	mOutsidePos = kPosInside;
	mCatLoc = BPoint(16,32);
	mMouseLoc = BPoint(16,32);
	mCatBox.Set(0,0,31,31);
	mCatFace = kSitting;
	
	//	Fire up the animation thread
	mThread = spawn_thread(thread_proc,"neko",B_DISPLAY_PRIORITY,this);
	status_t status = resume_thread(mThread);
}


/*********************************************************************
	ThreadProc
	Args:	<none>
	Returns:	0
	This routine calculates the current image and location for neko
	in an infinite loop. Further documentation is included inline.
*********************************************************************/
int32 VuKitten::ThreadProc(void)
{
	mRunning = true;
	while(mRunning)
	{
		status_t status = snooze(mSleepTime);
/*		
		if(mMouseOutside)
		{
			BWindow* theWindow = Window();
			if(theWindow != NULL)
			{
				if(theWindow->Lock())
				{
					BPoint thePoint;
					uint32 theButtons;
					GetMouse(&thePoint,&theButtons); 
					CalcVector(thePoint);
					theWindow->Unlock();
				}
			}
		}
*/		
		if(mDistance > 16)
		{
			mSleepTime = kShortDelay;
			mCatLoc.x = mCatLoc.x + (cos(mTheta) * 16);
			mCatLoc.y = mCatLoc.y - (sin(mTheta) * 16);
			mDistance -= 16;
			
			if((mTheta >= -kSmallAngle) && (mTheta <= kSmallAngle))
				mCatFace = (mCatFace == kEast1) ? kEast2 : kEast1;
			if((mTheta > kSmallAngle) && (mTheta < 3*kSmallAngle))
				mCatFace = (mCatFace == kNorthEast1) ? kNorthEast2 : kNorthEast1;
			if((mTheta >= 3*kSmallAngle) && (mTheta <= 5*kSmallAngle))
				mCatFace = (mCatFace == kNorth1) ? kNorth2 : kNorth1;
			if((mTheta > 5*kSmallAngle) && (mTheta < 7*kSmallAngle))
				mCatFace = (mCatFace == kNorthWest1) ? kNorthWest2 : kNorthWest1;
			if((mTheta >= 7*kSmallAngle) || (mTheta <= -7*kSmallAngle))
				mCatFace = (mCatFace == kWest1) ? kWest2 : kWest1;
			if((mTheta > -7*kSmallAngle) && (mTheta <= -5*kSmallAngle))
				mCatFace = (mCatFace == kSouthWest1) ? kSouthWest2 : kSouthWest1;
			if((mTheta >= -5*kSmallAngle) && (mTheta <= -3*kSmallAngle))
				mCatFace = (mCatFace == kSouth1) ? kSouth2 : kSouth1;
			if((mTheta > -3*kSmallAngle) && (mTheta < -kSmallAngle))
				mCatFace = (mCatFace == kSouthEast1) ? kSouthEast2 : kSouthEast1;

			mMouseMoved = false;
		}
		else
		{
			mCatLoc = mMouseLoc;
			mSleepTime = kLongDelay;
			
			switch(mCatFace)
			{
				case kSitting:
					if(mMouseOutside)
					{
						switch(mOutsidePos)
						{
							case kPosAbove:	mCatFace = kAbove1;	break;
							case kPosBelow:	mCatFace = kBelow1;	break;
							case kPosLeft:	mCatFace = kLeft1;		break;
							case kPosRight:	mCatFace = kRight1;	break;
						}
						mOutsidePos = kPosInside;
						break;
					}
					mCatFace = kBathing;
					break;
					
				case kAbove1:
					mCatFace = kAbove2;
					if(++mEdgeCount == 6)
					{
						mCatFace = kScratch1;
						mEdgeCount = 0;
					}
					break;
				case kAbove2:
					mCatFace = kAbove1;
					break;
					
				case kBelow1:
					mCatFace = kBelow2;
					if(++mEdgeCount == 6)
					{
						mCatFace = kScratch1;
						mEdgeCount = 0;
					}
					break;
				case kBelow2:
					mCatFace = kBelow1;
					break;
					
				case kLeft1:
					mCatFace = kLeft2;
					if(++mEdgeCount == 6)
					{
						mCatFace = kScratch1;
						mEdgeCount = 0;
					}
					break;
				case kLeft2:
					mCatFace = kLeft1;
					break;
					
				case kRight1:
					mCatFace = kRight2;
					if(++mEdgeCount == 6)
					{
						mCatFace = kScratch1;
						mEdgeCount = 0;
					}
					break;
				case kRight2:
					mCatFace = kRight1;
					break;
					
				case kBathing:
					mCatFace = kSitting;
					if(++mEdgeCount == 6)
					{
						mCatFace = kScratch1;
						mEdgeCount = 0;
					}
					break;
				
				case kScratch1:
					mCatFace = kScratch2;
					break;
				case kScratch2:
					mCatFace = kScratch1;
					if(++mScratchCount == 4)
					{
						mCatFace = kYawning;
						mScratchCount = 0;
					}
					break;
					
				case kYawning:
					mCatFace = kSleep1;
					break;
					
				case kSleep1:
					mCatFace = kSleep2;
					break;
				case kSleep2:
					mCatFace = kSleep1;
					break;
				
				default:
					mCatFace = kSitting;
					break;
			}
			
			if(mMouseMoved)
			{
				mCatFace = kSurprise;
				mEdgeCount = 0;
				mScratchCount = 0;
				mMouseMoved = false;
			}
		}
		
		BWindow* theWindow = Window();
		if(theWindow != NULL)
		{
			if(theWindow->Lock())
			{
				Invalidate(mCatBox);
				mCatBox.Set(mCatLoc.x-16,mCatLoc.y-32,mCatLoc.x+15,mCatLoc.y-1);
				Invalidate(mCatBox);
				if(mMouseOutside)
				{
					BPoint thePoint;
					uint32 theButtons;
					GetMouse(&thePoint,&theButtons); 
					CalcVector(thePoint);
				}
				theWindow->Unlock();
			}
		}
	}
	return 0;
}

#pragma mark -

/*********************************************************************
	thread_proc
	Args:	inData - a pointer to the kitten object
	Returns:	the result of the thread execution
	This routine calls through to a member function in the object.
	That member does the real work of the thread.
*********************************************************************/
int32 VuKitten::thread_proc(void* inData)
{
	return ((VuKitten*)inData)->ThreadProc();
}
