/******************************************************************************/
/* MikConverter header file.                                                  */
/******************************************************************************/
/* This is part of the APlayer Programming Package (APPP).                    */
/* Copyright (C) 1998-1999 by The APlayer-Team.                               */
/* All rights reserved.                                                       */
/*                                                                            */
/* This source, or parts thereof, may only be used in APlayer related         */
/* software. If you want to use it elsewhere, please contact the author for a */
/* permission.                                                                */
/******************************************************************************/


#ifndef __MikConverter_h
#define __MikConverter_h

// PolyKit headers
#include "PFile.h"

// APlayer headers
#include "APAddOns.h"

// Needed headers
#include "MikAgent.h"
#include "MUniTrk.h"
#include "MikMod_Internals.h"


/******************************************************************************/
/* MikConverter class                                                         */
/******************************************************************************/
class MikConverter : public MUniTrk
{
public:
	MikConverter(uint32 strID);
	virtual ~MikConverter(void);

	virtual bool CheckModule(AfterLoadingBeforeCheckInfo *info) = 0;
	ap_result ConvertModule(AfterLoadingBeforeCheckInfo *info);
	virtual ap_result ModuleConverter(AfterLoadingBeforeCheckInfo *info) = 0;

	void FreeAll(void);

	ap_result ConvertToUniMod(PFile *source, PFile *dest);

protected:
	void ShowError(uint32 id);

	bool AllocPositions(int32 total);
	bool AllocPatterns(void);
	bool AllocTracks(void);
	bool AllocSamples(void);
	bool AllocInstruments(void);
	bool AllocLinear(void);
	void FreeLinear(void);

	void ReadComment(PFile *file, uint16 len);

	// Shared converter functions
	void S3MIT_ProcessCmd(uint8 cmd, uint8 inf, int32 oldEffect);
	void S3MIT_CreateOrders(int32 curious);
	int32 SpeedToFinetune(int32 speed, int32 sampNum);
	uint32 GetFrequency(uint8 flags, uint32 period);
	uint16 GetLinearPeriod(uint8 note, uint32 fine);

	uint32 strings;

	MODULE of;

	// S3M/IT variables
	int8 remap[64];
	uint8 *posLookup;

	uint8 posLookupCnt;
	uint16 *origPositions;

	bool filters;				// Resonant filters in use
	uint8 activeMacro;			// Active midi macro number for Sxx,xx < 80h
	uint8 filterMacros[16];		// Midi macros settings
	FILTER filterSettings[256];	// Computed filter settings

	int32 *noteIndex;
	int32 noteIndexCount;

	// User variables
	bool curious;
};



/******************************************************************************/
/* Mik669 class                                                               */
/******************************************************************************/
struct S69HEADER;
struct S69NOTE;

class Mik669 : public MikConverter
{
public:
	Mik669(uint32 strID) : MikConverter(strID) {};
	virtual ~Mik669(void) {};

	bool CheckModule(AfterLoadingBeforeCheckInfo *info);
	ap_result ModuleConverter(AfterLoadingBeforeCheckInfo *info);

protected:
	void LoadPatterns(PFile *file);

	S69HEADER *mh;
	S69NOTE *s69Pat;
};



/******************************************************************************/
/* MikAMF class                                                               */
/******************************************************************************/
struct AMFHEADER;
struct AMFNOTE;

class MikAMF : public MikConverter
{
public:
	MikAMF(uint32 strID) : MikConverter(strID) {};
	virtual ~MikAMF(void) {};

	bool CheckModule(AfterLoadingBeforeCheckInfo *info);
	ap_result ModuleConverter(AfterLoadingBeforeCheckInfo *info);

protected:
	bool UnpackTrack(PFile *file);
	uint8 *ConvertTrack(void);

	AMFHEADER *mh;
	AMFNOTE *track;
};



/******************************************************************************/
/* MikDSM class                                                               */
/******************************************************************************/
struct DSMSONG;
struct DSMNOTE;

class MikDSM : public MikConverter
{
public:
	MikDSM(uint32 strID) : MikConverter(strID) {};
	virtual ~MikDSM(void) {};

	bool CheckModule(AfterLoadingBeforeCheckInfo *info);
	ap_result ModuleConverter(AfterLoadingBeforeCheckInfo *info);

protected:
	void GetBlockHeader(PFile *file);
	void ReadPattern(PFile *file);
	uint8 *ConvertTrack(DSMNOTE *tr);

	DSMSONG *mh;
	DSMNOTE *dsmBuf;

	uint32	blockId;
	uint32	blockLn;
	uint32	blockLp;
};



/******************************************************************************/
/* MikFAR class                                                               */
/******************************************************************************/
struct FARHEADER1;
struct FARHEADER2;
struct FARNOTE;

class MikFAR : public MikConverter
{
public:
	MikFAR(uint32 strID) : MikConverter(strID) {};
	virtual ~MikFAR(void) {};

	bool CheckModule(AfterLoadingBeforeCheckInfo *info);
	ap_result ModuleConverter(AfterLoadingBeforeCheckInfo *info);

protected:
	uint8 *ConvertTrack(FARNOTE *n, int32 rows);

	FARHEADER1 *mh1;
	FARHEADER2 *mh2;
	FARNOTE *pat;
};



/******************************************************************************/
/* MikGDM class                                                               */
/******************************************************************************/
struct GDMHEADER;
struct GDMNOTE;

class MikGDM : public MikConverter
{
public:
	MikGDM(uint32 strID) : MikConverter(strID) {};
	virtual ~MikGDM(void) {};

	bool CheckModule(AfterLoadingBeforeCheckInfo *info);
	ap_result ModuleConverter(AfterLoadingBeforeCheckInfo *info);

protected:
	void ReadPattern(PFile *file);
	uint8 *ConvertTrack(GDMNOTE *tr);

	GDMHEADER *mh;
	GDMNOTE *gdmBuf;
};



/******************************************************************************/
/* MikIMF class                                                                */
/******************************************************************************/
struct IMFHEADER;
struct IMFNOTE;

class MikIMF : public MikConverter
{
public:
	MikIMF(uint32 strID) : MikConverter(strID) {};
	virtual ~MikIMF(void) {};

	bool CheckModule(AfterLoadingBeforeCheckInfo *info);
	ap_result ModuleConverter(AfterLoadingBeforeCheckInfo *info);

protected:
	void ReadPattern(PFile *file, int32 size, uint16 rows);
	void ProcessCmd(uint8 eff, uint8 inf);
	uint8 *ConvertTrack(IMFNOTE *tr, uint16 rows);

	IMFHEADER *mh;
	IMFNOTE *imfPat;
};



/******************************************************************************/
/* MikIT class                                                                */
/******************************************************************************/
struct ITHEADER;
struct ITNOTE;

class MikIT : public MikConverter
{
public:
	MikIT(uint32 strID) : MikConverter(strID) {};
	virtual ~MikIT(void) {};

	bool CheckModule(AfterLoadingBeforeCheckInfo *info);
	ap_result ModuleConverter(AfterLoadingBeforeCheckInfo *info);

protected:
	void GetNumChannels(PFile *file, uint16 patRows);
	uint8 *ConvertTrack(ITNOTE *tr, uint16 numRows);
	void ReadPattern(PFile *file, uint16 patRows);
	void LoadMidiString(PFile *file, char *dest);
	void LoadMidiConfiguration(PFile *file);

	ITHEADER *mh;
	ITNOTE *itPat;
	uint8 *mask;
	ITNOTE *last;
	uint32 *paraPtr;

	int32 numTrk;
	int32 oldEffect;
};



/******************************************************************************/
/* MikS3M class                                                               */
/******************************************************************************/
struct S3MHEADER;
struct S3MNOTE;

class MikS3M : public MikConverter
{
public:
	MikS3M(uint32 strID) : MikConverter(strID) {};
	virtual ~MikS3M(void) {};

	bool CheckModule(AfterLoadingBeforeCheckInfo *info);
	ap_result ModuleConverter(AfterLoadingBeforeCheckInfo *info);

protected:
	void GetNumChannels(PFile *file);
	void ReadPattern(PFile *file);
	uint8 *ConvertTrack(S3MNOTE *tr);

	S3MHEADER *mh;
	S3MNOTE *s3mBuf;
	uint16 *paraPtr;
};



/******************************************************************************/
/* MikSTM class                                                               */
/******************************************************************************/
struct STMHEADER;
struct STMNOTE;

class MikSTM : public MikConverter
{
public:
	MikSTM(uint32 strID) : MikConverter(strID) {};
	virtual ~MikSTM(void) {};

	bool CheckModule(AfterLoadingBeforeCheckInfo *info);
	ap_result ModuleConverter(AfterLoadingBeforeCheckInfo *info);

protected:
	void LoadPatterns(PFile *file);
	uint8 *ConvertTrack(STMNOTE *n);
	void ConvertNote(STMNOTE *n);

	STMHEADER *mh;
	STMNOTE *stmBuf;
};



/******************************************************************************/
/* MikSTX class                                                               */
/******************************************************************************/
struct STXHEADER;
struct STXNOTE;

class MikSTX : public MikConverter
{
public:
	MikSTX(uint32 strID) : MikConverter(strID) {};
	virtual ~MikSTX(void) {};

	bool CheckModule(AfterLoadingBeforeCheckInfo *info);
	ap_result ModuleConverter(AfterLoadingBeforeCheckInfo *info);

protected:
	void ReadPattern(PFile *file);
	uint8 *ConvertTrack(STXNOTE *tr);

	STXHEADER *mh;
	STXNOTE *stxBuf;
	uint16 *paraPtr;
};



/******************************************************************************/
/* MikULT class                                                               */
/******************************************************************************/
struct ULTEVENT;

class MikULT : public MikConverter
{
public:
	MikULT(uint32 strID) : MikConverter(strID) {};
	virtual ~MikULT(void) {};

	bool CheckModule(AfterLoadingBeforeCheckInfo *info);
	ap_result ModuleConverter(AfterLoadingBeforeCheckInfo *info);

protected:
	int32 ReadUltEvent(PFile *file, ULTEVENT *event);
};



/******************************************************************************/
/* MikUNI class                                                               */
/******************************************************************************/
struct UNISMP05;

class MikUNI : public MikConverter
{
public:
	MikUNI(uint32 strID) : MikConverter(strID) {};
	virtual ~MikUNI(void) {};

	bool CheckModule(AfterLoadingBeforeCheckInfo *info);
	ap_result ModuleConverter(AfterLoadingBeforeCheckInfo *info);

protected:
	uint8 *ReadTrack(PFile *file);
	void LoadSmp6(PFile *file);
	void LoadInstr6(PFile *file);
	void LoadInstr5(PFile *file);
	void LoadSmp5(void);

	UNISMP05 *wh;
	UNISMP05 *s;
	uint16 uniVersion;
};



/******************************************************************************/
/* MikXM class                                                                */
/******************************************************************************/
struct XMHEADER;
struct XMWAVHEADER;
struct XMNOTE;

class MikXM : public MikConverter
{
public:
	MikXM(uint32 strID) : MikConverter(strID) {};
	virtual ~MikXM(void) {};

	bool CheckModule(AfterLoadingBeforeCheckInfo *info);
	ap_result ModuleConverter(AfterLoadingBeforeCheckInfo *info);

protected:
	uint8 ReadNote(PFile *file, XMNOTE *n);
	uint8 *Convert(XMNOTE *xmTrack, uint16 rows);
	void LoadPatterns(PFile *file, bool dummyPat);
	void LoadInstruments(PFile *file);

	XMHEADER *mh;
	XMWAVHEADER *wh;
	XMWAVHEADER *s;
	uint32 *nextWav;
};

#endif
