//
// Calculator.cpp
//
// Implements calculator model
//

#include "Calculator.h"
#include "DvBeUtils.h"
#include <assert.h>

const int8 kMaxDigits = 10;

Calculator::Calculator()
{
	x = 0.0;
	y = 0.0;
	enterFlag = false;
	yFlag = false;
	operation = 0;
	digitsEntered = 0;
	decimal = 0;
	radix = RADIX_DECIMAL;
}

void Calculator::Clear()
{
	x = 0.0;
	digitsEntered = 0;
	decimal = 0;
}

void Calculator::ClearAll()
{
	x = 0.0;
	y = 0.0;
	yFlag = false;
	enterFlag = false;
	digitsEntered = 0;
	decimal = 0;
}

void Calculator::EnterDigit(const char* digitStr)
{
	if (enterFlag)
	{
		y = x;
		x = 0.0;
		digitsEntered = 0;
		enterFlag = false;
		decimal = 0;
	}
	
	// convert digit if necessary
	//
	double digit;
	switch (digitStr[0])
	{
		case 'A':
			digit = 10;
			break;
		case 'B':
			digit = 11;
			break;
		case 'C':
			digit = 12;
			break;
		case 'D':
			digit = 13;
			break;
		case 'E':
			digit = 14;
			break;
		case 'F':
			digit = 15;
			break;
		default:
			digit = digitStr[0] - '0';
			break;
	}
	
	if (digitStr[0] == '.')
	{
		decimal = 10.0;
	}
	else if (digitsEntered < kMaxDigits)
	{
		if (decimal > 0)
		{
			x = x + (digit / decimal);
			decimal *= 10.0;
		}
		else
		{
			//x = (x * 10.0) + digit;
			x = (x * radix) + digit;
		}
		++digitsEntered;
	}
}

void Calculator::EnterOp(Operator op)
{
	if (yFlag)	// if something is stored in y
	{
		switch (operation)
		{
			case OP_PLUS:
				x = y + x;
				break;
				
			case OP_SUBTRACT:
				x = y - x;
				break;
				
			case OP_MULTIPLY:
				x = y * x;
				break;
				
			case OP_DIVIDE:
				x = y / x;
				break;
				
			case OP_AND:
				x = uint32(y) & uint32(x);
				break;
				
			case OP_OR:
				x = uint32(y) | uint32(x);
				break;
			
			case OP_NOT:
				break;
				
			case OP_XOR:
				x = uint32(y) ^ uint32(x);
				break;
		}
	}

	y = x;
	yFlag = true;
	enterFlag = true;
	operation = op;
	decimal = 0;
}

static void IntToBinaryString(int32 i, char* buf, uint32 bufSize)
{
	uint32 mask = 0x80000000;
	bool firstOne = false;		// used to strip leading zeros
	uint32 len = 0;
	
	*buf = 0;
	do
	{
		if (i & mask)
		{
			if (++len < bufSize)
				strcat(buf, "1");
			firstOne = true;
		}
		else if (firstOne)
		{
			if (++len < bufSize)
				strcat(buf, "0");
		} 
		mask = mask >> 1;
	}
	while (mask > 0);
	
	if (*buf == 0 && bufSize > 1)
		strcpy(buf, "0");
}

void Calculator::GetResults(char* buf, uint32 bufSize)
{
	int32	result;
	
	switch (Radix())
	{
		case RADIX_BINARY:
			result = x;
			IntToBinaryString(result, buf, bufSize);
			break;
			
		case RADIX_OCTAL:
			result = x;
			sprintf(buf, "%o", result);
			break;
			
		case RADIX_HEX:
			result = x;
			sprintf(buf, "%X", result);
			break;
			
		case RADIX_DECIMAL:
		default:
			sprintf(buf, "%.10g", x);
			if (decimal > 0 && strchr(buf, '.') == 0)
				strcat(buf, ".");
		break;
	}
}

void Calculator::Not()
{
	x = ~int(x);
}

void Calculator::SetRadix(RADIX newRadix)
{
	if (radix != newRadix)
		radix = newRadix;
}

Calculator::RADIX Calculator::Radix()
{
	return radix;
}

void Calculator::UnaryMinus()
{
	x = -x;
}

void Calculator::SetX(double d)
{
	x = d;
}

double Calculator::X()
{
	return x;
}

