/* 

    TiMidity -- Experimental MIDI to WAVE converter
    Copyright (C) 1995 Tuukka Toivonen <toivonen@clinet.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    raw_audio.c

    Functions to output raw sound data to a file or stdout.

*/

#include <KernelKit.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>

#include "config.h"
#include "output.h"
#include "controls.h"
#include "tables.h"


static int open_output(void); /* 0=success, 1=warning, -1=fatal error */
static void close_output(void);
static void output_data(int32 *buf, int32 count);
static void flush_output(void);
static void purge_output(void);

/* export the playback mode */

#define dpm beos_play_mode

PlayMode dpm = {
  DEFAULT_RATE, PE_16BIT|PE_SIGNED,
  -1,
  {0,0,0,0,0},
  "beos output", 'b',
  "output.beos",
  open_output,
  close_output,
  output_data,
  flush_output,
  purge_output  
};

void s32tos8(int32 *lp, int32 c)
{
  int8 *cp=(int8 *)(lp);
  int32 l;
  while (c--)
    {
      l=(*lp++)>>(32-8-GUARD_BITS);
      if (l>127) l=127;
      else if (l<-128) l=-128;
      *cp++ = (int8) (l);
    }
}

void s32tou8(int32 *lp, int32 c)
{
  uint8 *cp=(uint8 *)(lp);
  int32 l;
  while (c--)
    {
      l=(*lp++)>>(32-8-GUARD_BITS);
      if (l>127) l=127;
      else if (l<-128) l=-128;
      *cp++ = 0x80 ^ ((uint8) l);
    }
}

void s32tos16(int32 *lp, int32 c)
{
  int16 *sp=(int16 *)(lp);
  int32 l;
  while (c--)
    {
      l=(*lp++)>>(32-16-GUARD_BITS);
      if (l > 32767) l=32767;
      else if (l<-32768) l=-32768;
      *sp++ = (int16)(l);
    }
}

void s32tou16(int32 *lp, int32 c)
{
  uint16 *sp=(uint16 *)(lp);
  int32 l;
  while (c--)
    {
      l=(*lp++)>>(32-16-GUARD_BITS);
      if (l > 32767) l=32767;
      else if (l<-32768) l=-32768;
      *sp++ = 0x8000 ^ (uint16)(l);
    }
}

void s32tos16x(int32 *lp, int32 c)
{
  int16 *sp=(int16 *)(lp);
  int32 l;
  while (c--)
    {
      l=(*lp++)>>(32-16-GUARD_BITS);
      if (l > 32767) l=32767;
      else if (l<-32768) l=-32768;
      *sp++ = XCHG_SHORT((int16)(l));
    }
}

void s32tou16x(int32 *lp, int32 c)
{
  uint16 *sp=(uint16 *)(lp);
  int32 l;
  while (c--)
    {
      l=(*lp++)>>(32-16-GUARD_BITS);
      if (l > 32767) l=32767;
      else if (l<-32768) l=-32768;
      *sp++ = XCHG_SHORT(0x8000 ^ (uint16)(l));
    }
}

void s32toulaw(int32 *lp, int32 c)
{
  uint8 *up=(uint8 *)(lp);
  int32 l;
  while (c--)
    {
      l=(*lp++)>>(32-13-GUARD_BITS);
      if (l > 4095) l=4095;
      else if (l<-4096) l=-4096;
      *up++ = _l2u[l];
    }
}

/*************************************************************************/

#define OPEN_MODE O_WRONLY | O_CREAT | O_TRUNC

static int open_output(void)
{
  if (dpm.encoding & PE_ULAW)
    dpm.encoding &= ~PE_16BIT;

  if (!(dpm.encoding & PE_16BIT))
    dpm.encoding &= ~PE_BYTESWAP;

  return 0;
}

extern int 	 	 stream_size;
extern char 	*stream_data;
extern sem_id 	 stream_sem1;
extern sem_id 	 stream_sem2;

static void output_data(int32 *buf, int32 count)
{
  	if (!(dpm.encoding & PE_MONO)) count*=2; /* Stereo samples */  
  	if (dpm.encoding & PE_16BIT)
    {
      	if (dpm.encoding & PE_BYTESWAP)
		{
	  		if (dpm.encoding & PE_SIGNED)
	    		s32tos16x(buf, count);
	  		else
	    		s32tou16x(buf, count);
		}
      	else
		{
	  		if (dpm.encoding & PE_SIGNED)
	    		s32tos16(buf, count);
	  		else 
	    		s32tou16(buf, count);
		}
		
		int down=0;
		do
		{
			acquire_sem(stream_sem1);
			for(int c=0;c<(stream_size/2);c++)
			{
				((short *)stream_data)[c]=((short *)buf)[c+down];
			}
			down+=stream_size/2;
			release_sem(stream_sem2);
		}
		while(down<count);
	}
  	else
    {
     	if (dpm.encoding & PE_ULAW)
		{
	  		s32toulaw(buf, count);
		}
      	else
		{
	  		if (dpm.encoding & PE_SIGNED)
	    		s32tos8(buf, count);
	  		else 
	    		s32tou8(buf, count);
		}
		int down=0;
		do
		{
			acquire_sem(stream_sem1);
			for(int c=0;c<(stream_size);c++)
			{
				((char *)stream_data)[c]=((char *)buf)[c+down];
			}
			down+=stream_size;
			release_sem(stream_sem2);
		}
		while(down<count);
    }
}

static void close_output(void)
{
}

static void flush_output(void) { }
static void purge_output(void) { }
