// This is a sample plugin snippet for Eddie
//
// (c) 1998 P.Cisler
//
// Any portion of this code may be used freely to develop Eddie
// plugins 
//
// The plugin sample shows you how to:
//
// 		install a plugin with a button
//		handle a double-click on a specific string
//		swap a button bitmap after a plugin state change
//		show text in a clipboard panel during a button press
//		handle a message - intercept drag&drop of a bitmap and paste as hex-dump
//		use single composite undo for multiple insert operations
//
//
// This plugin is pretty much useless, it is just intended as a code
// snippet to be cut & pasted into your code (even though the icon pasting is kind of fun)


#include <Alert.h>
#include <Bitmap.h>

#include "Plugin.h"

extern const unsigned char kBlankButtonBits[];
extern const unsigned char kPressedBlankButtonBits[];
extern const unsigned char kUnderscoreButtonBits[];
extern const unsigned char kPressedUnderscoreButtonBits[];

extern void DumpBitmap(PluginInterface *, const BBitmap *);


PluginInterface::PluginResult 
PluginMain(PluginInterface::PluginCallSelector selector, 
	PluginInterface *pluginInterface)
{
	PluginInterface::PluginResult result = PluginInterface::kIgnored;
	
	switch (selector) {

		case PluginInterface::kGetPluginVersion:
			// this is the first selector call a plugin will get
			// Just include the following two lines to let Eddie know the
			// version of plugin API you support and the version you require
			// from Eddie
			//
			// this is the only selector call your plugin is required to support
			//
			pluginInterface->ReturnSupportedPluginVersion();
			pluginInterface->ReturnRequiredPluginVersion();
			result = PluginInterface::kAccepted;
			break;

		case PluginInterface::kInit:
			// a button may allocate any permanent data structures it will
			// need during it's lifetime and initialize it's state
			// 
			// ask for any non-default selector calls you may need
			// these include:
			// kButtonDraw		- if you want to draw the button yourself instead of just
			//					  passing an icon
			// kPulse			- if you want to get called periodically
			// kMessage			- if you want to get a peek at every message sent to the
			//					  text view
			// kMouseDown		- if you want to get called every time the mouse is
			//					  pressed over a text view
			// kKeyDown			- if you want to get called every time a key is hit in a
			//					  text view
			// kDocumentChanged	- if you want to get called every time the document changes
			//
			
			// in this plugin intercept mouse clicks
			pluginInterface->RequireSelectors(PluginInterface::kRequestMouseDownMask
				| PluginInterface::kRequestDocumentChangedMask
				| PluginInterface::kRequestMessageMask
				| PluginInterface::kRequestPulseMask);
		
			// if your plugin supports any keyboard shortcuts, Init is used to publish the
			// editor primitives used by the shortcuts:
			pluginInterface->RegisterHandledPrimitiveCommand("InvokePluginSample1",
				"This is the description for the first shortuct defined by this sample plugin");
			pluginInterface->RegisterHandledPrimitiveCommand("InvokePluginSample2",
				"This is the description for the second shortuct defined by this sample plugin");

			// if your plugin handles any settings, register them here; see KeyPresshandling.cpp
			// for example
			// ...

			result = PluginInterface::kAccepted;
			break;

		case PluginInterface::kWindowOpened:
			// a new window opened
			// the pointer to the window may be obtained from
			// pluginInterface->Window()
			//
			// if your plugin has a button, now is the time to make sure it gets instaled
			pluginInterface->SetupPluginButton(kBlankButtonBits, kPressedBlankButtonBits);			
			// the above is for a simple solid button, if you want a split button, you
			// may use the following:
			// pluginInterface->SetupPluginButton(normalBitmap, pressedBitmap, kVerticallySplitButton);
			// and include the selector for the splitting you need
			//
			result = PluginInterface::kAccepted;
			break;

		case PluginInterface::kMouseDown:
			BPoint point;
			ulong buttons;
			static float lastClickTime = 0.0;
			pluginInterface->GetMouse(&point, &buttons);
			
			BRect documentBounds(pluginInterface->TextViewBounds());
			
			if (!documentBounds.Contains(point)
				// clicks outside the text view are not interresting
				|| buttons != B_PRIMARY_MOUSE_BUTTON)
					// only respond to left mouse button
				break;
			
			bigtime_t clickSpeed;
			get_click_speed(&clickSpeed);

			if (system_time() - lastClickTime > clickSpeed) {
				lastClickTime = system_time();
				// only interrested in double click
				break;
			}

			long pos = pluginInterface->PointToOffset(point);
			char ch = pluginInterface->CharAt(pos);
			if (ch != '_')
				break;

			// clicked on an '_' do something here
			
			(new BAlert("", "You like underscores, huh?", "Yup"))->Go();
	
			// change the bitmaps of the button, now that we double-clicked an underscore
			pluginInterface->ChangeButtonBitmaps(kUnderscoreButtonBits, kPressedUnderscoreButtonBits);

			// tell Eddie that we already took care of the mouse click
			result = PluginInterface::kExhausted;
			break;

		case PluginInterface::kButtonClick:
			// the plugin button was clicked
			// if you have a split button, use the call:
			//	pluginInterface->ClickSelector()
			// to determine which part was clicked
			result = PluginInterface::kAccepted;
			break;

		case PluginInterface::kButtonPress:
			// the plugin button was pressed, a press is a long click;
			// typically you will use this call if you want to open a popup menu, etc.
			//
			// show a clipboard panel with some fun text in it:
			//
			char buffer[512];
			sprintf(buffer, "%s%s %s",
				pluginInterface->ReadOnly() ? "ReadOnly " : "",
				pluginInterface->IsShellWindow() ?
					(pluginInterface->IsWorksheet() ? "Worksheet" : "Shell") : "Document",
				pluginInterface->DocumentDirty() ? "Needs Saving" : "Saved");

			pluginInterface->ShowClipboardPanel(buffer);					
					 
			result = PluginInterface::kAccepted;
			break;

		case PluginInterface::kPrimitiveInvoked:
			// handle a primitive from a keyboard command
			const char *primitive = pluginInterface->CurrentPrimitive();
			
			// figure out which of our multiple shortcuts was pressed
			if (strcmp(primitive, "InvokePluginSample1") == 0)
				/*DoSomething1()*/;
			else if (strcmp(primitive, "InvokePluginSample2") == 0)
				/*DoSomething2()*/;

			// animate the button as if it was pressed by a mouse to give the user
			// a positive feedback
			pluginInterface->AnimateButtonClick();
			result = PluginInterface::kAccepted;
			break;

		case PluginInterface::kArgvPassed:
			// read a setting from the settings file
			// ...
			break;
			
		case PluginInterface::kGlobalSettingsSaved:
			// save our settings into the settings file
			// ...
			break;

		case PluginInterface::kGlobalPrefsShowing:
			// install ourselves into the preferences dialog
			// ...
			break;

		case PluginInterface::kPulse:
			// document getting a pulse event
			// used for periodic checking of some document state, etc.
			// document that is getting pulsed is identified by:
			// pluginInterface->DocumentUniqueID();
			// To receive this selector call, plugin has to request it in kInit by calling
			// pluginInterface->RequireSelectors(PluginInterface::kRequestPulseMask | ... other masks);
			// Plugin should only request this selector call if it needs it to not
			// slow down the editor unnecessarily
			//
			// Do nothing in the sample plugin, just ignore, case only for illustration
			// if you don't need this selector, don't request it and just leave it out
			//
			result = PluginInterface::kIgnored;
			break;
			
		case PluginInterface::kMessage:
			{
				// handle drag&drop of a bitmap by pasting the bitmap in a
				// hex-dump form
				
				result = PluginInterface::kIgnored;
				BMessage *message = pluginInterface->CurrentMessage();
				
				if (!message->WasDropped())
					// not a drag&drop message, ignore
					break;
				
				// try three bitmap formats
				const char *kFormats[] = {
					"image/x-vnd.Be-bitmap",	// generic BBitmap 
					"icon/large",				// large icon from FileTypes, etc.
					"icon/mini",				// mini icon from FileTypes, etc.
				};

				for (int32 index = 0; index < 3; index++) {
					BMessage iconMessage;
					if (message->FindMessage(kFormats[index], &iconMessage) == B_OK) {

						BBitmap *bitmap = static_cast<BBitmap *>
							(BBitmap::Instantiate(&iconMessage));

						if (bitmap && bitmap->BitsLength() < 10 * 1024)
							// if bitmap instantiated fine and not too large, paste into
							// document
							DumpBitmap(pluginInterface, bitmap);
						
						delete bitmap;
						// got an actual bitmap, mark message as handled by plugin
						result = PluginInterface::kExhausted;
					}
				}
			}
			break;
			
		case PluginInterface::kDocumentTextChanged:
			// text in a document changed, as a result of an edit
			// (text typed, deleted, pasted, inserted, etc.)
			// document identified by:
			// pluginInterface->DocumentUniqueID()
			// changed text:
			// start: pluginInterface->ChangeStart()
			// size of removed text: pluginInterface->RemovedLength()
			// size of deleted text: pluginInterface->InsertedLength()
			//
			// Do nothing in the sample plugin, just ignore, case only for illustration
			// if you don't need this selector, don't request it and just leave it out
			//
			result = PluginInterface::kIgnored;
			break;
			
		case PluginInterface::kDocumentStateChanged:
			// document became read-only, etc.
			break;

		case PluginInterface::kDocumentClosed:
			// a document closed
			// delete data structures/caches associated with the document
			//
			// document being closed is identified by:
			// pluginInterface->DocumentUniqueID()
			//
			// Do nothing in the sample plugin, just ignore, case only for illustration;
			// if you don't need this selector, just leave it out
			//
			result = PluginInterface::kIgnored;
			break;

		case PluginInterface::kUnload:
			// clean up our act
			// delete permanent plugin data structures that we allocated
			// during the course of the plugins lifetime
			//
			// Do nothing in the sample plugin, just accept, case only for illustration
			// if you don't need this selector, just leave it out
			//
			result = PluginInterface::kAccepted;
			break;
	}
	return result;
}

const char *
PluginHelp(void)
{
	return "This is a sample Eddie plugin\n";
}


void 
DumpBitmap(PluginInterface *pluginInterface, const BBitmap *bitmap)
{	
	// start an undo record
	int32 selStart, selEnd;
	pluginInterface->GetSelection(&selStart, &selEnd);
	pluginInterface->StartUndo(selStart, selEnd, "Drop bitmap");

#define kPrefix "const unsigned char kBits [] = {"

	// insert the bitmap bits into the editor
	pluginInterface->CurrentUndoInsert(kPrefix, sizeof(kPrefix));

	int32 bytesPerPixel;

	switch (bitmap->ColorSpace()) {
		case B_GRAYSCALE_8_BIT:
		case B_COLOR_8_BIT:
			bytesPerPixel = 1;
			break;
		case B_RGB_16_BIT:
			bytesPerPixel = 2;
			break;
		case B_BIG_RGB_32_BIT:
			bytesPerPixel = 3;
			break;
		default:
			return;
	}

	const unsigned char *bits = (const unsigned char *)bitmap->Bits();
	const int32 kMaxColumnWidth = 16;
	int32 bytesPerRow = bitmap->BytesPerRow();
	int32 columnWidth = (bytesPerRow < kMaxColumnWidth) ? bytesPerRow : kMaxColumnWidth;
	
	char buffer[256];
	for (int32 remaining = bitmap->BitsLength(); remaining; ) {

		strcpy(buffer, "\n\t");
		int32 bufferIndex = 2;

		for (int32 column = 0; column < columnWidth; column++) {
		
			// stream out individual pixel components
			for (int32 count = 0; count < bytesPerPixel; count++) {
				--remaining;
				sprintf(&buffer[bufferIndex], "0x%02x%s", *bits++, remaining ? "," : "");
				bufferIndex += strlen(&buffer[bufferIndex]);
			}
			
			if (!remaining)
				break;

		}
		// buffer now contains a line of output, insert it into the document
		pluginInterface->CurrentUndoInsert(buffer, bufferIndex);
	}

	// insert closing brackets and newlines
	pluginInterface->CurrentUndoInsert("\n};\n\n", 5);
	// we are done, close undo
	pluginInterface->SealUndo();
}

const unsigned char kBlankButtonBits [] = {
	0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1c,
	0x1f,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1c,0x17,
	0x1f,0x1e,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1a,0x17,
	0x1f,0x1e,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1a,0x17,
	0x1f,0x1e,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1a,0x17,
	0x1f,0x1e,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1a,0x17,
	0x1f,0x1e,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1a,0x17,
	0x1f,0x1e,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1a,0x17,
	0x1f,0x1e,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1a,0x17,
	0x1f,0x1e,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1a,0x17,
	0x1f,0x1e,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1a,0x17,
	0x1f,0x1e,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1a,0x17,
	0x1f,0x1e,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1a,0x17,
	0x1f,0x1e,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1a,0x17,
	0x1f,0x1c,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x17,0x17,
	0x1c,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x15
};

const unsigned char kPressedBlankButtonBits [] = {
	0x15,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x1c,
	0x17,0x15,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1c,0x1f,
	0x17,0x1a,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1e,0x1f,
	0x17,0x1a,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1e,0x1f,
	0x17,0x1a,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1e,0x1f,
	0x17,0x1a,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1e,0x1f,
	0x17,0x1a,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1e,0x1f,
	0x17,0x1a,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1e,0x1f,
	0x17,0x1a,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1e,0x1f,
	0x17,0x1a,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1e,0x1f,
	0x17,0x1a,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1e,0x1f,
	0x17,0x1a,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1e,0x1f,
	0x17,0x1a,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1e,0x1f,
	0x17,0x1a,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1e,0x1f,
	0x17,0x1c,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x3f,0x1f,
	0x1c,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x3f
};

const unsigned char kUnderscoreButtonBits [] = {
	0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1c,
	0x1f,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1c,0x17,
	0x1f,0x1e,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1a,0x17,
	0x1f,0x1e,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1a,0x17,
	0x1f,0x1e,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1a,0x17,
	0x1f,0x1e,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1a,0x17,
	0x1f,0x1e,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1a,0x17,
	0x1f,0x1e,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1a,0x17,
	0x1f,0x1e,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1a,0x17,
	0x1f,0x1e,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1a,0x17,
	0x1f,0x1e,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1a,0x17,
	0x1f,0x1e,0x1c,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x1c,0x1a,0x17,
	0x1f,0x1e,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1a,0x17,
	0x1f,0x1e,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1a,0x17,
	0x1f,0x1c,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x17,0x17,
	0x1c,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x15
};

const unsigned char kPressedUnderscoreButtonBits [] = {
	0x15,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x1c,
	0x17,0x15,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1c,0x1f,
	0x17,0x1a,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1e,0x1f,
	0x17,0x1a,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1e,0x1f,
	0x17,0x1a,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1e,0x1f,
	0x17,0x1a,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1e,0x1f,
	0x17,0x1a,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1e,0x1f,
	0x17,0x1a,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1e,0x1f,
	0x17,0x1a,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1e,0x1f,
	0x17,0x1a,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1e,0x1f,
	0x17,0x1a,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1e,0x1f,
	0x17,0x1a,0x1c,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x1a,0x1e,0x1f,
	0x17,0x1a,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1e,0x1f,
	0x17,0x1a,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1e,0x1f,
	0x17,0x1c,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x3f,0x1f,
	0x1c,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x3f
};
