


#include "Parser.h"
#include <stdio.h>


void 	GetString( char *string);
void 	HandleParseError( ParseErrorMsg error );
double	ParseString( char *string);
void 	Underline( short start, short end );


void main()
{
	char 					string[256];
	

	printf( "\n\nExpression Parser\n\n\t\tby Ian Russell Ollmann\n\n");
	printf( "A C++ object for parsing numerical expressions with variables.\n\n");
	printf( "\n\nInstructions:\n ");
	printf( "Type in a function below. Hit ^C to quit.\n\n");
	printf( "Currently supported functions are +-*/%^ as well as sin, cos, tan, \n");
	printf( "asin, acos, atan, sinh, cosh, tanh, log, ln, exp, abs, sqrt and rand. \n");
	printf( "Also supported are parentheses to arbitrary depths (subject to machine \n");
	printf( "memory limitations) and numerical constants such as 3, 234.2, 1e-12. \n");
	printf( "The parser should be case insensitive, except where variables are \n");
	printf( "concerned, and ignore spaces. \n\n");
	printf( "Please report all bugs to Ian Ollmann (iano@scripps.edu) \nor if that fails, ");
	printf( "report them to BeDevTalk@be.com.\n\n");

	
	do
	{	
		printf( "\nEnter a new function: \n" );	
		
	//Read in the function
		GetString( string );
		
	//Parse the function	
		ParseString( string ); 
				
	}
	while (*string );
}


void GetString( char *string )
{
	int 	i = 0;
	int 	c;
	
	c = getchar();		
	while(  c != EOF && c != '\r' && c != '\n' && i < 255 )
	{
		string[i++] = c;
		c = getchar();
	}
	string[i] = '\0';

}

//
// Note -- the ParseString function is easier to write than it looks. 
// It is simply way over commented in this case and as a result, looks 
// complicated.
//
double ParseString( char *string )
{
	FunctionInterpreter		*interpreter = NULL;
	ParseInfo				parseInfo;
	double					value;

// The FunctionInterpreter class will throw exceptions in the 
// constructor if there is a syntax error. Both the constructor
// and the Evaluate function may throw exceptions in response to
// division by zero errors, square root of negative numbers, etc. 
// caught at evaluation time (the constructor evaluates constant
// portions of the function ahead of time to speed the Evaluate()
// call later.) For this reason, it is IMPERATIVE that you bracket 
// all new and Evaluate() calls with try{} and catch all of the 
// error types listed below

//Create a new Function Interpreter
	try 
	{
		//String points to the beginning of the whole string
		parseInfo.string = string;
		
		//segmentStart points to the first char of the string segment that is to be parsed
		parseInfo.segmentStart = string;

		//segmentEnd points to the last char of the string segment that is to be parsed
		parseInfo.segmentEnd = string + strlen( string ) - 1;
		
		//Init the new interpreter. Note that new is overloaded to throw a pOutOfMemory error if 
		//allocation fails, so we don't have to check for it here.
		interpreter = new FunctionInterpreter( &parseInfo );
	}
	
	//Note, this function can recursively call itself, which is why
	catch( ParseErrorMsg error )
	{
		HandleParseError( error );
		
		delete interpreter;
		interpreter = NULL;
		return 0;
	}
		
					
//Handle variables if there are any
	{
		int		variableCount;
		int		i;
		char	*string;
		char 	variableString[256];
		double	variableValue;
		
		//Get the number of variables
		variableCount = interpreter->GetNumberOfVariables();
		
		//Loop once for each variable
		for( i = 0; i < variableCount; i++ )
		{
			//Get the variable name
			string = interpreter->GetVariableName(i); //string now holds a pointer to a NULL terminated string. Note that if the index is out of bounds, NULL will be returned. We will skip checking for that error here. 
		
			//Print out a request for its value
			printf( "\nPlease enter a value for %s:", string );
			
			//input the new value from the command line
			GetString( variableString );
			
			//Convert the new value string to a double.  
				//Note: One could recursively call ParseString to accomplish this!
				//This will allow the user to type in whole expressions for each constant, 
				//possibly including more variables. Recursively operating in this fashion will have the 
				//slightly annoying side effect of having to reenter the value for the same constant 
				//if another instance of it appears on a different level, unless you implement your own
				//routine to scan for variable reuse. Recursively calling ParseString() also brings about 
				//the more difficult problem of what to do with syntax errors (with an obvious but inelegant 
				//solution of rethrowing errors after catching and handling them.) For simplicity, in this 
				//example, I've chosen the inelegant solution of resorting to the standard C sscanf to read
				//in constants.
			variableValue = atof( variableString );
			
			//Give the variable value to the intepreter
			interpreter->SetVariableValue( variableValue, i);  //Note that if i is out of bounds, this function will return false. 
		}
	}
	
	
//Evaluate the function
			
	try		// Don't forget to to bracket Evaluate() with a try{} !
	{
		value = interpreter->Evaluate();
		printf(" = %f\n", value );	
	}

	catch( ParseErrorMsg error )
	{
		HandleParseError( error );
	}


//Kill the old function to make room for the new - Each FunctionInterpreter object may only be used with a single string
	
	delete interpreter;
	
	
//Technically this line is unnecessary, but if you'd like to paste this function straight into your program, then it is
//handy
	
	return value;
}


//This function draws a ^---^ under the region passed to it delimited by start and end
void Underline( short start, short end )
{
	short 	i = 0;

	while ( i < end )
	{
		if( i < start )
			printf( " ");
		else
		{
			if( i == start )
				printf("^");
			else
				printf("-");
		}
		i++;
	}
	printf("^\n");
}





//Error Handling
	// The ParseErrorMsg catch type must be implemented. Otherwise if 
	// the parser throws an uncaught exception in response to a syntax error
	// it will terminate your program. All of the possible error types are listed below.
void HandleParseError( ParseErrorMsg error )
{
	// In the case of an out of memory error, SqrtNegNumber and DivByZero,  
	// error.errorInfo is undefined, and does not hold a set of text offsets.
	
	// For everything else, low and hiOffset(error.errorInfo) denote the string 
	// offsets between which the error occurs

	switch( error.errorType )
	{
		case pDivisionByZero:
			printf("\nError: Division by zero. \n\n");
			break;
		case pSqrtNegNumber:
			printf( "\nError: Square root of a negative number is not allowed.\n\n");
			break;
		case pLogNegNumber:
			printf( "\nError: Log of a negative number is not allowed.\n\n");
			break;
		case pLnNegNumber:
			printf( "\nError: Ln of a negative number is not allowed.\n\n");
			break;
		case pOutOfMemory:
			printf( "\nError: Out of Memory.\n" );
			break;
		case pArcSinOutOfBounds:
			printf( "\nError: ArcSine of a |number| > 1 not allowed.\n\n");
			break;
		case pArcCosOutOfBounds:
			printf( "\nError: ArcCosine of a |number| > 1 not allowed.\n\n");
			break;
		case pEmptyExpression:
			Underline( error.startOffset, error.endOffset );
			printf("Syntax Error. Empty Expression. \n\n");
			break;
		case pMissingOperator:
			Underline( error.startOffset, error.endOffset );
			printf("Syntax Error. Operator anticipated. \n\n");
			break;
		case pSyntaxError:
			Underline( error.startOffset, error.endOffset );
			printf("Syntax Error.\n\n");
			break;
		case pUnbalancedParen:
			Underline( error.startOffset, error.endOffset );
			printf("Error: Unbalanced parenthesis.\n\n");
			break;
		case pUnknownFunction:
			Underline( error.startOffset, error.endOffset );
			printf("Error: Unknown Function type.\n\n");
			break;
		case pUnknownOperator:
			Underline( error.startOffset, error.endOffset );
			printf("Error: Unknown Operator type.\n\n");
			break;
		default:
			printf( "Unknown Error Type!\n\n");
			break;
	}
}