#include "NodeMonitorHelpers.h"
#include <Messenger.h>
#include <Node.h>
#include <stdio.h>
#include <assert.h>
#include <DataIO.h>
#include <string.h>

/////
bool sNodeMonitorDebugging = true;

//////////////

class NodeMonitorMessenger : public BLooper {
public:
	NodeMonitorMessenger();
	
	virtual void MessageReceived(BMessage *msg);

private:
	BMallocIO mLastMessage;
	ssize_t mLastMessageLength;
};

class NodeMonitorManager : public BMessenger {
public:
	static NodeMonitorManager *Instance();

	void AddMonitor(NodeMonitor *which);
	void RemoveMonitor(NodeMonitor *which);

	NodeMonitor *FindMonitorFor(const node_ref &ref);
	void MessageReceived(BMessage *msg);

protected:
	static bool Iterator(void *monitor, void *arguments);		
	static NodeMonitor *sIteratorRC;	// valid since this is a single-threaded message loop

	BList nodeMonitors;
	
	NodeMonitorManager(BLooper *forMessages);
	~NodeMonitorManager();
};

//////////////

NodeMonitorManager *sMonitor;

NodeMonitor *NodeMonitorManager::sIteratorRC = NULL;

NodeMonitorManager::NodeMonitorManager(BLooper *forMessages)
	: BMessenger(forMessages) {
}

NodeMonitorManager::~NodeMonitorManager() {
	stop_watching(*this);
}

NodeMonitorManager *NodeMonitorManager::Instance() {
	if (sMonitor==NULL)
		sMonitor = new NodeMonitorManager(new NodeMonitorMessenger());
	return sMonitor;
}

void NodeMonitorManager::AddMonitor(NodeMonitor *which) {
	nodeMonitors.AddItem(which);
}

void NodeMonitorManager::RemoveMonitor(NodeMonitor *which) {
	nodeMonitors.RemoveItem(which);
}

NodeMonitor *NodeMonitorManager::FindMonitorFor(const node_ref &ref) {
	sIteratorRC = NULL;
	nodeMonitors.DoForEach(Iterator, (void*)&ref);
	return sIteratorRC;
}

bool NodeMonitorManager::Iterator(void *param, void *arguments) {
	node_ref *ref = (node_ref*)arguments;
	NodeMonitor *monitor = (NodeMonitor*) param;

	if (monitor->mTargetEntry == *ref) {
		sIteratorRC = monitor;
		return true;
	} else {
		return false;
	}
}

////////////

NodeMonitor::NodeMonitor(const BStatable &toWatch):
	mWatched(&toWatch)
{
	mStatus = toWatch.GetNodeRef((node_ref*)&mTargetEntry);
	NodeMonitorManager *mgr = NodeMonitorManager::Instance();

	if (sNodeMonitorDebugging) {
		if (mTargetEntry.device == -1) {
			printf("ERROR: Node monitoring failed, because your BStatable is not valid. InitCheck says %s.\n", strerror(mStatus));
			return;
		} else {
			if (mgr->FindMonitorFor(mTargetEntry))  {
				printf("WARNING: Monitoring set twice for the same node.  One of your classes will not be called.\n");
				return;
			}
			printf("%ld:%Ld - Beginning monitoring of node.\n", mTargetEntry.device, mTargetEntry.node);
		}
	}
	
	mgr->AddMonitor(this);

	watch_node(&mTargetEntry, B_WATCH_ALL|B_WATCH_DIRECTORY, *((BMessenger*)mgr));
}

NodeMonitor::~NodeMonitor() {
	if (mTargetEntry.device == -1)
		return;
		
	if (sNodeMonitorDebugging)
		printf("%ld:%Ld - Terminating monitoring of node.\n", mTargetEntry.device, mTargetEntry.node);

	NodeMonitorManager *mgr = NodeMonitorManager::Instance();
	mgr->RemoveMonitor(this);
	watch_node(&mTargetEntry, B_STOP_WATCHING, *((BMessenger*)mgr));
}

status_t NodeMonitor::InitCheck() {
	return mStatus;
}

void NodeMonitor::EntryDeleted(node_ref &fromDirectory) {
	delete this;
}

void NodeMonitor::EntryMoved(node_ref &from_directory, node_ref &to_directory, const char *name) {
}

void NodeMonitor::StatChanged() {
}

void NodeMonitor::AttributeChanged(const char *which) {
}

// implementation of utility functions
BDirectory NodeMonitor::GetDirectory(node_ref &directory_ref) {
	BDirectory result;
	result.SetTo(&directory_ref);
	return result;
}

BEntry NodeMonitor::GetEntry(node_ref &directory_ref, const char *name) {
	entry_ref rc = GetEntryRef(directory_ref, name);
	return BEntry(&rc);
}

entry_ref NodeMonitor::GetEntryRef(node_ref &from_directory, const char *name) {
	entry_ref rc; 

	rc.device = from_directory.device;
	rc.directory = from_directory.node;
	rc.set_name(name);
	return rc;
}



//////////

NodeMonitorMessenger::NodeMonitorMessenger() {
	mLastMessageLength = 0;
	Run();
}

void NodeMonitorMessenger::MessageReceived(BMessage *msg) {
	/* determine if we've seen this message before.
	why? because if the message has arrived, then more than one entity was interested in
	the event.
	*/
	BMallocIO currentMessage;
	ssize_t currentMessageLength;
	msg->Flatten(&currentMessage, &currentMessageLength);
	
	if (currentMessageLength==mLastMessageLength &&
	    !memcmp(currentMessage.Buffer(), mLastMessage.Buffer(), currentMessageLength)) {
		return;
	} else {
		mLastMessage.Seek(0, SEEK_SET);
		msg->Flatten(&mLastMessage, &mLastMessageLength);
	}

	// what opcode is this for?
	int32 opcode;
	msg->FindInt32("opcode", &opcode);

	// configure the node refs if we can
	node_ref node;
	node_ref directory;
	node_ref fromDirectory;
	node_ref toDirectory;
	const char *name;
	
	msg->FindInt32("device", &node.device); 
	directory.device = fromDirectory.device = toDirectory.device = node.device;

	status_t sNode = msg->FindInt64("node", &node.node);
	status_t sDirectory = msg->FindInt64("directory", &directory.node);
	status_t sFrom = msg->FindInt64("from directory", &fromDirectory.node);
	status_t sTo = msg->FindInt64("to directory", &toDirectory.node);
	msg->FindString("name", &name);

	// get a handle of the relevant node monitors if we found the entries
	NodeMonitor *foundNode;
	DirectoryMonitor *foundDirectory, *foundFrom, *foundTo;
	foundNode = foundDirectory = foundFrom = foundTo = NULL;
	
	if (sNode == B_NO_ERROR)
		foundNode = NodeMonitorManager::Instance()->FindMonitorFor(node);
	if (sDirectory == B_NO_ERROR)
		foundDirectory = dynamic_cast<DirectoryMonitor*> (NodeMonitorManager::Instance()->FindMonitorFor(directory));
	if (sFrom == B_NO_ERROR)
		foundFrom = dynamic_cast<DirectoryMonitor*> (NodeMonitorManager::Instance()->FindMonitorFor(fromDirectory));
	if (sTo == B_NO_ERROR)
		foundTo = dynamic_cast<DirectoryMonitor*> (NodeMonitorManager::Instance()->FindMonitorFor(toDirectory));

	// send notifications	
	switch (opcode) {
		case B_ENTRY_CREATED:
			assert(foundDirectory);
	
			if (sNodeMonitorDebugging)
				printf("%ld:%Ld - Entry '%s' (node %Ld) was created in monitored directory.\n", directory.device, directory.node, name, node.node);
			foundDirectory->EntryCreated(name, node.node);
		
			break;

		case B_ENTRY_REMOVED:
			if (foundNode) {
				if (sNodeMonitorDebugging)
					printf("%ld:%Ld - Monitored node was deleted from directory node %Ld.\n", node.device, node.node, directory.node);
				foundNode->EntryDeleted(directory);
			}
			
			if (foundDirectory) {
				if (sNodeMonitorDebugging)
					printf("%ld:%Ld - Node %Ld deleted from monitored directory.\n", directory.device, directory.node, node.node);
				foundDirectory->EntryRemoved(node.node);
			}
			break;

		case B_ENTRY_MOVED:
			if (foundNode) {
				if (sNodeMonitorDebugging)
					printf("%ld:%Ld - Monitored entry was moved from directory node %Ld to %Ld (new name '%s').\n", node.device, node.node, fromDirectory.node, toDirectory.node, name);
				foundNode->EntryMoved(fromDirectory, toDirectory, name);
			}
			
			if (foundFrom) {
				if (sNodeMonitorDebugging)
					printf("%ld:%Ld - Monitored directory lost entry %Ld: moved into directory node %Ld (new name '%s').\n", fromDirectory.device, fromDirectory.node, node.node, toDirectory.node, name);
				foundFrom->EntryMovedOut(toDirectory, name, node.node);
			}
			
			if (foundTo) {
				if (sNodeMonitorDebugging)
					printf("%ld:%Ld - Monitored directory received entry '%s' (node %Ld) from directory node %Ld.\n", toDirectory.device, toDirectory.node, name, node.node, fromDirectory.node);
				foundTo->EntryMovedIn(fromDirectory, name, node.node);
			}
			break;

		case B_STAT_CHANGED:
			assert(foundNode);
			if (sNodeMonitorDebugging)
				printf("%ld:%Ld - Monitored node's stat changed.\n", node.device,node.node);
			foundNode->StatChanged();
			break;
		
		case B_ATTR_CHANGED:
			assert(foundNode);
			
			const char *whichAttribute;
			msg->FindString("attr", &whichAttribute);
			

			if (sNodeMonitorDebugging)
				printf("%ld:%Ld - Monitored node's attribute '%s' changed.\n", node.device,node.node, whichAttribute);
			foundNode->AttributeChanged(whichAttribute);
			break;
		
		default:
			printf("**** what a surprising place to be.\n");
			break;		
	}
}

//////////////////////////

DirectoryMonitor::DirectoryMonitor(const BDirectory &toWatch) :
	NodeMonitor(toWatch) {	
}

void DirectoryMonitor::EntryCreated(const char *created_name, ino_t created_node) {
}

void DirectoryMonitor::EntryRemoved(ino_t node) {
}

void DirectoryMonitor::EntryMovedIn(node_ref &from_directory, const char *name_of_node, ino_t node) {
}
	
void DirectoryMonitor::EntryMovedOut(node_ref &to_directory, const char *name_of_node, ino_t node_of_node) {
}
