/* ++++++++++
	driver.c
	A skeletal device driver
+++++ */
#include <KernelExport.h>
#include <Drivers.h>
#include <Errors.h>
#include <OS.h>
#include <string.h>
#include <stdlib.h>


#define RAM_DRIVE_SIZE			(16*1024*1024) // size of the RAMDrive in bytes






static void format_ram_drive(void* buf);
static uchar* create_ram_drive_area(size_t drive_size);
static status_t delete_ram_drive_area(void);
static void emulate_seek(off_t pos);


#define RAM_DRIVE_RELEASE_MEMORY	(B_DEVICE_OP_CODES_END+1)
#define RAM_DRIVE_EMULATE_SEEK		(B_DEVICE_OP_CODES_END+2)

#define RAM_BLOCK_SIZE			512
#define MAX_SEEK_TIME			1000.0 /* microseconds */
#define	PREFETCH_BUFFER_SIZE	(32*1024)

static const char* const ram_drive_area_name =  "RAM drive area"; 
uchar icon_disk [] = {
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0x00,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x00,0xff,0x00,
	0x15,0x00,0x00,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x00,0x3f,0x00,0x15,
	0x3f,0x15,0x15,0x00,0x00,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x00,0x00,0x15,0x3f,
	0x15,0x15,0x15,0x16,0x15,0x00,0x00,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x00,0xff,0x00,0x15,0x3f,0x15,
	0x15,0x15,0x16,0x15,0x15,0x15,0x16,0x00,0x00,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x00,0x3f,0x00,0x15,0x3f,0x15,0x15,
	0x15,0x16,0x15,0x15,0x15,0x16,0x15,0x15,0x15,0x00,0x00,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x00,0x00,0x15,0x3f,0x15,0x15,0x15,
	0x16,0x15,0x15,0x15,0x16,0x15,0x15,0x15,0x16,0x3f,0x0e,0x00,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x00,0xff,0x00,0x15,0x3f,0x15,0x15,0x15,0x16,
	0x15,0x15,0x15,0x16,0x15,0x15,0x15,0x16,0x00,0x00,0x00,0x00,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0x00,0x3f,0x00,0x15,0x3f,0x15,0x15,0x15,0x16,0x15,
	0x15,0x15,0x16,0x15,0x15,0x15,0x16,0x3f,0x00,0x3f,0x3f,0x00,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0x00,0x00,0x15,0x3f,0x15,0x15,0x15,0x16,0x15,0x15,
	0x15,0x16,0x15,0x15,0x15,0x16,0x3f,0x00,0x00,0x00,0x3f,0x00,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0x00,0xff,0x00,0x15,0x3f,0x15,0x15,0x15,0x16,0x15,0x15,0x15,
	0x16,0x15,0x15,0x15,0x16,0x00,0x00,0x00,0x00,0x00,0x15,0x00,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0x00,0x3f,0x00,0x15,0x3f,0x15,0x15,0x15,0x16,0x15,0x15,0x15,0x16,
	0x15,0x15,0x15,0x16,0x3f,0x00,0x3f,0x3f,0x00,0x00,0x15,0x00,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0x00,0x00,0x15,0x3f,0x15,0x15,0x16,0x16,0x15,0x15,0x15,0x16,0x15,
	0x16,0x15,0x15,0x3f,0x00,0x00,0x00,0x3f,0x00,0x00,0x15,0x00,0x0f,0xff,0xff,0xff,
	0xff,0xff,0xff,0x00,0x15,0x3f,0x15,0x15,0x16,0x16,0x15,0x15,0x15,0x16,0x15,0x15,
	0x15,0x16,0x00,0x00,0x00,0x00,0x00,0x15,0x00,0x00,0x15,0x00,0x0f,0xff,0xff,0xff,
	0xff,0xff,0x00,0x15,0x3f,0x15,0x15,0x15,0x16,0x15,0x15,0x15,0x16,0x16,0x15,0x15,
	0x15,0x3f,0x00,0x3f,0x3f,0x00,0x00,0x15,0x00,0x00,0x15,0x00,0x0f,0xff,0xff,0xff,
	0xff,0xff,0x00,0x3f,0x3f,0x15,0x15,0x16,0x15,0x15,0x15,0x16,0x15,0x15,0x16,0x15,
	0x3f,0x00,0x00,0x00,0x3f,0x00,0x00,0x15,0x00,0x0f,0x00,0x0f,0xff,0xff,0xff,0xff,
	0xff,0xff,0x00,0x00,0x15,0x3f,0x3f,0x15,0x16,0x15,0x16,0x15,0x15,0x15,0x15,0x00,
	0x00,0x00,0x00,0x00,0x16,0x00,0x00,0x15,0x00,0x0f,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0x00,0x15,0x00,0x00,0x15,0x3f,0x3f,0x16,0x15,0x15,0x15,0x16,0x3f,0x00,
	0x3f,0x3f,0x00,0x00,0x15,0x00,0x00,0x15,0x00,0x0f,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0x00,0x15,0x38,0x35,0x00,0x00,0x15,0x3f,0x3f,0x15,0x15,0x3f,0x00,0x00,
	0x00,0x3f,0x00,0x00,0x15,0x00,0x0f,0x00,0x0f,0x0e,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0x00,0x00,0x35,0x35,0xeb,0x2c,0x00,0x00,0x15,0x3f,0x3f,0x0e,0x0f,0x00,
	0x00,0x16,0x00,0x00,0x15,0x00,0x0f,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0x00,0x00,0x2c,0x2c,0x15,0x15,0x00,0x00,0x15,0x0f,0x00,0x16,
	0x00,0x15,0x00,0x00,0x15,0x00,0x0f,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0x00,0x00,0x15,0x15,0x15,0x15,0x00,0x00,0x15,0x0f,
	0x00,0x15,0x00,0x0f,0x00,0x0f,0x0f,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x00,0x00,0x15,0x15,0x15,0x00,0x0f,0x00,
	0x00,0x16,0x00,0x0e,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x00,0x00,0x15,0x0f,0x00,0x0f,
	0x00,0x15,0x00,0x0f,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x00,0x00,0x0e,0x0f,
	0x0f,0x00,0x0f,0x0f,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff
}
;







//uchar	icon_disk_mini[B_MINI_ICON * B_MINI_ICON];
uchar icon_disk_mini [] = {
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0x00,0x00,0x00,0x00,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0x00,0x00,0x3f,0x15,0x00,0x00,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0x00,0x00,0x00,0x3f,0x15,0x15,0x15,0x16,0x00,0x00,0xff,0xff,
	0xff,0xff,0xff,0xff,0x00,0x00,0x3f,0x15,0x15,0x15,0x16,0x15,0x15,0x15,0x00,0xff,
	0xff,0xff,0x00,0x00,0x00,0x3f,0x15,0x15,0x15,0x16,0x15,0x15,0x00,0x00,0x00,0xff,
	0xff,0xff,0x00,0x00,0x3f,0x15,0x15,0x15,0x16,0x15,0x15,0x3f,0x15,0x00,0x00,0xff,
	0xff,0xff,0x00,0x3f,0x15,0x15,0x15,0x16,0x16,0x15,0x00,0x00,0x08,0x00,0xff,0xff,
	0xff,0x00,0x3f,0x15,0x15,0x15,0x16,0x15,0x15,0x3f,0x15,0x00,0x00,0x00,0xff,0xff,
	0xff,0x00,0x3f,0x3f,0x15,0x16,0x15,0x15,0x00,0x00,0x08,0x00,0xff,0x00,0xff,0xff,
	0xff,0x00,0x0e,0x34,0x3f,0x3f,0x16,0x3f,0x15,0x00,0x00,0x00,0xff,0xff,0xff,0xff,
	0xff,0xff,0x00,0x00,0x2a,0x0e,0x3f,0x0e,0x09,0x00,0xff,0x00,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0x00,0x00,0x0e,0x09,0x00,0x00,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0x00,0x00,0xff,0x00,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff
}
;


int	emulate_seek_flag = TRUE;
uchar * ram = NULL;

static const char *vd_name[] = {
	"disk/virtual/ram_drive",
		NULL
};


/* ----------
	init_hardware - called once the first time the driver is loaded
----- */
status_t
init_hardware (void)
{	 		
	return B_OK;
}


/* ----------
	init_driver - optional function - called every time the driver
	is loaded.
----- */
status_t
init_driver (void)
{	dprintf("vd driver: %s %s, init_driver()\n", __DATE__, __TIME__);
	
	ram = create_ram_drive_area(RAM_DRIVE_SIZE);
	dprintf("vd driver: ram_buffer=%08x\n", ram);	
	if(ram == NULL)
		return B_ERROR;
	return B_OK;
}


/* ----------
	uninit_driver - optional function - called every time the driver
	is unloaded
----- */
void
uninit_driver(void)
{
	dprintf("vd driver: uninit_driver()\n"); 
}


	
/* ----------
	my_device_open - handle open() calls
----- */

static status_t
my_device_open (const char *name, uint32 flags, void** cookie)
{
	dprintf("vd driver: open(%s)\n", name); 
    return B_OK;
}


/* ----------
	my_device_read - handle read() calls
----- */

static status_t
my_device_read (void* cookie, off_t pos, void *buf, size_t* count)
{
	size_t len;
	status_t ret = B_NO_ERROR;
	
	if(pos >= RAM_DRIVE_SIZE)
	{
		len = 0;
	}
	else
	{
		len = (pos + (*count) > RAM_DRIVE_SIZE) ? (RAM_DRIVE_SIZE - pos) : (*count);   
		emulate_seek(pos);
		memcpy(buf, ram+pos, len);
	}
	*count = len;
	return ret;
}


/* ----------
	my_device_write - handle write() calls
----- */

static status_t
my_device_write (void* cookie, off_t pos, const void* buf, size_t* count)
{
	size_t len;
	status_t ret = B_NO_ERROR;

	if(pos >= RAM_DRIVE_SIZE)
	{
		len = 0;
	}
	else
	{
		len = (pos + (*count) > RAM_DRIVE_SIZE) ? (RAM_DRIVE_SIZE - pos) : (*count);   
		emulate_seek(pos);
		memcpy(ram+pos, buf, len);
	}
	*count = len;
	return ret;
}


/* ----------
	my_device_control - handle ioctl calls
----- */

static status_t
my_device_control (void* cookie, uint32 msg, void* arg1, size_t len)
{
	device_geometry	*dinfo;

	dprintf("vd driver: control(%d)\n", msg); 
	switch (msg) 
	{
	case B_GET_GEOMETRY:
		dinfo = (device_geometry *) arg1;
		
		dinfo->sectors_per_track = RAM_DRIVE_SIZE/RAM_BLOCK_SIZE;
		dinfo->cylinder_count    = 1;
		dinfo->head_count        = 1;
		
		dinfo->bytes_per_sector = RAM_BLOCK_SIZE;
		dinfo->removable = FALSE ;
		dinfo->read_only = FALSE;
		dinfo->device_type = B_DISK;
		dinfo->write_once = FALSE;	
		
		return B_NO_ERROR;
		
	case B_FORMAT_DEVICE:
		if (!*((char *) arg1))
			return B_NO_ERROR;

		format_ram_drive(ram);		
		return B_NO_ERROR;
		
	case B_GET_DEVICE_SIZE:
		*(size_t*)arg1 = RAM_DRIVE_SIZE;
		return B_NO_ERROR;
		
	case B_GET_ICON:
		switch (((device_icon *)arg1)->icon_size)
		{
		case 32: //B_LARGE_ICON:
			memcpy(((device_icon *)arg1)->icon_data, icon_disk,
				32*32);//B_LARGE_ICON * B_LARGE_ICON);
			break;
			
		case 16://B_MINI_ICON:
			memcpy(((device_icon *)arg1)->icon_data, icon_disk_mini,
				16*16);//B_MINI_ICON * B_MINI_ICON);
			break;
			
		default:
			memcpy(((device_icon *)arg1)->icon_data, icon_disk,
				32*32);//B_LARGE_ICON * B_LARGE_ICON);
			break;
//			return B_BAD_TYPE;
		}
		return B_NO_ERROR;

	/* device specific IO control codes */	
	case RAM_DRIVE_RELEASE_MEMORY:
		return delete_ram_drive_area(); 

	case RAM_DRIVE_EMULATE_SEEK:
		emulate_seek_flag = *(int*)arg1;
		return B_NO_ERROR; 
		
	default:
		return B_ERROR;
	}
}


/* ----------
	my_device_close - handle close() calls
----- */

static status_t
my_device_close (void* cookie)
{
	return B_OK;
}


/* -----
	my_device_free - called after the last device is closed, and after
	all i/o is complete.
----- */
static status_t
my_device_free (void* cookie)
{
	return B_OK;
}


/* -----
	null-terminated array of device names supported by this driver
----- */

static const char *my_device_name[] = {
	"disk/virtual/ram_drive",
	NULL
};

/* -----
	function pointers for the device hooks entry points
----- */

device_hooks my_device_hooks = {
	my_device_open, 			/* -> open entry point */
	my_device_close, 			/* -> close entry point */
	my_device_free,			/* -> free cookie */
	my_device_control, 		/* -> control entry point */
	my_device_read,			/* -> read entry point */
	my_device_write			/* -> write entry point */
};

/* ----------
	publish_devices - return a null-terminated array of devices
	supported by this driver.
----- */

const char**
publish_devices()
{
	return my_device_name;
}

/* ----------
	find_device - return ptr to device hooks structure for a
	given device name
----- */

device_hooks*
find_device(const char* name)
{
	return &my_device_hooks;
}


static void format_ram_drive(void* buf)
{
	static const char format_str[16] = "RAM drive      ";
	uchar* ptr = (uchar*)buf;
	off_t i;
	
	dprintf("vd driver: format_ram_drive(%08x)\n", buf); 
	for(i=0; i<RAM_DRIVE_SIZE/16; i++)
	{
		memcpy(ptr, format_str, 16);
		ptr += 16;
	}
	
}


static uchar* 
create_ram_drive_area(size_t drive_size)
{
	void* addr;
	area_id area = find_area(ram_drive_area_name);
	
	if(area == B_NAME_NOT_FOUND)
	{
		area = create_area (ram_drive_area_name, 
        	&addr, 
        	B_ANY_KERNEL_ADDRESS, /* kernel team will own this area */ 
        	drive_size, 
        	B_LAZY_LOCK, 
        	B_READ_AREA | B_WRITE_AREA);
        	
        if((area==B_ERROR) || (area==B_NO_MEMORY) || (area==B_BAD_VALUE))
        	addr = NULL;
           	
	}
	else
	{
		area_info info;
		
		get_area_info(area, &info);
		addr = info.address; 		
	}
	return (uchar*)addr;
}


static status_t
delete_ram_drive_area(void)
{
	area_id	area = find_area(ram_drive_area_name);
	
	if(area == B_NAME_NOT_FOUND)
		return B_ERROR;
	else
		return delete_area(area);
}

static void 
emulate_seek(off_t pos)
{
	static off_t old_pos = 0;

	if(!emulate_seek_flag)
		return;
		
	if(abs(pos-old_pos)>PREFETCH_BUFFER_SIZE)
	{
		old_pos = pos;		
		//snooze((int)(rand()* MAX_SEEK_TIME)/RAND_MAX);
	}
}

