#include <Application.h>
#include <Roster.h>
#include "GrowingArrayC.h"

/*********************************************************
	AppSwitcher 1.1
	
	Small background application to provide ALT-TAB
	Process-switching facility. Customise your keys
	by holding Command key at startup.
	
	Currently, I poll (yeah, it's a Bad Thing(tm) ) the
	system for the two magic key.
	
	Serge Beauchamp
	SoftMedia SENC
	softmedia@kagi.com

**********************************************************/

/*********************************************************/
/*      Replace these keys to customise the key bindings */

#define kCmdKey		0x66		// LEFT OPTION KEY ON A MACINTOSH KEYBOARD
#define kEscapeKey	0x01		// ESC
#define kTabKey		0x26		// TAB
#define kInverseKey	0x4b		// LEFT SHIFT KEY

// If the kInverseKey is interpreted as a modifier to the kTabKey
#define kInverseIsAModifier true

#define	kIconWidth			48
#define kPollingRate		50000 // 20 times per seconds
#define kDebug				false

/*                     									 */
/*********************************************************/

/**********************/
// Application Messages
/**********************/

#define SWITCH_APP 			'sapp'
#define QUIT_APP 			'qapp'
#define SWITCH_INVERSE_APP	'siap'
#define	ESCAPE_APP			'eapp'
#define	CLOSE_WINDOW		'cwin'
#define	OPEN_WINDOW			'owin'
#define	WAIT_FOR_KEYS		'wkey'

/**********************/
// Preference Window Messages
/**********************/

#define kCmdKeyChoose		'cmdC'
#define kSwitchKeyChoose	'swiC'
#define kEscapeKeyChoose	'escC'
#define kInverseKeyChoose	'invC'
#define kDefaultChoose		'defC'
#define kOKChoose			'_okC'
#define kCancelChoose		'cnlC'
#define kIsAModifierChoose	'modC'

/**********************/
// global Variables
/**********************/

uint8 	gCmdKey;
uint8 	gEscapeKey;
uint8 	gTabKey;
uint8 	gInverseKey;
bool	gInverseIsAModifier;

/**********************/
// Inline Functions
/**********************/

inline bool GetKeyState(uint8 *map,unsigned char state)
{
	return map[state >>3] & (1 << (7 - (state % 8)));
}

inline bool CommandIsDown()
{
	key_info map; 
	get_key_info(&map);
	return (GetKeyState(map.key_states,0x5d) || GetKeyState(map.key_states,0x5f));
}

/**********************/
//  SwitchStruct
/**********************/

struct SwitchStruct 
{
	char name[B_FILE_NAME_LENGTH];
	BBitmap *bmp;
	team_id who;
	int strlenght;
};

/**********************/
// SwitchView
/**********************/

class SwitchView : public BView
{
	private:
		GrowingArray<SwitchStruct> *array;
		int selected;
		thread_id aThread;
	public:
		SwitchView(GrowingArray<SwitchStruct> *array);
		
		~SwitchView() {}
		
virtual void Draw(BRect updateRect);
		void DrawOne(int old);
virtual	void AttachedToWindow();
virtual	void MessageReceived(BMessage * msg);
static	int32 keyIntercept(void * data);

		void BoiteVideCopland(const BRect & src);
		void RectangleGrisOmbre();
};

/**********************/
// SwitchWindow
/**********************/

class SwitchWindow : public BWindow
{
	public:
		SwitchWindow(GrowingArray<SwitchStruct> *array);
};	

/**********************/
// SwitchPrefWindow
/**********************/

class SwitchPrefWindow : public BWindow
{
	private:
static	char 	strings[256][32];
		uint8 	locCmdKey;
		uint8 	locEscapeKey;
		uint8 	locTabKey;
		uint8 	locInverseKey;
		bool	locInverseIsAModifier;
	public:
		SwitchPrefWindow();
virtual void MessageReceived(BMessage *msg);
		uint8 GetKeyDown();
};	

/**********************/
// SwitchApp
/**********************/

class SwitchApp : public BApplication
{
	private:
		GrowingArray<SwitchStruct> *array;
	public:
		SwitchApp();
		~SwitchApp() {
			kill_thread(aThread);
			WritePrefs();
		}

		bool QuitRequested();
		void AboutRequested();
		void LauchSwitcher();
		void WritePrefs();
		void ReadPrefs();
virtual void MessageReceived(BMessage *msg);
static int32 KeyPolling(void * data);

		int32 			lastSelected[2];
		SwitchView 		*view;
		SwitchWindow 	*window;
		thread_id		aThread;
};

SwitchApp *app;

/**********************/
// Main()
/**********************/

void main()
{
	app = new SwitchApp;
	
	app->Run();
	
	delete app; 
}

/**********************/
// SwitchApp
// MessageReceived()
/**********************/

void SwitchApp::MessageReceived(BMessage *msg)
{	
	if (kDebug)
		msg->PrintToStream();

	switch (msg->what)
	{
		case OPEN_WINDOW:
			if (window == NULL)
				LauchSwitcher();
			break;
		case CLOSE_WINDOW:
			window->Lock();
			window->Quit();
			window = NULL;
			view = NULL;
			break;
		default:
			if (view)
			{
				if (window->Lock())
				{
					view->MessageReceived(msg);
					window->Unlock();
				}
			}
			inherited::MessageReceived(msg);
	}
}

/**********************/
// SwitchApp
// KeyPolling()
/**********************/

int32 SwitchApp::KeyPolling(void * /*data */)
{
	key_info map; 
	
	while (1)
	{
		snooze(kPollingRate);
		get_key_info(&map);

		// cmdKey
		if (GetKeyState(map.key_states,gCmdKey) && GetKeyState(map.key_states,gTabKey))
			app->PostMessage(OPEN_WINDOW);
	}
}

/**********************/
// SwitchApp
// SwitchApp()
/**********************/

SwitchApp::SwitchApp() : BApplication("application/x-softmedia_AppSwitcher")
{
	window = NULL;
	view = NULL;
	lastSelected[0] = 0;
	lastSelected[1] = 0;
	array = NULL;

	gCmdKey 			= kCmdKey;
	gEscapeKey 			= kEscapeKey;
	gTabKey 			= kTabKey;
	gInverseKey 		= kInverseKey;
	gInverseIsAModifier = kInverseIsAModifier;

	uint16 id;
	get_keyboard_id(&id); 
	 
	ReadPrefs();   
	aThread = spawn_thread(KeyPolling,
								"KeyPooling",
								B_LOW_PRIORITY,
								(void *) NULL); 
		
	resume_thread(aThread);
	
	if (CommandIsDown())
	{
		SwitchPrefWindow * win = new SwitchPrefWindow();
		win->Show();
	}
}

/**********************/
// SwitchApp
// ReadPrefs()
/**********************/

void SwitchApp::ReadPrefs()
{
	// getting the preferences from the file
	BPath path_obj;
	find_directory(B_USER_SETTINGS_DIRECTORY,&path_obj);
    path_obj.Append("AppSwitcher1.1preferences");
	BFile aFile(path_obj.Path(),B_READ_ONLY);
	if (aFile.InitCheck() == B_NO_ERROR)
	{	
		uint8 buffer[5];
		aFile.Read(buffer,sizeof(uint8) * 5);

		gCmdKey 			= buffer[0];
		gEscapeKey 			= buffer[1];
		gTabKey 			= buffer[2];
		gInverseKey 		= buffer[3];
		gInverseIsAModifier = buffer[4];
    }
    else 	
	{
		SwitchPrefWindow * win = new SwitchPrefWindow();
		win->Show();
	}
}

/**********************/
// SwitchApp
// WritePrefs()
/**********************/

void SwitchApp::WritePrefs()
{
	BPath path_obj;
	find_directory(B_USER_SETTINGS_DIRECTORY,&path_obj);
    path_obj.Append("AppSwitcher1.1preferences");
	BFile aFile(path_obj.Path(),B_READ_WRITE | B_CREATE_FILE);
	if (aFile.InitCheck() == B_NO_ERROR)
	{	
		uint8 buffer[5];

		buffer[0] = gCmdKey;
		buffer[1] = gEscapeKey;
		buffer[2] = gTabKey;
		buffer[3] = gInverseKey;
		buffer[4] = gInverseIsAModifier;

		aFile.Write(buffer,sizeof(uint8) * 5);
    }
}

/**********************/
// SwitchApp
// QuitRequested()
/**********************/

bool SwitchApp::QuitRequested()
{
	be_app->PostMessage(B_QUIT_REQUESTED);
	return true;
}

/**********************/
// SwitchApp
// AboutRequested()
/**********************/

void SwitchApp::AboutRequested()
{
}

/**********************/
// SwitchApp
// LaunchSwitcher()
/**********************/

void SwitchApp::LauchSwitcher()
{
	BList *teams = new BList; 
	be_roster->GetAppList(teams); 
	
	if (array)
		delete array;
		
	array = new GrowingArray<SwitchStruct>;
	
	int32 max = teams->CountItems();
	for (int i = 0; i < max;i++)
	{
		team_id who = (team_id)teams->ItemAt(i);
		app_info appInfo;
		be_roster->GetRunningAppInfo(who,&appInfo);
		
		if (((appInfo.flags & B_BACKGROUND_APP) == false) &&
			 ((appInfo.flags & B_ARGV_ONLY) == false) &&
			 (strcmp(appInfo.signature,"application/x-vnd.Be-TSKB") != 0 ) &&
			 (strcmp(appInfo.signature,"application/x-softmedia_AppSwitcher") != 0 ) )
		{
			status_t status;
			SwitchStruct sw;
			sw.who = who;
			sw.strlenght = 0;
			BEntry(&(appInfo.ref),true).GetName(sw.name);
			BAppFileInfo appFileInfo;
			BFile file(&(appInfo.ref),B_READ_ONLY);
			status = appFileInfo.SetTo(&file);
			sw.bmp = new BBitmap(BRect(0, 0, 31, 31),B_COLOR_8_BIT);
			status = appFileInfo.GetIcon(sw.bmp,B_LARGE_ICON);
			array->Add(&sw);
		}
	}

	window = new SwitchWindow(array);
	window->Show();
}

/**********************/
// SwitchWindow
// SwitchWindow()
/**********************/

SwitchWindow::SwitchWindow(GrowingArray<SwitchStruct> *array) : BWindow(BRect(0,0,50,50),"AppSwitcher",B_MODAL_WINDOW,B_NOT_MOVABLE | B_NOT_RESIZABLE)
{
	int largeur = array->GetSize() * kIconWidth + 31;
	BRect ecran = BScreen().Frame();
	int screen_rows = ecran.right - ecran.left;
	
	MoveTo((screen_rows - largeur)/2,200);
	ResizeTo(largeur,63);
	app->view = new SwitchView(array);
	AddChild(app->view);
}

/**********************/
// SwitchView
// SwitchView()
/**********************/

SwitchView::SwitchView(GrowingArray<SwitchStruct> *_array) : BView(BRect(0,0,31 + _array->GetSize() * kIconWidth,63),"AppSwitcher",B_FOLLOW_NONE,B_WILL_DRAW)
{
	array = _array;
	
	selected = 0;
	for (int i = 0;i < array->GetSize();i++)
	{
		if (array->Get(i)->who == app->lastSelected[1])
		{
			selected = i;
			break;
		}
	}
	aThread = 0;
}

/**********************/
// SwitchView
// AttachedToWindow()
/**********************/

void SwitchView::AttachedToWindow()
{
	aThread = spawn_thread(keyIntercept,
								"keyIntercept",
								B_NORMAL_PRIORITY,
								(void *) Window()); 
		
	resume_thread(aThread);
}

/**********************/
// SwitchView
// keyIntercept()
/**********************/

int32 SwitchView::keyIntercept(void * /* data */)
{
	bool tabKey = true,inverseKey = true;
	key_info map; 
	
	if (kDebug)
		printf("starting thread...\n");
	while (1)
	{
		snooze(5000);
			
		get_key_info(&map);

		// cmdKey
		if (! GetKeyState(map.key_states,gCmdKey))
		{
			app->PostMessage(QUIT_APP);
			return 0;
		}
		// escape
		if (GetKeyState(map.key_states,gEscapeKey))
		{
			app->PostMessage(ESCAPE_APP);
			return 0;
		}

		// tabKey
		if (GetKeyState(map.key_states,gTabKey))
		{
			if (tabKey == false)
			{
				if (kInverseIsAModifier && GetKeyState(map.key_states,gInverseKey))
					app->PostMessage(SWITCH_INVERSE_APP);
				else
					app->PostMessage(SWITCH_APP);
			}
			tabKey = true;
		}
		else
			tabKey = false;
			
		if (!gInverseIsAModifier)
		{
			// tabKey
			if (GetKeyState(map.key_states,gInverseKey))
			{
				if (inverseKey == false)
					app->PostMessage(SWITCH_INVERSE_APP);
				inverseKey = true;
			}
			else
				inverseKey = false;
		}
	}

	if (kDebug)
		printf("stopping thread...\n");
}

/**********************/
// SwitchView
// MessageReceived()
/**********************/

void SwitchView::MessageReceived(BMessage * msg)
{
	int old;
	switch(msg->what)
	{
		case SWITCH_APP:
			old = selected++;
			if (selected >= array->GetSize())
				selected = 0;
			DrawOne(old);
			break;
		case SWITCH_INVERSE_APP:
			old = selected--;
			if (selected < 0)
				selected = array->GetSize() -1;
			DrawOne(old);
			break;
		case QUIT_APP:
			be_roster->ActivateApp(array->Get(selected)->who);
			app->lastSelected[1] = app->lastSelected[0];
			app->lastSelected[0] = array->Get(selected)->who;
		case ESCAPE_APP:
			be_app->PostMessage(CLOSE_WINDOW);
			break;
		case B_QUIT_REQUESTED:
			be_app->PostMessage(B_QUIT_REQUESTED);
			break;			
	}
}

/**********************/
// SwitchView
// Draw()
/**********************/

void SwitchView::Draw(BRect /*updateRect */)
{
	RectangleGrisOmbre();
	SwitchStruct * sw;
	
	SetDrawingMode(B_OP_OVER);

	for (int i = 0;i < array->GetSize();i++)
	{
		BPoint pt;
		pt.x = 16 + i * kIconWidth + (kIconWidth - 32)/2;
		pt.y = 8;
		sw = array->Get(i);
		DrawBitmapAsync(sw->bmp,pt);
	}
	
	SetDrawingMode(B_OP_COPY);
	BoiteVideCopland(BRect(16,45,16 + array->GetSize() * kIconWidth,60));
	
	Sync();
	DrawOne(0);
}

/**********************/
// SwitchView
// DrawOne()
/**********************/

void SwitchView::DrawOne(int old)
{
	SetFont(be_plain_font);
	SetFontSize(12);
	
	BPoint pt;
	pt.x = 16 + old * kIconWidth + (kIconWidth - 32)/2;
	pt.y = 8;
	
	BRect r(pt.x -2,pt.y -2,pt.x + 35,pt.y + 35);
	SetHighColor(222,222,222);
	StrokeRect(r);

	pt.x = 16 + selected * kIconWidth + (kIconWidth - 32)/2;
	r.Set(pt.x -2,pt.y -2,pt.x + 35,pt.y + 35);
	SetHighColor(0,0,0);
	StrokeRect(r);

	SwitchStruct * sw = array->Get(selected);

	SetDrawingMode(B_OP_OVER);
			
	if (sw->strlenght == 0)
		sw->strlenght = StringWidth (sw->name);
	
	int left = array->GetSize() * kIconWidth;
	BoiteVideCopland(BRect(16,45,16 + left,60));
	MovePenTo (16 + (left - sw->strlenght) /2,58);
	
	SetHighColor(0,0,0,0);
	DrawString (sw->name);
	SetDrawingMode(B_OP_COPY);
	Sync();
}

/**********************/
// SwitchView
// RectangleGrisOmbre()
/**********************/

void SwitchView::RectangleGrisOmbre()
{
	BRect r = Frame();
	
	r.Set(0,-1,r.right - r.left,r.bottom - r.top);
	
	rgb_color maCouleur;
		
	maCouleur.alpha = 0;
	maCouleur.green = 221;maCouleur.red = 221;maCouleur.blue = 221;
	SetHighColor (maCouleur);
	StrokeRect (r);
	r.left += 1;r.top += 1;	
	maCouleur.green = 85;maCouleur.red = 85;maCouleur.blue = 85;
	SetHighColor (maCouleur);
	StrokeRect (r);
	r.right -= 1;r.bottom -= 1;	
	maCouleur.green = 255;maCouleur.red = 255;maCouleur.blue = 255;
	SetHighColor (maCouleur);
	StrokeRect (r);
	r.left += 1;r.top += 1;	
	maCouleur.green = 153;maCouleur.red = 153;maCouleur.blue = 153;
	SetHighColor (maCouleur);
	StrokeRect (r);
	r.right -= 1;r.bottom -= 1;	
	maCouleur.green = 221;maCouleur.red = 221;maCouleur.blue = 221;
//	SetHighColor (maCouleur);
	SetHighColor(222,222,222);
	FillRect (r);
}

/**********************/
// SwitchView
// BoiteVideCopland()
/**********************/

void SwitchView::BoiteVideCopland(const BRect & src)
{
	BRect r(src);

	rgb_color maCouleur;
		
	maCouleur.alpha = 0;
	maCouleur.green = 170;maCouleur.red = 170;maCouleur.blue = 170;
	SetHighColor (maCouleur);
	StrokeRect (r);
	r.left += 1;r.top += 1;	
	maCouleur.green = 255;maCouleur.red = 255;maCouleur.blue = 255;
	SetHighColor (maCouleur);
	FillRect (r);
	r.right -= 1;r.bottom -= 1;	
	maCouleur.green = 0;maCouleur.red = 0;maCouleur.blue = 0;
	SetHighColor (maCouleur);
	StrokeRect (r);
}



/**********************/
// SwitchPrefWindow
/**********************/

char SwitchPrefWindow::strings[256][32] =  {
//			0x00
			"Nothing",
			"Escape",
			"F1",
			"F2",
			"F3",
			"F4",
			"F5",
			"F6",
			"F7",
			"F8",
			"F9",
			"F10",
			"F11",
			"F12",
			"Print Screen",
			"Scroll Lock",

//			0x10
			"Pause",
			"~",
			"1 (keyboard)",
			"2 (keyboard)",
			"3 (keyboard)",
			"4 (keyboard)",
			"5 (keyboard)",
			"6 (keyboard)",
			"7 (keyboard)",
			"8 (keyboard)",
			"9 (keyboard)",
			"0 (keyboard)",
			"- (keyboard)",
			"= (keyboard)",
			"BackSpace",
			"Insert",

//			0x20
			"Home",
			"Page Up",
			"NumLock",
			"/ (keyPad)",
			"* (keyPad)",
			"- (keyPad)",
			"Tab",
			"q",
			"w",
			"e",
			"r",
			"t",
			"y",
			"u",
			"i",
			"o",

//			0x30
			"p",
			"[",
			"]",
			"\\",
			"Delete",
			"End",
			"Page Down",
			"7 (keyPad)",
			"8 (keyPad)",
			"9 (keyPad)",
			"+ (keyPad)",
			"Caps Lock",
			"a",
			"s",
			"d",
			"f",
			
//			0x40
			"g",
			"h",
			"j",
			"k",
			"l",
			";",
			"'",
			"Return (keyboard)",
			"4 (keyPad)",
			"5 (keyPad)",
			"6 (keyPad)",
			"Left Shift",
			"z",
			"x",
			"c",
			"v",
//			0x50
			"b",
			"n",
			"m",
			", (keyboard)",
			". (keyboard)",
			"/",
			"Right Shift",
			"Up Arrow",
			"1 (keyPad)",
			"2 (keyPad)",
			"3 (keyPad)",
			"Enter (keyPad)",
			"Left Control",
			"Left Command",
			"Space Bar",
			"Right Command",
//			0x60
			"Right Control",
			"Left Arrow",
			"Down Arrow",
			"Right Arrow",
			"0 (keyPad)",
			". (keyPad)",
			"Left Option",
			"Right Option",
			"Menu (Win95 keyboard)",
			"?",
			"?",			
			"Macintosh PowerKey",
			"?",
			"?",
			"?",
			"?",
//			0x70
			"?",
			"?",
			"?",
			"?",
			"?",
			"?",
			"?",
			"?",
			"?",
			"?",
			"?",
			"?",
			"?",
			"?",
			"SysRequest",
			"Break",

};

/**********************/
// SwitchPrefWindow
// SwitchPrefWindow()
/**********************/

SwitchPrefWindow::SwitchPrefWindow() : BWindow(BRect(200,200,525,400),"AppSwitcher Preferences",B_TITLED_WINDOW,B_NOT_RESIZABLE)
{
	locCmdKey 		= gCmdKey;
	locEscapeKey 	= gEscapeKey;
	locTabKey		= gTabKey;
	locInverseKey	= gInverseKey;
	locInverseIsAModifier = gInverseIsAModifier;

	AddChild(new BStringView(BRect(20,30,100,50),"Command Key","Command Key:"));
	AddChild(new BStringView(BRect(20,55,100,75),"Switch Key","Switch Key:"));
	AddChild(new BStringView(BRect(20,80,100,100),"Escape Key","Escape Key:"));
	AddChild(new BStringView(BRect(20,105,100,125),"Inverse Key","Inverse Key:"));

	AddChild(new BStringView(BRect(120,30,200,50),"Command Key_Value",strings[locCmdKey]));
	AddChild(new BStringView(BRect(120,55,200,75),"Switch Key_Value",strings[locTabKey]));
	AddChild(new BStringView(BRect(120,80,200,100),"Escape Key_Value",strings[locEscapeKey]));
	AddChild(new BStringView(BRect(120,105,200,125),"Inverse Key_Value",strings[locInverseKey]));

	AddChild(new BButton(BRect(220,30,300,50),"Command KeyChoose","Choose...",new BMessage(kCmdKeyChoose)));
	AddChild(new BButton(BRect(220,55,300,75),"Switch KeyChoose","Choose...",new BMessage(kSwitchKeyChoose)));
	AddChild(new BButton(BRect(220,80,300,100),"Escape KeyChoose","Choose...",new BMessage(kEscapeKeyChoose)));
	AddChild(new BButton(BRect(220,105,300,125),"Inverse KeyChoose","Choose...",new BMessage(kInverseKeyChoose)));

	AddChild(new BCheckBox(BRect(30,130,130,140),"IsAModifier","Is a Modifier",new BMessage(kIsAModifierChoose)));

	AddChild(new BButton(BRect(20,150,100,170),"Default","Default",new BMessage(kDefaultChoose)));
	AddChild(new BButton(BRect(120,150,200,170),"Cancel","Cancel",new BMessage(kCancelChoose)));
	AddChild(new BButton(BRect(220,150,300,170),"OK","OK",new BMessage(kOKChoose)));
	
	((BButton *)FindView("OK"))->MakeDefault(true);
	((BCheckBox *)FindView("IsAModifier"))->SetValue(locInverseIsAModifier);
}

/**********************/
// SwitchPrefWindow
// MessageReceived()
/**********************/

void SwitchPrefWindow::MessageReceived(BMessage *msg)
{
	switch(msg->what)
	{
		case kDefaultChoose:
			locCmdKey 		= kCmdKey;
			locEscapeKey 	= kEscapeKey;
			locTabKey		= kTabKey;
			locInverseKey	= kInverseKey;
			locInverseIsAModifier = kInverseIsAModifier;
			((BCheckBox *)FindView("IsAModifier"))->SetValue(locInverseIsAModifier);
 			((BStringView *)FindView("Command Key_Value"))->SetText(strings[locCmdKey]);
		 	((BStringView *)FindView("Switch Key_Value"))->SetText(strings[locTabKey]);
 			((BStringView *)FindView("Escape Key_Value"))->SetText(strings[locEscapeKey]);
		 	((BStringView *)FindView("Inverse Key_Value"))->SetText(strings[locInverseKey]);
			break;
		case kCancelChoose:
			PostMessage(B_QUIT_REQUESTED);
			break;
		case kOKChoose:
			gCmdKey 		= locCmdKey;
			gEscapeKey 		= locEscapeKey;
			gTabKey			= locTabKey;
			gInverseKey		= locInverseKey;
			gInverseIsAModifier = locInverseIsAModifier;
			PostMessage(B_QUIT_REQUESTED);
			app->WritePrefs();
			break;
		case kIsAModifierChoose:
			locInverseIsAModifier = ((BCheckBox *)FindView("IsAModifier"))->Value();
			((BCheckBox *)FindView("IsAModifier"))->SetValue(locInverseIsAModifier);
			break;
		case kCmdKeyChoose:
			locCmdKey = GetKeyDown();
 			((BStringView *)FindView("Command Key_Value"))->SetText(strings[locCmdKey]);
			break;
		case kSwitchKeyChoose:
			locTabKey = GetKeyDown();
		 	((BStringView *)FindView("Switch Key_Value"))->SetText(strings[locTabKey]);
			break;
		case kEscapeKeyChoose:
			locEscapeKey = GetKeyDown();
 			((BStringView *)FindView("Escape Key_Value"))->SetText(strings[locEscapeKey]);
			break;
		case kInverseKeyChoose:
			locInverseKey = GetKeyDown();
		 	((BStringView *)FindView("Inverse Key_Value"))->SetText(strings[locInverseKey]);
			break;
		default:
			inherited::MessageReceived(msg);
	}
}


/**********************/
// TextModalWindow
/**********************/

class TextModalWindow : public BWindow
{
	public:
		TextModalWindow(char * str,BRect r,BRect winSize) : BWindow(winSize,str,B_MODAL_WINDOW,B_NOT_MOVABLE | B_NOT_RESIZABLE) {
			int largeur = winSize.right - winSize.left;
			BRect ecran = BScreen().Frame();
			int screen_rows = ecran.right - ecran.left;
		
			MoveTo((screen_rows - largeur)/2,200);
			ResizeTo(largeur,63);
			AddChild(new BStringView(r,str,str));
		}
};


/**********************/
// SwitchPrefWindow
// GetKeyDown()
/**********************/

uint8 SwitchPrefWindow::GetKeyDown()
{
	TextModalWindow * win = new TextModalWindow("Press ANY key...",BRect(20,20,100,40),BRect(200,200,320,260));
	win->Show();

	uint8 returnValue = 0;
	key_info map; 
	
	while(returnValue == 0)
	{
		snooze(5000);
		
		get_key_info(&map);
		for (int i = 0;i < 0x80;i++)
		{
			if (GetKeyState(map.key_states,i))
			{
				if ((i != 0x22) && (i != 0x3b))
				{
					returnValue = i;
					break;
				}
			}
		}
	}
	
	if (! win->IsLocked())
		win->Lock();
	win->Close();
	
	if (kDebug)
		printf("returnValue = %x\n",returnValue);
	return returnValue;
}


/************ end of file *******************/




