// Aspell SC_Error defination
// Copyright 1998 by Kevin Atkinson under the terms of the LGPL

#ifndef sc_error_hh
#define sc_error_hh

#include <string>
#include "object.hh"

struct aspell_error {
  const aspell_object * culprit;
  aspell_error() {}
  aspell_error(const aspell_object *o) : culprit(o) {}
  virtual string message() const = 0;
  virtual ~aspell_error() {}
};

struct ae_file_problem : virtual public aspell_error {
  string file;
  ae_file_problem() {}
  ae_file_problem(const string &f) : file(f) {}
};

struct ae_cant_read : public ae_file_problem {
  string message() const;
  ae_cant_read() {}
  ae_cant_read(const string &f, const aspell_object *o) 
    : aspell_error(o), ae_file_problem(f) {}
};

struct ae_bad_format : public ae_file_problem {
  string message() const;
  ae_bad_format() {}
  ae_bad_format(const string &f, const aspell_object *o) 
    : aspell_error(o), ae_file_problem(f) {}
};

struct ae_cant_write : public ae_file_problem {
  string message() const;
  ae_cant_write() {}
  ae_cant_write(const string &f, const aspell_object *o) 
    : aspell_error(o), ae_file_problem(f) {}
};

struct ae_duplicates : public ae_file_problem {
  string message() const;
  ae_duplicates() {}
  ae_duplicates(const string &f, const aspell_object *o) 
    : aspell_error(o), ae_file_problem(f) {}
};

struct ae_language_problem : virtual public aspell_error {
  string language;
  ae_language_problem() {}
  virtual void rethrow_w_file(const string &f, const aspell_object *o) const = 0;
  ae_language_problem(const string &l)
    : language(l) {}
};

struct ae_file_language_problem : public ae_file_problem,
				  virtual public ae_language_problem 
{
  ae_file_language_problem() {}
  ae_file_language_problem(const string &f) : ae_file_problem(f) {}
};

struct ae_unknown_lang : virtual public ae_language_problem {
  string message() const;
  void rethrow_w_file(const string &f, const aspell_object *o) const;
  ae_unknown_lang() {}
  ae_unknown_lang(const string &l, const aspell_object * o)
    : aspell_error(o), ae_language_problem(l) {}
};

struct ae_unknown_phonetic : virtual public ae_language_problem {
  string phonetic;
  string message() const;
  void rethrow_w_file(const string &f, const aspell_object *o) const;
  ae_unknown_phonetic() {}
  ae_unknown_phonetic(const string &p, const string &l, const aspell_object *o)
    : aspell_error(o), ae_language_problem(l), phonetic(p) {}
};

struct ae_mismatched_lang : virtual public ae_language_problem {
  string message() const;
  void rethrow_w_file(const string &f, const aspell_object *o) const;
  ae_mismatched_lang() {}
  ae_mismatched_lang(const string &l, const aspell_object * o)
    : aspell_error(o), ae_language_problem(l) {}
};



struct ae_unknown_lang_in_file : public ae_unknown_lang,
				 public ae_file_language_problem
{
  string message() const;
  ae_unknown_lang_in_file() {}
  ae_unknown_lang_in_file(const string &f, const string &l, 
			  const aspell_object * o) 
    : aspell_error(o), ae_language_problem(l), ae_file_language_problem(f) {}
};

struct ae_mismatched_lang_in_file : public ae_mismatched_lang,
				    public ae_file_language_problem
{
  string message() const;
  ae_mismatched_lang_in_file() {}
  ae_mismatched_lang_in_file(const string &f, const string &l, 
			     const aspell_object * o) 
    : aspell_error(o), ae_file_language_problem(f), ae_language_problem(l) {}
};

class WordList_Base;

struct ae_dont_own_wordlist : public aspell_error {
  const WordList_Base * word_list;
  string message() const;
  ae_dont_own_wordlist() {}
  ae_dont_own_wordlist(const WordList_Base *wl, const aspell_object * o)
    : aspell_error(o), word_list(wl) {}
};


#endif
