//:  BeOSBorders.java

package com.be.swing.plaf.beos;

import java.awt.*;
import javax.swing.*;
import javax.swing.plaf.*;
import javax.swing.border.*;

/**
 * Factory class for creating Border resources for the BeOS
 * Look and Feel.  To support BeOS system preferences for colors,
 * use the static color accessor methods in BeOSLookAndFeel.
 *
 * @see BeOSLookAndFeel#getWhite
 * @author  Michael Heuer
 * @author  $Id: BeOSBorders.java,v 1.5 1999/11/10 04:01:18 heuermh Exp $
 */ 
public class BeOSBorders {

	/**
	 * Dynamic Button border that changes appearance
	 * based on the state of the button and the button model.
	 */
	static class ButtonBorder extends AbstractBorder implements UIResource {

		public void paintBorder(Component c, Graphics g, int x, int y, int w, int h) {

			JButton button = (JButton) c;
			ButtonModel model = button.getModel();
			
			g.translate(x,y);
			        
			if (model.isEnabled()) {
		        if (model.isPressed() && model.isArmed()) {
                
	                drawPressedBorder(c,g,w,h);
		        } else {
                
	                drawActiveBorder(c,g,w,h);
		        }
			} else {
        
		        drawDisabledBorder(c,g,w,h);
			}

			g.translate(-x,-y);

		}

		private void drawActiveBorder(Component c, Graphics g, int w, int h) {

			// dark border
			g.setColor(BeOSLookAndFeel.getGray99());
			g.drawLine(1,0,w-2,0); // r
			g.drawLine(w-2,h-1,1,h-1); // l
			g.drawLine(w-1,1,w-1,h-2); // d
			g.drawLine(0,h-2,0,1); // u
			
			// lower-right inner shadow
			g.setColor(BeOSLookAndFeel.getGray156());
			g.drawLine(w-2,2,w-2,h-2); // d
			g.drawLine(w-2,h-2,2,h-2); // l
			
			// upper-left highlight
			g.setColor(BeOSLookAndFeel.getGray239());
			g.drawLine(1,1,w-2,1); // r
			g.drawLine(1,1,1,h-2); // d
			
			// lower-right shadow
			g.drawLine(4,h-4,w-4,h-4); // r
			g.drawLine(w-4,h-4,w-4,4); // u
			
			// white highlight
			g.setColor(BeOSLookAndFeel.getWhite());
			g.drawLine(2,2,w-3,2); // r
			g.drawLine(2,2,2,h-3); // d
			g.drawLine(3,3,w-4,3); // r
			g.drawLine(3,3,3,h-4); // d
			
			// lower-right middle part
			g.setColor(BeOSLookAndFeel.getGray222());
			g.drawLine(3,h-3,w-3,h-3); // r
			g.drawLine(w-3,h-3,w-3,3); // u

		}

		private void drawPressedBorder(Component c, Graphics g, int w, int h) {

			// dark border
			g.setColor(BeOSLookAndFeel.getGray99());
			g.drawLine(1,0,w-2,0); // r
			g.drawLine(w-2,h-1,1,h-1); // l
			g.drawLine(w-1,1,w-1,h-2); // d
			g.drawLine(0,h-2,0,1); // u
			
			// everything else is black
			g.setColor(BeOSLookAndFeel.getBlack());
			g.drawLine(w-2,2,w-2,h-2); // d
			g.drawLine(w-2,h-2,2,h-2); // l
			
			g.drawLine(1,1,w-2,1); // r
			g.drawLine(1,1,1,h-2); // d
			
			g.drawLine(4,h-4,w-4,h-4); // r
			g.drawLine(w-4,h-4,w-4,4); // u
			
			g.drawLine(2,2,w-3,2); // r
			g.drawLine(2,2,2,h-3); // d
			g.drawLine(3,3,w-4,3); // r
			g.drawLine(3,3,3,h-4); // d
			
			g.drawLine(3,h-3,w-3,h-3); // r
			g.drawLine(w-3,h-3,w-3,3); // u

		}

		private void drawDisabledBorder(Component c, Graphics g, int w, int h) {

			// dark border
			g.setColor(BeOSLookAndFeel.getGray156());
			g.drawLine(1,0,w-2,0); // r
			g.drawLine(w-2,h-1,1,h-1); // l
			g.drawLine(w-1,1,w-1,h-2); // d
			g.drawLine(0,h-2,0,1); // u
			                        
			// upper-left highlight
			g.setColor(BeOSLookAndFeel.getGray239());
			g.drawLine(1,1,w-2,1); // r
			g.drawLine(1,1,1,h-2); // d
			
			// lower-right shadow
			g.drawLine(4,h-4,w-4,h-4); // r
			g.drawLine(w-4,h-4,w-4,4); // u
			
			// white highlight
			g.setColor(BeOSLookAndFeel.getWhite());
			g.drawLine(2,2,w-3,2); // r
			g.drawLine(2,2,2,h-3); // d
			g.drawLine(3,3,w-4,3); // r
			g.drawLine(3,3,3,h-4); // d
			
			g.setColor(BeOSLookAndFeel.getGray222());
			// lower-right inner shadow
			g.drawLine(w-2,2,w-2,h-2); // d
			g.drawLine(w-2,h-2,2,h-2); // l
			// lower-right middle part
			g.drawLine(3,h-3,w-3,h-3); // r
			g.drawLine(w-3,h-3,w-3,3); // u

		}

		public Insets getBorderInsets(Component c) {
			return BeOSLookAndFeel.getButtonInsets();
		}
	}

	/**
	 * Dynamic Button border that draws the "default button" border
	 * around the button if it happens to be the default button.
	 */     
	static class DefaultButtonBorder extends AbstractBorder implements UIResource {
	
		public void paintBorder(Component c, Graphics g, int x, int y, int w, int h) {

			JButton button = (JButton) c;
			ButtonModel model = button.getModel();

			if (button.isDefaultButton()) {

		        g.translate(x,y);
        
		        if (model.isEnabled()) {
                
	                drawDefaultBorder(c,g,w,h);
		        } else {
                
	                drawDisabledDefaultBorder(c,g,w,h);
		        }

		        g.translate(-x,-y);
			}
		}

		public void drawDefaultBorder(Component c, Graphics g, int w, int h) {

			// dark border          
			g.setColor(BeOSLookAndFeel.getGray99());
			g.drawLine(1,0,w-2,0); // r
			g.drawLine(w-2,h-1,1,h-1); // l
			g.drawLine(w-1,1,w-1,h-2); // d
			g.drawLine(0,h-2,0,1); // u
			
			g.setColor(BeOSLookAndFeel.getGray189());
			g.drawRect(1,1,w-3,h-3);

		}

		public void drawDisabledDefaultBorder(Component c, Graphics g, int w, int h) {

			// dark border          
			g.setColor(BeOSLookAndFeel.getGray189());
			g.drawLine(1,0,w-2,0); // r
			g.drawLine(w-2,h-1,1,h-1); // l
			g.drawLine(w-1,1,w-1,h-2); // d
			g.drawLine(0,h-2,0,1); // u

		}       

		public Insets getBorderInsets(Component c) {
			JButton b = (JButton) c;

			if (b.isDefaultCapable()) {
		        return BeOSLookAndFeel.getDefaultButtonInsets();
			} else {
		        return BeOSLookAndFeel.getZeroInsets();
			}
		}                       
	}

	/**
	 * Dynamic TextComponent border that changes appearance
	 * based on the state of the component.  Also draws the blue
	 * BeOS focus ring if the TextComponent has focus.
	 */
	static class TextComponentBorder extends AbstractBorder implements UIResource {

		public void paintBorder(Component c, Graphics g, int x, int y, int w, int h) {
		
			g.translate(x,y);

			if (c.isEnabled()) {
			        
		        drawBorder(c, g, w, h);
			} else {
			
		        drawDisabledBorder(c, g, w, h);
			}
			
			g.translate(-x,-y);
		}

		public void drawBorder(Component c, Graphics g, int w, int h) {

			g.setColor(BeOSLookAndFeel.getGray189());
			g.drawLine(0,0,w-1,0);
			g.drawLine(0,0,0,h-1);
			
			g.setColor(BeOSLookAndFeel.getWhite());
			g.drawLine(1,h-1,w-1,h-1);
			g.drawLine(w-1,h-1,w-1,1);
			
			// for 1.1 only
			JComponent jc = (JComponent) c;
			// end 1.1 only
			
			if (jc.hasFocus()) {

		        g.setColor(BeOSLookAndFeel.getBlue());
		        g.drawLine(1,1,w-2,1);
		        g.drawLine(1,1,1,h-2);
		        g.drawLine(2,h-2,w-2,h-2);
		        g.drawLine(w-2,h-2,w-2,2);

			} else {

		        g.setColor(BeOSLookAndFeel.getGray99());
		        g.drawLine(1,1,w-2,1);
		        g.drawLine(1,1,1,h-2);
		        
		        g.setColor(BeOSLookAndFeel.getGray222());
		        g.drawLine(2,h-2,w-2,h-2);
		        g.drawLine(w-2,h-2,w-2,2);
			}                                               
		}

		public void drawDisabledBorder(Component c, Graphics g, int w, int h) {

			g.setColor(BeOSLookAndFeel.getGray222());
			g.drawLine(0,0,w-1,0);
			g.drawLine(0,0,0,h-1);
			
			g.setColor(BeOSLookAndFeel.getGray239());
			g.drawLine(1,h-1,w-1,h-1);
			g.drawLine(w-1,h-1,w-1,1);
			
			g.setColor(BeOSLookAndFeel.getGray156());
			g.drawLine(1,1,w-2,1);
			g.drawLine(1,1,1,h-2);
			
			g.setColor(BeOSLookAndFeel.getGray222());
			g.drawLine(2,h-2,w-2,h-2);
			g.drawLine(w-2,h-2,w-2,2);
                        
		}

		public Insets getBorderInsets(Component c) {
//			return BeOSLookAndFeel.getZeroInsets();
			return BeOSLookAndFeel.getTextComponentInsets();
		}
	}

	/**
	 * Simple border for the ProgressBar.
	 */
	static class ProgressBarBorder extends AbstractBorder implements UIResource {

		public void paintBorder(Component c, Graphics g, int x, int y, int w, int h) {

			g.translate(x,y);
			
			if (c.isEnabled()) {
			        
		        drawBorder(c, g, w, h);
			} else {
			
		        drawDisabledBorder(c, g, w, h);
			}
			
			g.translate(-x,-y);
		}

		public void drawBorder(Component c, Graphics g, int w, int h) {

			g.setColor(BeOSLookAndFeel.getGray189());
			g.drawLine(0,0,w-1,0);
			g.drawLine(0,0,0,h-1);
			
			g.setColor(BeOSLookAndFeel.getWhite());
			g.drawLine(1,h-1,w-1,h-1);
			g.drawLine(w-1,h-1,w-1,1);
			                
			g.setColor(BeOSLookAndFeel.getGray99());
			g.drawLine(1,1,w-2,1);
			g.drawLine(1,1,1,h-2);
			
			g.setColor(BeOSLookAndFeel.getGray222());
			g.drawLine(2,h-2,w-2,h-2);
			g.drawLine(w-2,h-2,w-2,2);
		}

		public void drawDisabledBorder(Component c, Graphics g, int w, int h) {

			g.setColor(BeOSLookAndFeel.getGray222());
			g.drawLine(0,0,w-1,0);
			g.drawLine(0,0,0,h-1);
			
			g.setColor(BeOSLookAndFeel.getGray239());
			g.drawLine(1,h-1,w-1,h-1);
			g.drawLine(w-1,h-1,w-1,1);
			
			g.setColor(BeOSLookAndFeel.getGray156());
			g.drawLine(1,1,w-2,1);
			g.drawLine(1,1,1,h-2);
			
			g.setColor(BeOSLookAndFeel.getGray222());
			g.drawLine(2,h-2,w-2,h-2);
			g.drawLine(w-2,h-2,w-2,2);
                        
		}

		public Insets getBorderInsets(Component c) {
//			return BeOSLookAndFeel.getZeroInsets();
			return BeOSLookAndFeel.getProgressBarInsets();
		}
	}
}