//:  BeOSLookAndFeel.java

package com.be.swing.plaf.beos;

import java.awt.*;
import javax.swing.*;
import javax.swing.plaf.*;
import javax.swing.border.*;
import javax.swing.plaf.basic.*;

/**
 * The BeOS pluggable Look and Feel.
 *
 * @author  Michael Heuer
 * @version $Id: BeOSLookAndFeel.java,v 1.5 1999/11/10 04:01:19 heuermh Exp $
 */
public class BeOSLookAndFeel extends BasicLookAndFeel {

	/** 
	* Returns a one line description of this look and feel implementation, 
	* e.g. "The CDE/Motif Look and Feel".   This string is intended for 
	* the user, e.g. in the title of a window or in a ToolTip message.
	*
	* @return a one line description of this look and feel
	*/
	public String getDescription() {
		return "The BeOS (tm) Look and Feel";
	}

	/**
	* Returns a string that identifies this look and feel.  This string 
	* will be used by applications/services that want to recognize
	* well known look and feel implementations.  Presently
	* the well known names are "Motif", "Windows", "Mac", "Metal".  Note 
	* that a LookAndFeel derived from a well known superclass 
	* that doesn't make any fundamental changes to the look or feel 
	* shouldn't override this method.
	*
	* @return a string that identifies this look and feel
	*/
	public String getID() {
		return "BeOS";
	}

	/**
	* Returns a short string that identifies this look and feel, e.g.
	* "CDE/Motif".  This string should be appropriate for a menu item.
	*
	* @return a short string that identifies this look and feel
	*/
	public String getName() {
		return "BeOS";
	}

	/**
	* If the underlying platform has a "native" look and feel, and this
	* is an implementation of it, return true.  For example a CDE/Motif
	* look and implementation would return true when the underlying 
	* platform was Solaris.
	*
	* @returns <code>true</code> if the platform is the BeOS
	*/
	public boolean isNativeLookAndFeel() {
		String osName = System.getProperty("os.name");
		return (osName != null) && (osName.indexOf("Be") != -1);
	}

	/**
	* Returns true if the underlying platform supports and or permits
	* this look and feel.  This method returns false if the look 
	* and feel depends on special resources or legal agreements that
	* aren't defined for the current platform.  
	* 
	* @see javax.swing.UIManager#setLookAndFeel
	* @returns <code>true</code> for all platforms
	*/ 
	public boolean isSupportedLookAndFeel() {
		return true;
	}

	/**
	* Provide BeOS L&F class names for UI components that
	* override the Basic L&F ones.
	*
	* @param table the array of UIDefaults
	*/
	protected void initClassDefaults(UIDefaults table) {
		super.initClassDefaults(table);

		String beOSLnFPackage = "com.be.swing.plaf.beos.";

		Object[] defaults = {

			"LabelUI", beOSLnFPackage + "BeOSLabelUI",
			"ButtonUI", beOSLnFPackage + "BeOSButtonUI",
			"CheckBoxUI", beOSLnFPackage + "BeOSCheckBoxUI",
			"TextAreaUI", beOSLnFPackage + "BeOSTextAreaUI",
			"TextFieldUI", beOSLnFPackage + "BeOSTextFieldUI",
			"PasswordFieldUI", beOSLnFPackage + "BeOSPasswordFieldUI",
			"ProgressBarUI", beOSLnFPackage + "BeOSProgressBarUI",
			"RadioButtonUI", beOSLnFPackage + "BeOSRadioButtonUI",
			"SliderUI", beOSLnFPackage + "BeOSSliderUI"
		};

		table.putDefaults(defaults);
	}

	/**
	* Initializes the system color defaults table.  If the
	* underlying platform is the BeOS, some colors should be
	* defined using native calls to the OS (selection color,
	* window colors, etc).
	*
	* @param table the array of UIDefaults
	*/
	protected void initSystemColorDefaults(UIDefaults table) {

		// why this isn't impl the same as the other methods
		//   is anyone's guess... f&%&% Sun

		Object[] systemColors = {

			"desktop", getBlue(),
			"activeCaption", getYellow(),
			"activeCaptionText", getBlack(),
			"activeCaptionBorder", getGray99(),
			"inactiveCaption", getGray239(),
			"inactiveCaptionText", getBlack(),
			"inactiveCaptionBorder", getGray156(),
			"window", getGray239(),
			"windowBorder", getGray99(),
			"windowText", getBlack(),
			"menu", getGray222(),
			"menuText", getBlack(),
			"text", getWhite(),
			"textText", getBlack(),
			"textHighlight", getGray156(),
			"textHighlightText", getBlack(),
			"textInactiveText", getGray132(),
			"control", getGray239(),
			"controlText", getBlack(),
//			"controlHighlight", getGray222(),
			"controlHighlight", getWhite(),
			"controlShadow", getGray156(),
			"controlDkShadow", getGray132(),
			"scrollbar", getGray206(),
			"info", getGray156(),
			"infoText", getBlack()
		};

		for (int i = 0; i < systemColors.length; i += 2) {
			table.put( (String) systemColors[i], systemColors[i+1] );
		}
	}       

	/**
	* Provide properties for all of the UI components.
	*
	* @param table the array of UIDefaults
	*/
	protected void initComponentDefaults(UIDefaults table) {
		super.initComponentDefaults(table);

		Border marginBorder = new BasicBorders.MarginBorder();

		// all BeOS L&F specific Borders should be defined in the factory class BeOSBorders
		
		Border buttonBorder = new BorderUIResource.CompoundBorderUIResource(new BeOSBorders.ButtonBorder(), marginBorder);
		Border defaultButtonBorder = new BorderUIResource.CompoundBorderUIResource(new BeOSBorders.DefaultButtonBorder(), buttonBorder);
		
		Border textComponentBorder = new BorderUIResource.CompoundBorderUIResource(new BeOSBorders.TextComponentBorder(), marginBorder);
		
		Border progressBarBorder = new BorderUIResource.CompoundBorderUIResource(new BeOSBorders.ProgressBarBorder(), marginBorder);
		
		Object[] defaults = {

			"RadioButton.icon", BeOSIconFactory.getRadioButtonIcon(),
			"RadioButton.on", LookAndFeel.makeIcon(getClass(), "icons/RadioButtonEnabledOn.gif"),
			"RadioButton.off", LookAndFeel.makeIcon(getClass(), "icons/RadioButtonEnabledOff.gif"),
			"RadioButton.pressedOn", LookAndFeel.makeIcon(getClass(), "icons/RadioButtonPressedOn.gif"),
			"RadioButton.pressedOff", LookAndFeel.makeIcon(getClass(), "icons/RadioButtonPressedOff.gif"),
			"RadioButton.disabledOn", LookAndFeel.makeIcon(getClass(), "icons/RadioButtonDisabledOn.gif"),
			"RadioButton.disabledOff", LookAndFeel.makeIcon(getClass(), "icons/RadioButtonDisabledOff.gif"),
			"RadioButton.foreground", table.get("controlText"),
			"RadioButton.background", table.get("control"),
			"RadioButton.disabledText", table.get("textInactiveText"),
			
			"CheckBox.icon", BeOSIconFactory.getCheckBoxIcon(),
			"CheckBox.on", LookAndFeel.makeIcon(getClass(), "icons/CheckBoxEnabledOn.gif"),
			"CheckBox.off", LookAndFeel.makeIcon(getClass(), "icons/CheckBoxEnabledOff.gif"),
			"CheckBox.pressedOn", LookAndFeel.makeIcon(getClass(), "icons/CheckBoxPressedOn.gif"),
			"CheckBox.pressedOff", LookAndFeel.makeIcon(getClass(), "icons/CheckBoxPressedOff.gif"),
			"CheckBox.disabledOn", LookAndFeel.makeIcon(getClass(), "icons/CheckBoxDisabledOn.gif"),
			"CheckBox.disabledOff", LookAndFeel.makeIcon(getClass(), "icons/CheckBoxDisabledOff.gif"),
			"CheckBox.foreground", table.get("controlText"),
			"CheckBox.background", table.get("control"),
			"CheckBox.disabledText", table.get("textInactiveText"),
			
			"Button.foreground", table.get("controlText"),
			"Button.background", table.get("control"),
			"Button.disabledText", table.get("textInactiveText"),
			"Button.select", getBlack(),
			"Button.selectText", getWhite(),
			"Button.border", defaultButtonBorder,
			"Button.insets", getZeroInsets(),
			"Button.defaultButtonInsets", getDefaultButtonInsets(),
			
//			"Label.font", ??
			"Label.foreground", table.get("controlText"),
			"Label.background", table.get("control"),
			"Label.disabledText", table.get("textInactiveText"),
			// to cover Sun's naming inconsistencies
			"Label.disabledForeground", table.get("textInactiveText"),
			
//			"TextField.font", ??
			"TextField.foreground", table.get("controlText"),
			"TextField.background", getWhite(),
			"TextField.caretForeground", getBlack(),
			"TextField.selectionBackground", table.get("textHighlight"),
			"TextField.selectionForeground", table.get("textHightlightText"),                       
			"TextField.disabledText", table.get("textInactiveText"),
			// YASPNI yet another Sun property naming inconsistency
			"TextField.inactiveForeground", table.get("textInactiveText"),
			"TextField.border", textComponentBorder,
			"TextField.margin", getTextMargin(),
			
//			"TextArea.font", ??
			"TextArea.foreground", table.get("controlText"),
			"TextArea.background", getWhite(),
			"TextArea.caretForeground", getBlack(),
			"TextArea.selectionBackground", table.get("textHighlight"),
			"TextArea.selectionForeground", table.get("textHightlightText"),                        
			"TextArea.disabledText", table.get("textInactiveText"),
			"TextArea.inactiveForeground", table.get("textInactiveText"),
			"TextArea.border", textComponentBorder,
			"TextArea.margin", getTextMargin(),
			
//			"PasswordField.font", ??
			"PasswordField.foreground", table.get("controlText"),
			"PasswordField.background", getWhite(),
			"PasswordField.caretForeground", getBlack(),
			"PasswordField.selectionBackground", table.get("textHighlight"),
			"PasswordField.selectionForeground", table.get("textHightlightText"),                   
			"PasswordField.disabledText", table.get("textInactiveText"),
			"PasswordField.inactiveForeground", table.get("textInactiveText"),
			"PasswordField.border", textComponentBorder,
			"PasswordField.margin", getTextMargin(),
			
			"ProgressBar.background", getWhite(),
			"ProgressBar.border", progressBarBorder,
			"ProgressBar.cellLength", new Integer(10),
			"ProgressBar.cellSpacing", new Integer(0),
//			"ProgressBar.font", ??
//			"ProgressBar.foreground", getBlue()
			"ProgressBar.foreground", getBlue49(),

//			"ProgressBar.selectionBackground", getWhite(), 
//			"ProgressBar.selectionForeground", getBlue()

			// custom ProgressBar color properties
			"ProgressBar.barMiddle", getBlue49(),
			"ProgressBar.barHighlight", getBlue132(),
			"ProgressBar.barShadow", getBlue24(),
			
			"Slider.background", table.get("control"),
//			"Slider.border", ??
			"Slider.focus", getBlue(),
			"Slider.focusInsets", getZeroInsets(),
			"Slider.foreground", table.get("controlText"),
			"Slider.highlight", table.get("controlHighlight"),
			"Slider.shadow", table.get("controlShadow"),
			"Slider.trackWidth", new Integer(6),
			"Slider.majorTickLength", new Integer(5),
			"Slider.minorTickLength", new Integer(3),
			
			// custom Slider colors (man, there's a lot of em)
			"Slider.borderColor", getGray148(),
			"Slider.trackColor", getGray82(),
			"Slider.tickColor", getGray148(),
			"Slider.cornerColor", getGray181(),
			"Slider.highlightColor", getWhite(),
			"Slider.shadowColor", getBlack(),
			"Slider.disabledTextColor", table.get("textInactiveText"),
			"Slider.horizontalThumbIcon", BeOSIconFactory.getHorizontalThumbIcon(),
			"Slider.verticalThumbIcon", BeOSIconFactory.getVerticalThumbIcon()
		};

		table.putDefaults(defaults);
	}

	// define all of the BeOS L&F UIResource objects here using
	//   a private static final variable & a package-level static method
	
	// Insets resources
	static final Insets getZeroInsets() { return zeroInsets; }
	static final Insets getTextMargin() { return textMargin; }
	static final Insets getButtonInsets() { return buttonInsets; }
	static final Insets getProgressBarInsets() { return textComponentInsets; }
	static final Insets getTextComponentInsets() { return textComponentInsets; }
	static final Insets getDefaultButtonInsets() { return defaultButtonInsets; }
	
	private static final InsetsUIResource zeroInsets = new InsetsUIResource(0,0,0,0);
	private static final InsetsUIResource textMargin = new InsetsUIResource(0,5,0,5);
	private static final InsetsUIResource buttonInsets = new InsetsUIResource(4,4,4,4);
	private static final InsetsUIResource textComponentInsets = new InsetsUIResource(2,2,2,2);
	private static final InsetsUIResource defaultButtonInsets = new InsetsUIResource(4,4,4,4);
	
	// Color resources
	static final ColorUIResource getWhite() { return white; }
	
	static final ColorUIResource getGray239() { return gray239; }
	static final ColorUIResource getGray222() { return gray222; }
	static final ColorUIResource getGray206() { return gray206; }
	static final ColorUIResource getGray189() { return gray189; }
	static final ColorUIResource getGray181() { return gray181; }
	static final ColorUIResource getGray156() { return gray156; }
	static final ColorUIResource getGray148() { return gray148; }
	static final ColorUIResource getGray132() { return gray132; }
	static final ColorUIResource getGray99() { return gray99; }
	static final ColorUIResource getGray82() { return gray82; }
	
	static final ColorUIResource getBlack() { return black; }
	
	static final ColorUIResource getBlue() { return blue; }
	static final ColorUIResource getYellow() { return yellow; }
	
	static final ColorUIResource getBlue24() { return blue24; }
	static final ColorUIResource getBlue49() { return blue49; }
	static final ColorUIResource getBlue132() { return blue132; }
	
	private static final ColorUIResource white = new ColorUIResource(255,255,255);
	
	private static final ColorUIResource gray239 = new ColorUIResource(239,235,239);
	private static final ColorUIResource gray222 = new ColorUIResource(222,219,222);
	private static final ColorUIResource gray206 = new ColorUIResource(206,203,206);
	private static final ColorUIResource gray189 = new ColorUIResource(189,186,189);
	private static final ColorUIResource gray181 = new ColorUIResource(181,178,181);
	private static final ColorUIResource gray156 = new ColorUIResource(156,154,156);
	private static final ColorUIResource gray148 = new ColorUIResource(148,146,148);
	private static final ColorUIResource gray132 = new ColorUIResource(132,130,132);
	private static final ColorUIResource gray99 = new ColorUIResource(99,97,99);
	private static final ColorUIResource gray82 = new ColorUIResource(82,81,82);
	
	private static final ColorUIResource black = new ColorUIResource(0,0,0);
	
	private static final ColorUIResource blue = new ColorUIResource(0,0,231);
	private static final ColorUIResource yellow = new ColorUIResource(255,203,0);

	// progress bar blues
	private static final ColorUIResource blue24 = new ColorUIResource(24,89,148);
	private static final ColorUIResource blue49 = new ColorUIResource(49,150,255);
	private static final ColorUIResource blue132 = new ColorUIResource(132,195,255);
	
}