/*--------------------------------------------------------------------*\
  File:      kdbg.cpp
  Creator:   Matt Bogosian <mbogosian@usa.net>
  Copyright: (c)1998, Matt Bogosian. All rights reserved.
  Description: Source file containing the hook functions for the
      kernel-loadable device driver as defined in Drivers.h. This may
      look a little like C++, but don't use any classes; the new and
      delete operators are not to be seen here. Besides, everything is
      extern "C"'d anyway. I just happen to like the conveniences that
      C++ offers (though I'm not even sure I use any of them here)....
  ID:        $Id: kdbg.cpp,v 1.3 1998/06/16 06:22:49 mattb Exp $
  Conventions:
      #defines - all uppercase letters with words separated by
          underscores.
          (E.G., #define MY_DEFINE 16).
      New data types (classes, structs, typedefs, etc.) - begin with
          an uppercase letter followed by lowercase words separated by
          uppercase letters.
          (E.G., typedef int MyTypedef;).
      Global constants (declared with const) - begin with "k_"
          followed by lowercase words separated by underscores.
          (E.G., const int k_my_constant = 16;).
      Global variables - begin with "g_" followed by lowercase words
          separated by underscores.
          (E.G., int g_my_global;).
      Local variables - begin with a lowercase letter followed by
          lowercase words separated by underscores.
          (E.G., int my_local;).
      Argument variables - begin with "a_" followed by lowercase words
          separated by underscores.
          (E.G., ..., int a_my_argument, ...).
      Functions - begin with a lowercase letter followed by lowercase
          words separated by uppercase letters.
          (E.G., void myFunction(void);).
      Static member constants (declared with const) - begin with "mk_"
          followed by lowercase words separated by underscores.
          (E.G., static const int MyClass::mk_my_constant = 16;).
      Static member variables - begin with "mg_" followed by lowercase
          words separated by underscores.
          (E.G., static int MyClass::mg_my_global = 16;).
      Member variables - begin with "m_" followed by lowercase words
          separated by underscores.
          (E.G., int m_my_member;).
      Member Functions - begin with an uppercase letter followed by
          lowercase words separated by uppercase letters.
          (E.G., void MyClass::MyFunction(void);).
\*--------------------------------------------------------------------*/


/*--------------------------------------------------------------------*\
  =-=-=-=-=-=-=-=-=-=-=-=-=- Included Files -=-=-=-=-=-=-=-=-=-=-=-=-=
\*--------------------------------------------------------------------*/

#include <KernelExport.h>
#include <Drivers.h>
#include <string.h>


/*--------------------------------------------------------------------*\
  =-=-=-=-=-=-=-= Definitions, Enums, Typedefs, Consts =-=-=-=-=-=-=-=
\*--------------------------------------------------------------------*/

// This is returned by publish_devices() to indicate to the kernel
// which devices this driver defines
static const char * const k_dev_names[] =
{
	"kdbg",
	NULL
};


/*--------------------------------------------------------------------*\
  =-=-=-=-=-=-=-=-=-=-=-=- Function Prototypes =-=-=-=-=-=-=-=-=-=-=-=
\*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*\
  Function name: static kdbgClose
  Arguments:     void * const a_cookie - the cookie associated with
                     this call.
  Returns:       status_t - B_NO_ERROR on success, an error code on
                     failure.
  Throws:        none
  Description: Hook function to handle the closing of the device.
\*--------------------------------------------------------------------*/

static status_t kdbgClose(void * const a_cookie);

/*--------------------------------------------------------------------*\
  Function name: static kdbgCtrl
  Arguments:     void * const a_cookie - the cookie associated with
                     this call.
                 const uint32 a_op - the control command.
                 const void * const a_data - the data associated with
                     the command.
                 const size_t a_data_len - the length (in bytes) of
                     a_data.
  Returns:       status_t - B_NO_ERROR on success, an error code on
                     failure.
  Throws:        none
  Description: Hook function to handle calls the controlling (e.g.,
      via ioctl()) of the device.
\*--------------------------------------------------------------------*/

static status_t kdbgCtrl(void * const a_cookie, const uint32 a_op, const void * const a_data, const size_t a_data_len);

/*--------------------------------------------------------------------*\
  Function name: static kdbgFree
  Arguments:     void * const a_cookie - the cookie associated with
                     this call.
  Returns:       status_t - B_NO_ERROR on success, an error code on
                     failure.
  Throws:        none
  Description: Hook function to handle the freeing of the device.
\*--------------------------------------------------------------------*/

static status_t kdbgFree(void * const a_cookie);

/*--------------------------------------------------------------------*\
  Function name: static kdbgOpen
  Arguments:     const char * const a_name - the name of the device to
                     open. Some device drivers can define several
                     devices; this let's you know which one is being
                     opened.
                 const uint32 a_flags - the flags (e.g., O_RDONLY)
                     sent by the call to open().
                 const void ** const a_cookie - a placeholder for the
                     a cookie to be associated with further calls to
                     the device. This is passed back to the device as
                     the first argument of all other device hook
                     functions.
  Returns:       status_t - B_NO_ERROR on success, an error code on
                     failure.
  Throws:        none
  Description: Hook function to handle the opening of the device.
\*--------------------------------------------------------------------*/

static status_t kdbgOpen(const char * const a_name, const uint32 a_flags, const void ** const a_cookie);

/*--------------------------------------------------------------------*\
  Function name: static kdbgRead
  Arguments:     void * const a_cookie - the cookie associated with
                     this call.
                 const off_t a_pos - the position from which to read.
                 void * const a_data - a placeholder for the data to
                     be read.
                 size_t * const a_data_len - the length (in bytes) of
                     a_data, or the maximum amount of data that should
                     be read from the device. The actual amount read
                     should be placed in this variable before this
                     function returns.
  Returns:       status_t - B_NO_ERROR on success, an error code on
                     failure.
  Throws:        none
  Description: Hook function to handle reading from the device.
\*--------------------------------------------------------------------*/

static status_t kdbgRead(void * const a_cookie, const off_t a_pos, void * const a_data, size_t * const a_data_len);

/*--------------------------------------------------------------------*\
  Function name: static kdbgWrite
  Arguments:     void * const a_cookie - the cookie associated with
                     this call.
                 const off_t a_pos - the position to which to write.
                 void * const a_data - the data to be written.
                 size_t * const a_data_len - the length (in bytes) of
                     a_data, or the maximum amount of data that should
                     be written to the device. The actual amount
                     written should be placed in this variable before
                     this function returns.
  Returns:       status_t - B_NO_ERROR on success, an error code on
                     failure.
  Throws:        none
  Description: Hook function to handle writing to the device.
\*--------------------------------------------------------------------*/

static status_t kdbgWrite(void * const a_cookie, const off_t a_pos, const void * const a_data, size_t * const a_data_len);


/*--------------------------------------------------------------------*\
  =-=-=-=-=-=-=-=-=-=-=-=-= Global Variables =-=-=-=-=-=-=-=-=-=-=-=-=
\*--------------------------------------------------------------------*/

// This struct is returned by find_device() when called with the
// appropriate device name
static device_hooks g_dev_hooks =
{
	reinterpret_cast<device_open_hook>(&kdbgOpen),
	reinterpret_cast<device_close_hook>(&kdbgClose),
	reinterpret_cast<device_free_hook>(&kdbgFree),
	reinterpret_cast<device_control_hook>(&kdbgCtrl),
	reinterpret_cast<device_read_hook>(&kdbgRead),
	reinterpret_cast<device_write_hook>(&kdbgWrite)
};

// Semaphore used to guard access to the kernel debugging output (but
// only for this device)
static sem_id g_sem;


/*--------------------------------------------------------------------*\
  =-=-=-=-=-=-=-=-=-=-=-= Function Definitions =-=-=-=-=-=-=-=-=-=-=-=
\*--------------------------------------------------------------------*/

//====================================================================
static status_t kdbgClose(void * const /*a_cookie*/)
//====================================================================
{
	return B_NO_ERROR;
}

//====================================================================
static status_t kdbgCtrl(void * const /*a_cookie*/, const uint32 /*a_op*/, const void * const /*a_data*/, const size_t /*a_data_len*/)
//====================================================================
{
	return B_BAD_VALUE;
}

//====================================================================
static status_t kdbgFree(void * const /*a_cookie*/)
//====================================================================
{
	return B_NO_ERROR;
}

//====================================================================
static status_t kdbgOpen(const char * const /*a_name*/, const uint32 /*a_flags*/, const void ** const /*a_cookie*/)
//====================================================================
{
	return B_NO_ERROR;
}

//====================================================================
static status_t kdbgRead(void * const /*a_cookie*/, const off_t a_pos, void * const a_data, size_t * const a_data_len)
//====================================================================
{
	status_t result;
	
	// Allow reading from the beginning only
	if (a_pos != 0)
	{
		*a_data_len = 0;
		return B_NO_ERROR;
	}
	
	// Make sure we can write at least one byte to a_data
	if (*a_data_len > 0)
	{
		// We're goin' in
		while ((result = acquire_sem(g_sem)) == B_INTERRUPTED)
		{
			if (result != B_NO_ERROR)
			{
				return result;
			}
		}
		
		// Get the current state of debugging output
		bool enabled = set_dprintf_enabled(true);
		set_dprintf_enabled(enabled);
		
		// Fill out the data
		*a_data_len = 1;
		*static_cast<char * const>(a_data) = enabled ? '1' : '0';
		
		// Punch out
		if ((result = release_sem(g_sem)) != B_NO_ERROR)
		{
			return result;
		}
	}
	
	return B_NO_ERROR;
}

//====================================================================
static status_t kdbgWrite(void * const /*a_cookie*/, const off_t a_pos, const void * const a_data, size_t * const a_data_len)
//====================================================================
{
	status_t result;
	
	// Allow writing to the beginning only
	if (a_pos != 0)
	{
		*a_data_len = 0;
		return B_NO_ERROR;
	}
	
	// Make sure we can read at least one byte to a_data
	if (*a_data_len > 0)
	{
		// We're goin' in
		while ((result = acquire_sem(g_sem)) == B_INTERRUPTED)
		{
			if (result != B_NO_ERROR)
			{
				return result;
			}
		}
		
		// Get the current state of debugging output
		bool enabled = set_dprintf_enabled(true);
		set_dprintf_enabled(enabled);
		
		*a_data_len = 1;
		
		// Look at only the first character
		switch (*static_cast<const char * const>(a_data))
		{
			// Turn it off
			case 0:
			case '0':
				if (enabled)
				{
					dprintf("kdbg: kernel dubgging output off\n");
					set_dprintf_enabled(false);
				}
				
				break;
			
			// Turn it on
			case 1:
			case '1':
				if (!enabled)
				{
					set_dprintf_enabled(true);
					dprintf("kdbg: kernel dubgging output on\n");
				}
				
				break;
		}
		
		// Punch out
		if ((result = release_sem(g_sem)) != B_NO_ERROR)
		{
			return result;
		}
	}
	
	return B_NO_ERROR;
}

#pragma export on

//====================================================================
extern "C" device_hooks *find_device(const char * const a_name)
//====================================================================
{
	if (!strcmp (a_name, k_dev_names[0]))
	{
		return &g_dev_hooks;
	}
	
	return NULL;
}

//====================================================================
extern "C" status_t init_driver(void)
//====================================================================
{
	// Try to create the semaphore we'll use
	if ((g_sem = create_sem(1, k_dev_names[0])) < B_NO_ERROR)
	{
		return g_sem;
	}
	
	return B_NO_ERROR;
}

//====================================================================
extern "C" const char **publish_devices(void)
//====================================================================
{
	return k_dev_names;
}

//====================================================================
extern "C" void uninit_driver(void)
//====================================================================
{
	delete_sem(g_sem);
}

#pragma export off
