########################################################################
# File:      kdbg.sh
# Creator:   Matt Bogosian <mbogosian@usa.net>
# Copyright: (c)1997, Matt Bogosian. All rights reserved.
# Description: Shell script to proved a convenient command line
#     interface to /dev/kdbg and /dev/dprintf. Usage is simple:
#     
#     $ kdbg.sh -1         # Turns on kernel debugging output
#     $ kdbg.sh -n I am    # Sends the args to /dev/dprintf a la echo -n
#     $ kdbg.sh funny!     # Sends the args to /dev/dprintf a la echo
#     $ kdbg.sh -e \\a     # Sends the args to /dev/dprintf a la echo -e
#     $ kdbg.sh -0         # Turns if off again
#     
#     Note: you can also give the -0 and -1 command with arguments
#     (though -0 followed by anything doesn't really do anything).
#     kdbg.sh requires bash version 2 or later (or any version which
#     has the built-in "getopts" command).
# ID:        $Id: kdbg.sh,v 1.4 1998/07/21 19:01:35 mattb Exp $
# Conventions:
#     Constants - all uppercase letters with words separated by
#         underscores.
#         (E.G., MY_DEFINE=16).
#     Variables - begin with a lowercase letter followed by lowercase
#         words separated by underscores.
#         (E.G., my_local=16).
#     Functions - begin with an lowercase letter followed by lowercase
#         words separated by uppercase letters.
#         (E.G., function myFunction).
########################################################################

# Constants
NO_ERR='0'
MIN_NUM_ARGS='1'
ERR_BAD_CMD='-1'
ERR_BAD_SHELL='-2'
ERR_EXEC='-3'
MIN_SHELL_VERS='2'
DEV_KDBG='/dev/kdbg'
DEV_DPRINTF='/dev/dprintf'
KDBG_CMD_OFF='0'
KDBG_CMD_ON='1'
ECHO_NO_NEWL='-n'
ECHO_ESCAPE='-e'

prog_name="${0##*/}"

# Check for the right version of bash
if [ -z "${BASH_VERSION}" ] || ! type getopts >/dev/null 2>/dev/null ; then
	echo "${prog_name} requires bash version ${MIN_SHELL_VERS} or later"
	exit "${ERR_BAD_SHELL}"
fi

# Check command-line options
while getopts ":01egn" opt; do
	case "${opt}" in
		0 )	to_kdbg="${KDBG_CMD_OFF}" ;;
		
		1 )	to_kdbg="${KDBG_CMD_ON}" ;;
		
		e )	echo_args="${echo_args}${ECHO_ESCAPE} " ;;
		
		g )	DEBUG='true' ;;
		
		n )	echo_args="${echo_args}${ECHO_NO_NEWL} " ;;
		
		* )	echo "Usage: ${prog_name} [ -0 | -1 ] [ -e -n ] ARGS ..."
			exit "${ERR_BAD_CMD}"
	esac
done

shift $(($OPTIND - 1))

# Check command-line arguments
if [ "${#}" -lt "${MIN_NUM_ARGS}" ] && [ -z "${to_kdbg}" ] ; then
	echo "Usage: ${prog_name} [ -0 | -1 ] [ -e -n ] ARGS ..."
	exit "${ERR_BAD_CMD}"
fi

# Check to see if the appropriate devices actually exist
if [ ! -z "${to_kdbg}" ] ; then
	if  [ ! -r "${DEV_KDBG}" ] || [ ! -w "${DEV_KDBG}" ] ; then
		echo "${prog_name}: can't find ${DEV_KDBG}"
		exit "${ERR_EXEC}"
	fi
fi

if [ ! "${#}" -lt "${MIN_NUM_ARGS}" ] ; then
	if  [ ! -r "${DEV_DPRINTF}" ] || [ ! -w "${DEV_DPRINTF}" ] ; then
		echo "${prog_name}: can't find ${DEV_DPRINTF}"
		exit "${ERR_EXEC}"
	fi
fi

# Send the command to kdbg
if [ ! -z "${to_kdbg}" ] ; then
	if [ ! -z "${DEBUG}" ] ; then
		echo echo -n "${to_kdbg}" \>"${DEV_KDBG}"
	fi
	
	echo -n "${to_kdbg}" >"${DEV_KDBG}"
fi

# Send the arguments to dprintf
if [ ! "${#}" -lt "${MIN_NUM_ARGS}" ] ; then
	if [ ! -z "${DEBUG}" ] ; then
		echo echo ${echo_args} "${@}" \>"${DEV_DPRINTF}"
	fi
	
	echo ${echo_args} "${@}" >"${DEV_DPRINTF}"
fi

exit "${NO_ERR}"
