/*
 *  mon.cpp - Machine language monitor
 *
 *  (C) 1997-1998 Christian Bauer
 */

#include <SupportDefs.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <ctype.h>
#include <unistd.h>
#include <errno.h>

extern "C" {
extern char *readline(char *prompt);
extern void add_history(char *str);
}

#include "mon.h"
#include "mon_ppc.h"
#include "mon_68k.h"
#include "mon_6502.h"


// Version/revision
static const int VERSION = 2;
static const int REVISION = 0;


// Buffer we're operating on
static bool use_real_mem = false;
static uint8 *mem;
static int mem_size;


// Streams for input, output and error messages
static FILE *fin, *fout, *ferr;

// Input line
static const int INPUT_LENGTH = 256;
static char input[INPUT_LENGTH];
static char *in_ptr;

// Current address, value of '.' in expressions
static uint32 dot_address;

// Current value of ':' in expression
static uint32 colon_value;


// Processor types (for disassembler)
enum CPUType {
	CPU_PPC,
	CPU_68K,
	CPU_6502
};


// Input tokens
enum Token {
	T_NULL,		// Invalid token
	T_END,		// End of line
	T_NUMBER,	// Hexadecimal/decimal number (uint32)
	T_STRING,	// String enclosed in ""
	T_NAME,		// Variable name
	T_DOT,		// '.'
	T_COLON,	// ':'
	T_COMMA,	// ','
	T_LPAREN,	// '('
	T_RPAREN,	// ')'
	T_PLUS,		// '+'
	T_MINUS,	// '-'
	T_MUL,		// '*'
	T_DIV,		// '/'
	T_MOD,		// '%'
	T_AND,		// '&'
	T_OR,		// '|'
	T_EOR,		// '^'
	T_SHIFTL,	// '<<'
	T_SHIFTR,	// '>>'
	T_NOT,		// '~'
	T_ASSIGN	// '='
};

static enum Token the_token;			// Last token read
static uint32 the_number;				// Contains the number if the_token==T_NUMBER
static char the_string[INPUT_LENGTH];	// Contains the string if the_token==T_STRING
static char the_name[INPUT_LENGTH];		// Contains the variable name if the_token==T_NAME


// List of variables
struct Variable {
	Variable *next;	// Pointer to next variable (must be first element of struct)
	char *name;		// Variable name
	uint32 value;	// Variable value
};

Variable *first_var;	// Pointer to first variable


// Prototypes
static void error(const char *s);
static inline uint8 char2print(uint8 c);
static void handle_abort(int sig, void *arg, vregs *r);
static void init_abort(void);
static void exit_abort(void);
static bool aborted(void);

static inline uint8 read_byte(uint32 adr);	// Memory access
static inline void write_byte(uint32 adr, uint8 b);
static inline uint16 read_half(uint32 adr);
static inline void write_half(uint32 adr, uint16 w);
static inline uint32 read_word(uint32 adr);
static inline void write_word(uint32 adr, uint32 l);

static void read_line(char *prompt);		// Scanner
static char get_char(void);
static void put_back(char c);
static enum Token get_token(void);
static enum Token get_hex_number(uint32 &i);
static enum Token get_dec_number(uint32 &i);
static enum Token get_char_number(uint32 &i);
static enum Token get_string(char *str);
static enum Token get_hex_or_name(uint32 &i, char *name);

static bool expression(uint32 *number);	// Parser
static bool eor_expr(uint32 *number);
static bool and_expr(uint32 *number);
static bool shift_expr(uint32 *number);
static bool add_expr(uint32 *number);
static bool mul_expr(uint32 *number);
static bool factor(uint32 *number);
static Variable *lookup_var(const char *s);
static Variable *insert_var(const char *s);
static void remove_var(const char *s);
static bool range_args(uint32 *adr, uint32 *end_adr, uint32 def_range);
static bool byte_string(uint8 *s, uint32 &length);

static void help(void);					// Routines for commands
static void print_expr(void);
static void redir_output(void);
static void reallocate(void);
static void shell_command(void);
static void memory_dump(void);
static void ascii_dump(void);
static void disassemble(void);
static void modify(void);
static void fill(void);
static void apply(void);
static void transfer(void);
static void compare(void);
static void hunt(void);
static void load_data(void);
static void save_data(void);
static void set_var(void);
static void clear_vars(void);


void mon(int argc, char **argv)
{
	bool done = false, interactive = true;
	char c;

	// Setup input/output streams
	fin = stdin;
	fout = stdout;
	ferr = stdout;

	// Access real memory if mon was started as "rmon"
	char *prgname = argv[0];
	char *lastslash;
	if ((lastslash = strrchr(prgname, '/')) != NULL)
		prgname = lastslash + 1;
	if (strcmp(prgname, "rmon") == 0)
		use_real_mem = true;

	// Make argc/argv point to the actual arguments
	argc--;
	argv++;
	interactive = (argc == 0);

	// Allocate buffer
	if (!use_real_mem) {
		mem_size = 0x100000;
		mem = (uint8 *)malloc(mem_size);

		// Print banner
		if (interactive)
			fprintf(ferr, "\n *** mon V%d.%d by Christian Bauer ***\n"
							" ***      Press 'h' for help     ***\n\n", VERSION, REVISION);
	}

	init_abort();

	// Read and parse command line
	while (!done) {
		if (interactive) {
			char prompt[16];
			sprintf(prompt, "[%08x]-> ", dot_address);
			read_line(prompt);
		} else {
			if (argc == 0) {
				done = true;
				break;
			} else {
				strncpy(in_ptr = input, argv[0], INPUT_LENGTH);
				argc--;
				argv++;
			}
		}
		while ((c = get_char()) == ' ') ;

		// Interpret commands
		switch (c) {
			case 'c':
				if ((c = get_char()) == 'v')	// Clear variables
					clear_vars();
				else if (c == 'd') {			// Change directory
					while ((c = get_char()) == ' ') ;
					put_back(c);
					if (chdir(in_ptr) != EOK)
						error("Cannot change directory");
				} else {						// Compare
					get_token();
					compare();
				}
				break;

			case 'd':		// Disassemble
				disassemble();
				break;

			case 'f':		// Fill
				get_token();
				fill();
				break;

			case 'h':
				get_token();
				if (the_token == T_END)
					help();	// Help
				else
					hunt();	// Hunt
				break;

			case 'i':		// ASCII dump
				get_token();
				ascii_dump();
				break;

			case 'l':		// List directory contents
				if (get_char() == 's')
					system(input);
				else
					error("Unknown command");
				break;

			case 'm':		// Memory dump
				get_token();
				memory_dump();
				break;

			case 'o':		// Redirect output
				get_token();
				redir_output();
				break;

			case 'r':		// Remove file(s)
				if (get_char() == 'm')
					system(input);
				else
					error("Unknown command");
				break;

			case 's':		// Set/show variable
				if (get_char() == 'e' && get_char() == 't') {
					get_token();
					set_var();
				} else
					error("Unknown command");
				break;

			case 't':		// Transfer
				get_token();
				transfer();
				break;

			case 'v':		// Show version
				if (get_char() == 'e' && get_char() == 'r')
					fprintf(fout, "mon V%d.%d\n", VERSION, REVISION);
				else
					error("Unknown command");
				break;

			case 'x':		// Exit
				done = true;
				break;

			case 'y':		// Apply
				apply();
				break;

			case '[':		// Load data
				get_token();
				load_data();
				break;

			case ']':		// Save data
				get_token();
				save_data();
				break;

			case ':':		// Modify memory
				get_token();
				modify();
				break;

			case '?':		// Compute expression
				get_token();
				print_expr();
				break;

			case '@':		// Reallocate buffer
				get_token();
				reallocate();
				break;

			case '\\':		// Execute shell command
				get_token();
				shell_command();
				break;

			case 0:			// Blank line
				break;

			default:		// Unknown command
				error("Unknown command");
				break;
		}
	}

	exit_abort();

	// Free buffer
	if (!use_real_mem)
		free(mem);

	// Close output file if redirected
	if (fout != ferr)
		fclose(fout);
}


/*
 *  Print error message
 */

static void error(const char *s)
{
	fprintf(ferr, "*** %s\n", s);
}


/*
 *  Convert character to printable character
 */

static inline uint8 char2print(uint8 c)
{
	return (c >= 0x20 && c <= 0x7e) ? c : '.';
}


/*
 *  CTRL-C pressed?
 */

static bool WasAborted;
static struct sigaction my_sa;

static void handle_abort(int sig, void *arg, vregs *r)
{
	WasAborted = true;
}

static void init_abort(void)
{
	WasAborted = false;
	sigemptyset(&my_sa.sa_mask);
	my_sa.sa_handler = (__signal_func_ptr)handle_abort;
	my_sa.sa_flags = 0;
	my_sa.sa_userdata = 0;
	sigaction(SIGINT, &my_sa, NULL);
}

static void exit_abort(void)
{
	my_sa.sa_handler = SIG_DFL;
	sigaction(SIGINT, &my_sa, NULL);
}

static bool aborted(void)
{
	bool ret = WasAborted;

	WasAborted = false;
	return ret;
}


/*
 *  Access to buffer
 */

static inline uint8 read_byte(uint32 adr)
{
	if (use_real_mem)
		return *(uint8 *)adr;
	else
		return mem[adr % mem_size];
}

static inline void write_byte(uint32 adr, uint8 b)
{
	if (use_real_mem)
		*(uint8 *)adr = b;
	else
		mem[adr % mem_size] = b;
}

static inline uint16 read_half(uint32 adr)
{
	if (use_real_mem)
		return *(uint16 *)adr;
	else
		return read_byte(adr) << 8 | read_byte(adr+1);
}

static inline void write_half(uint32 adr, uint16 w)
{
	if (use_real_mem)
		*(uint16 *)adr = w;
	else {
		write_byte(adr, w >> 8);
		write_byte(adr+1, w);
	}
}

static inline uint32 read_word(uint32 adr)
{
	if (use_real_mem)
		return *(uint32 *)adr;
	else
		return read_byte(adr) << 24 | read_byte(adr+1) << 16 | read_byte(adr+2) << 8 | read_byte(adr+3);
}

static inline void write_word(uint32 adr, uint32 l)
{
	if (use_real_mem)
		*(uint32 *)adr = l;
	else {
		write_byte(adr, l >> 24);
		write_byte(adr+1, l >> 16);
		write_byte(adr+2, l >> 8);
		write_byte(adr+3, l);
	}
}


/*
 *  Read a line from the keyboard
 */

static void read_line(char *prompt)
{
	static char *line_read = NULL;

	if (line_read) {
		free(line_read);
		line_read = NULL;
	}

	line_read = readline(prompt);

	if (line_read && *line_read)
		add_history(line_read);

	strncpy(in_ptr = input, line_read, INPUT_LENGTH);
	input[INPUT_LENGTH-1] = 0;
}


/*
 *  Read a character from the input line
 */

static char get_char(void)
{
	return *in_ptr++;
}


/*
 *  Stuff back a character into the input line
 */

static void put_back(char c)
{
	*(--in_ptr) = c;
}


/*
 *  Scanner: Get a token from the input line
 */

static enum Token get_token(void)
{
	char c;

	// Skip spaces
	while ((c = get_char()) == ' ') ;

	switch (c) {
		case 0:
			return the_token = T_END;
		case '(':
			return the_token = T_LPAREN;
		case ')':
			return the_token = T_RPAREN;
		case '.':
			return the_token = T_DOT;
		case ':':
			return the_token = T_COLON;
		case ',':
			return the_token = T_COMMA;
		case '+':
			return the_token = T_PLUS;
		case '-':
			return the_token = T_MINUS;
		case '*':
			return the_token = T_MUL;
		case '/':
			return the_token = T_DIV;
		case '%':
			return the_token = T_MOD;
		case '&':
			return the_token = T_AND;
		case '|':
			return the_token = T_OR;
		case '^':
			return the_token = T_EOR;
		case '<':
			if (get_char() == '<')
				return the_token = T_SHIFTL;
			else {
				error("Unrecognized token");
				return the_token = T_NULL;
			}
		case '>':
			if (get_char() == '>')
				return the_token = T_SHIFTR;
			else {
				error("Unrecognized token");
				return the_token = T_NULL;
			}
		case '~':
			return the_token = T_NOT;
		case '=':
			return the_token = T_ASSIGN;

		case '$':
			if ((the_token = get_hex_number(the_number)) == T_NULL)
				error("'$' must be followed by hexadecimal number");
			return the_token;
		case '_':
			if ((the_token = get_dec_number(the_number)) == T_NULL)
				error("'_' must be followed by decimal number");
			return the_token;
		case '\'':
			return the_token = get_char_number(the_number);
		case '"':
			return the_token = get_string(the_string);

		default:
			if (isalnum(c)) {
				put_back(c);
				return the_token = get_hex_or_name(the_number, the_name);
			}
			error("Unrecognized token");
			return the_token = T_NULL;
	}
}

static enum Token get_hex_number(uint32 &i)
{
	char c = get_char();

	i = 0;
	if (!isxdigit(c))
		return T_NULL;

	do {
		if (c < 'a')
			i = (i << 4) + (c - '0');
		else
			i = (i << 4) + (c - 'a' + 10);
		c = get_char();
	} while (isxdigit(c));

	if (isalnum(c))
		return T_NULL;
	else {
		put_back(c);
		return T_NUMBER;
	}
}

static enum Token get_dec_number(uint32 &i)
{
	char c = get_char();

	i = 0;
	if (!isdigit(c))
		return T_NULL;

	do {
		i = (i * 10) + (c - '0');
		c = get_char();
	} while (isdigit(c));

	if (isalnum(c))
		return T_NULL;
	else {
		put_back(c);
		return T_NUMBER;
	}
}

static enum Token get_char_number(uint32 &i)
{
	char c;

	i = 0;
	while ((c = get_char()) != 0) {
		if (c == '\'')
			return T_NUMBER;
		i = (i << 8) + (uint8)c;
	}

	error("Unterminated character constant");
	return T_NULL;
}

static enum Token get_string(char *str)
{
	char c;

	while ((c = get_char()) != 0) {
		if (c == '"') {
			*str = 0;
			return T_STRING;
		}
		*str++ = c;
	}

	error("Unterminated string");
	return T_NULL;
}

static enum Token get_hex_or_name(uint32 &i, char *name)
{
	char *old_in_ptr = in_ptr;
	char c;

	// Try hex number first
	if (get_hex_number(i) == T_NUMBER)
		return T_NUMBER;

	// Not a hex number, must be a variable name
	in_ptr = old_in_ptr;
	c = get_char();
	do {
		*name++ = c;
		c = get_char();
	} while (isalnum(c));

	*name = 0;
	put_back(c);
	return T_NAME;
}


/*
 *  expression = eor_expr {OR eor_expr}
 *  true: OK, false: Error
 */

static bool expression(uint32 *number)
{
	uint32 accu, expr;

	if (!eor_expr(&accu))
		return false;

	for (;;)
		switch (the_token) {
			case T_OR:
				get_token();
				if (!eor_expr(&expr))
					return false;
				accu |= expr;
				break;

			default:
				*number = accu;
				return true;
		}
}


/*
 *  eor_expr = and_expr {EOR and_expr}
 *  true: OK, false: Error
 */

static bool eor_expr(uint32 *number)
{
	uint32 accu, expr;

	if (!and_expr(&accu))
		return false;

	for (;;)
		switch (the_token) {
			case T_EOR:
				get_token();
				if (!and_expr(&expr))
					return false;
				accu ^= expr;
				break;

			default:
				*number = accu;
				return true;
		}
}


/*
 *  and_expr = shift_expr {AND shift_expr}
 *  true: OK, false: Error
 */

static bool and_expr(uint32 *number)
{
	uint32 accu, expr;

	if (!shift_expr(&accu))
		return false;

	for (;;)
		switch (the_token) {
			case T_AND:
				get_token();
				if (!shift_expr(&expr))
					return false;
				accu &= expr;
				break;

			default:
				*number = accu;
				return true;
		}
}


/*
 *  shift_expr = add_expr {(SHIFTL | SHIFTR) add_expr}
 *  true: OK, false: Error
 */

static bool shift_expr(uint32 *number)
{
	uint32 accu, expr;

	if (!add_expr(&accu))
		return false;

	for (;;)
		switch (the_token) {
			case T_SHIFTL:
				get_token();
				if (!add_expr(&expr))
					return false;
				accu <<= expr;
				break;

			case T_SHIFTR:
				get_token();
				if (!add_expr(&expr))
					return false;
				accu >>= expr;
				break;

			default:
				*number = accu;
				return true;
		}
}


/*
 *  add_expr = mul_expr {(PLUS | MINUS) mul_expr}
 *  true: OK, false: Error
 */

static bool add_expr(uint32 *number)
{
	uint32 accu, expr;

	if (!mul_expr(&accu))
		return false;

	for (;;)
		switch (the_token) {
			case T_PLUS:
				get_token();
				if (!mul_expr(&expr))
					return false;
				accu += expr;
				break;

			case T_MINUS:
				get_token();
				if (!mul_expr(&expr))
					return false;
				accu -= expr;
				break;

			default:
				*number = accu;
				return true;
		}
}


/*
 *  mul_expr = factor {(MUL | DIV | MOD) factor}
 *  true: OK, false: Error
 */

static bool mul_expr(uint32 *number)
{
	uint32 accu, fact;

	if (!factor(&accu))
		return false;

	for (;;)
		switch (the_token) {
			case T_MUL:
				get_token();
				if (!factor(&fact))
					return false;
				accu *= fact;
				break;

			case T_DIV:
				get_token();
				if (!factor(&fact))
					return false;
				if (fact == 0) {
					error("Division by 0");
					return false;
				}
				accu /= fact;
				break;

			case T_MOD:
				get_token();
				if (!factor(&fact))
					return false;
				if (fact == 0) {
					error("Division by 0");
					return false;
				}
				accu %= fact;
				break;

			default:
				*number = accu;
				return true;
		}
}


/*
 *  factor = NUMBER | NAME | DOT | COLON | (PLUS | MINUS | NOT) factor | LPAREN expression RPAREN
 *  true: OK, false: Error
 */

static bool factor(uint32 *number)
{
	switch (the_token) {
		case T_NUMBER:
			*number = the_number;
			get_token();
			return true;

		case T_NAME:{
			Variable *var;
			if ((var = lookup_var(the_name)) != NULL) {
				*number = var->value;
				get_token();
				return true;
			} else
				return false;
		}

		case T_DOT:
			*number = dot_address;
			get_token();
			return true;

		case T_COLON:
			*number = colon_value;
			get_token();
			return true;

		case T_PLUS:
			get_token();
			return factor(number);

		case T_MINUS:
			get_token();
			if (factor(number)) {
				*number = -*number;
				return true;
			} else
				return false;

		case T_NOT:
			get_token();
			if (factor(number)) {
				*number = ~*number;
				return true;
			} else
				return false;

		case T_LPAREN:
			get_token();
			if (expression(number))
				if (the_token == T_RPAREN) {
					get_token();
					return true;
				} else {
					error("Missing ')'");
					return false;
				}
			else {
				error("Error in expression");
				return false;
			}

		case T_END:
			error("Required argument missing");
			return false;

		default:
			error("'(' or number expected");
			return false;
	}
}


/*
 *  Lookup the value of a variable
 */

static Variable *lookup_var(const char *s)
{
	// Lookup variable
	for (Variable *var=first_var; var; var=var->next)
		if (!strcmp(s, var->name))
			return var;

	// Not found, error
	error("Undefined variable");
	return NULL;
}


/*
 *  Insert new variable (or redefine old)
 */

static Variable *insert_var(const char *s)
{
	// Lookup variable
	for (Variable *var=first_var; var; var=var->next)
		if (!strcmp(s, var->name))
			return var;

	// Insert new variable
	Variable *var = new Variable;
	var->name = strdup(s);
	var->next = first_var;
	first_var = var;
	return var;
}


/*
 *  Remove variable
 */

static void remove_var(const char *s)
{
	Variable *var, *prev = (Variable *)&first_var;

	// Lookup variable and remove it
	for (var=prev->next; var; prev=var, var=var->next)
		if (!strcmp(s, var->name)) {
			prev->next = var->next;
			free(var->name);
			free(var);
			return;
		}
}


/*
 *  range_args = [expression] [[COMMA] expression] END
 *
 *  Read start address to "adr", end address to "end_adr".
 *  "adr" defaults to '.', "end_adr" defaults to '.'+def_range
 *
 *  true: OK, false: Error
 */

static bool range_args(uint32 *adr, uint32 *end_adr, uint32 def_range)
{
	*adr = dot_address;
	*end_adr = dot_address + def_range;

	if (the_token == T_END)
		return true;
	else {
		if (!expression(adr))
			return false;
		*end_adr = *adr + def_range;
		if (the_token == T_END)
			return true;
		else {
			if (the_token == T_COMMA) get_token();
			if (!expression(end_adr))
				return false;
			return the_token == T_END;
		}
	}
}


/*
 *  byte_string = (expression | STRING) {COMMA (expression | STRING)} END
 */

static bool byte_string(uint8 *s, uint32 &len)
{
	uint32 value;

	len = 0;
	goto start;

	for (;;) {
		if (the_token == T_COMMA) {
			get_token();

start:
			if (the_token == T_STRING) {
				uint8 *p = (uint8 *)the_string;
				while ((*s++ = *p++) != 0) ;
				s--;
				len += strlen(the_string);
				get_token();
			} else if (expression(&value)) {
				*s++ = value;
				len++;
			} else
				return false;

		} else if (the_token == T_END)
			return true;
		else {
			error("',' expected");
			return false;
		}
	}
}


/*
 *  Display help
 *  h
 */

static void help(void)
{
	fprintf(fout,
		"h                        This help text\n"
		"ver                      Show version\n"
		"x                        Quit mon\n"
		"? expression             Calculate expression\n"
		"@ [size]                 Reallocate buffer\n"
		"i [start [end]]          ASCII memory dump\n"
		"m [start [end]]          Hex/ASCII memory dump\n"
		"d [start [end]]          Disassemble PowerPC code\n"
		"d68 [start [end]]        Disassemble 680x0 code\n"
		"d65 [start [end]]        Disassemble 6502 code\n"
		": start string           Modify memory\n"
		"f start end string       Fill memory\n"
		"y[b|h|w] start end expr  Apply expression to memory\n"
		"t start end dest         Transfer memory\n"
		"c start end dest         Compare memory\n"
		"h start end string       Search for byte string\n"
		"\\ \"command\"              Execute shell command\n"
		"ls [args]                List directory contents\n"
		"rm [args]                Remove file(s)\n"
		"cd directory             Change current directory\n"
		"o [\"file\"]               Redirect output\n"
		"[ start \"file\"           Load data from file\n"
		"] start size \"file\"      Save data to file\n"
		"set [var[=value]]        Set/clear/show variables\n"
		"cv                       Clear all variables\n"
	);
}


/*
 *  Compute and display expression
 *  ? expression
 */

static void print_expr(void)
{
	uint32 val;

	if (!expression(&val))
		return;
	if (the_token != T_END) {
		error("Too many arguments");
		return;
	}

	if (val > 0x7fffffff) {
		fprintf(fout, "Hex unsigned:  $%08x\n"
					  "Hex signed  : -$%08x\n"
					  "Dec unsigned:  %u\n"
					  "Dec signed  : %d\n", val, -val, val, val);
		fprintf(fout, "Char        : '%c%c%c%c'\n", char2print(val >> 24), char2print(val >> 16), char2print(val >> 8), char2print(val));
	} else {
		fprintf(fout, "Hex : $%08x\n"
					  "Dec : %d\n", val, val);
		fprintf(fout, "Char: '%c%c%c%c'\n", char2print(val >> 24), char2print(val >> 16), char2print(val >> 8), char2print(val));
	}
}


/*
 *  Redirect output
 *  o [file]
 */

static void redir_output(void)
{
	// Close old file
	if (fout != ferr) {
		fclose(fout);
		fout = ferr;
		return;
	}

	// No argument given?
	if (the_token == T_END)
		return;

	// Otherwise open file
	if (the_token == T_STRING) {
		get_token();
		if (the_token != T_END) {
			error("Too many arguments");
			return;
		}
		if (!(fout = fopen(the_string, "w")))
			error("Unable to open file");
	} else
		error("'\"' around file name expected");
}


/*
 *  Reallocate buffer
 *  @ [size]
 */

static void reallocate(void)
{
	uint32 size;

	if (use_real_mem) {
		fprintf(ferr, "Cannot reallocate buffer in real mode\n", mem_size);
		return;
	}

	if (the_token == T_END) {
		fprintf(ferr, "Buffer size: %08x bytes\n", mem_size);
		return;
	}

	if (!expression(&size))
		return;
	if (the_token != T_END) {
		error("Too many arguments");
		return;
	}

	if ((mem = (uint8 *)realloc(mem, size)) != NULL)
		fprintf(ferr, "Buffer size: %08x bytes\n", mem_size = size);
	else
		fprintf(ferr, "Unable to reallocate buffer\n");
}


/*
 *  Execute shell command
 *  \ "command"
 */

static void shell_command(void)
{
	if (the_token != T_STRING) {
		error("'\"' around command expected");
		return;
	}
	get_token();
	if (the_token != T_END) {
		error("Too many arguments");
		return;
	}
	system(the_string);
}


/*
 *  Memory dump
 *  m [start [end]]
 */

#define MEMDUMP_BPL 16  // Bytes per line

static void memory_dump(void)
{
	uint32 adr, end_adr;
	uint8 mem[MEMDUMP_BPL + 1];

	mem[MEMDUMP_BPL] = 0;

	if (!range_args(&adr, &end_adr, 16 * MEMDUMP_BPL - 1))  // 16 lines unless end address specified
		return;

	while (adr <= end_adr && !aborted()) {
		fprintf(fout, "%08x:", use_real_mem ? adr: adr % mem_size);
		for (int i=0; i<MEMDUMP_BPL; i++, adr++) {
			if (i % 4 == 0)
				fprintf(fout, " %08x", read_word(adr));
			mem[i] = char2print(read_byte(adr));
		}
		fprintf(fout, "  '%s'\n", mem);
	}

	dot_address = adr;
}


/*
 *  ASCII dump
 *  i [start [end]]
 */

#define ASCIIDUMP_BPL 64  // Bytes per line

static void ascii_dump(void)
{
	uint32 adr, end_adr;
	uint8 str[ASCIIDUMP_BPL + 1];

	str[ASCIIDUMP_BPL] = 0;

	if (!range_args(&adr, &end_adr, 16 * ASCIIDUMP_BPL - 1))  // 16 lines unless end address specified
		return;

	while (adr <= end_adr && !aborted()) {
		fprintf(fout, "%08x:", use_real_mem ? adr : adr % mem_size);
		for (int i=0; i<ASCIIDUMP_BPL; i++, adr++)
			str[i] = char2print(read_byte(adr));
		fprintf(fout, " '%s'\n", str);
	}

	dot_address = adr;
}


/*
 *  Disassemble
 *  d [start [end]]
 */

static void disassemble(void)
{
	uint32 adr, end_adr;
	CPUType type = CPU_PPC;
	char c;

	if ((c = get_char()) == '6') {
		if ((c = get_char()) == '5')
			type = CPU_6502;
		else if (c == '8')
			type = CPU_68K;
		else {
			error("Unknown command");
			return;
		}
	} else if (c && c != ' ') {
		error("Unknown command");
		return;
	}

	get_token();

	if (!range_args(&adr, &end_adr, 16 * 4 - 1))  // 16 lines unless end address specified
		return;

	if (type == CPU_PPC)
		while (adr <= end_adr && !aborted()) {
			uint32 w = read_word(adr);
			fprintf(fout, "%08x: %08x\t", use_real_mem ? adr : adr % mem_size, w);
			disass_ppc(fout, use_real_mem ? adr : adr % mem_size, w);
			adr += 4;
		}
	else if (type == CPU_6502)
		while (adr <= end_adr && !aborted()) {
			uint8 op = read_byte(adr);
			uint8 lo = read_byte(adr + 1);
			uint8 hi = read_byte(adr + 2);
			fprintf(fout, "%08x: ", use_real_mem ? adr : adr % mem_size);
			adr += disass_6502(fout, use_real_mem ? adr : adr % mem_size, op, lo, hi);
		}
	else
		while (adr <= end_adr && !aborted()) {
			uint16 buf[8];
			buf[0] = read_half(adr);
			buf[1] = read_half(adr + 2);
			buf[2] = read_half(adr + 4);
			buf[3] = read_half(adr + 6);
			buf[4] = read_half(adr + 8);
			buf[5] = read_half(adr + 10);
			buf[6] = read_half(adr + 12);
			buf[7] = read_half(adr + 14);
			fprintf(fout, "%08x: ", use_real_mem ? adr : adr % mem_size);
			adr += disass_68k(fout, use_real_mem ? adr : adr % mem_size, buf);
		}

	dot_address = adr;
}


/*
 *  Modify memory
 *  : addr bytestring
 */

static void modify(void)
{
	uint32 adr, len, src_adr = 0;
	uint8 str[256];

	if (!expression(&adr))
		return;
	if (!byte_string(str, len))
		return;

	while (src_adr < len)
		write_byte(adr++, str[src_adr++]);

	dot_address = adr;
}


/*
 *  Fill
 *  f start end bytestring
 */

static void fill(void)
{
	uint32 adr, end_adr, len, src_adr = 0;
	uint8 str[256];

	if (!expression(&adr))
		return;
	if (!expression(&end_adr))
		return;
	if (!byte_string(str, len))
		return;

	while (adr <= end_adr)
		write_byte(adr++, str[src_adr++ % len]);
}


/*
 *  Apply expression to memory
 *  y[b|h|w] start end expression
 */

static void apply(void)
{
	uint32 adr, end_adr, value;
	int size;

	switch (get_char()) {
		case ' ':
		case 'b':
			size = 1;
			break;
		case 'h':
			size = 2;
			break;
		case 'w':
			size = 4;
			break;
		default:
			error("Unknown command");
			return;
	}

	char *old_in_ptr = in_ptr;
	get_token();
	if (!expression(&adr))
		return;
	if (!expression(&end_adr))
		return;
	if (!expression(&value))
		return;
	if (the_token != T_END) {
		error("Too many arguments");
		return;
	}

	switch (size) {
		case 1:
			while (adr<=end_adr) {
				colon_value = read_byte(adr);
				dot_address = adr;

				in_ptr = old_in_ptr;
				get_token();
				expression(&value);	// Skip start address
				expression(&value);	// Skip end address
				expression(&value);

				write_byte(adr, value);
				adr++;
			}
			break;
		case 2:
			while (adr<=end_adr) {
				colon_value = read_half(adr);
				dot_address = adr;

				in_ptr = old_in_ptr;
				get_token();
				expression(&value);	// Skip start address
				expression(&value);	// Skip end address
				expression(&value);

				write_half(adr, value);
				adr += 2;
			}
			break;
		case 4:
			while (adr<=end_adr) {
				colon_value = read_word(adr);
				dot_address = adr;

				in_ptr = old_in_ptr;
				get_token();
				expression(&value);	// Skip start address
				expression(&value);	// Skip end address
				expression(&value);

				write_word(adr, value);
				adr += 4;
			}
			break;
	}

	dot_address = adr;
}


/*
 *  Transfer memory
 *  t start end dest
 */

static void transfer(void)
{
	uint32 adr, end_adr, dest;
	int num;

	if (!expression(&adr))
		return;
	if (!expression(&end_adr))
		return;
	if (!expression(&dest))
		return;
	if (the_token != T_END) {
		error("Too many arguments");
		return;
	}

	num = end_adr - adr + 1;

	if (dest < adr)
		for (int i=0; i<num; i++)
			write_byte(dest++, read_byte(adr++));
	else {
		dest += end_adr - adr;
		for (int i=0; i<num; i++)
			write_byte(dest--, read_byte(end_adr--));
	}
}


/*
 *  Compare
 *  c start end dest
 */

static void compare(void)
{
	uint32 adr, end_adr, dest;
	int num = 0;

	if (!expression(&adr))
		return;
	if (!expression(&end_adr))
		return;
	if (!expression(&dest))
		return;
	if (the_token != T_END) {
		error("Too many arguments");
		return;
	}

	while (adr <= end_adr && !aborted()) {
		if (read_byte(adr) != read_byte(dest)) {
			fprintf(fout, "%08x ", use_real_mem ? adr : adr % mem_size);
			num++;
			if (!(num & 7))
				fputc('\n', fout);
		}
		adr++; dest++;
	}

	if (num & 7)
		fputc('\n', fout);
	fprintf(fout, "%d byte(s) different\n", num);
}


/*
 *  Search for byte string
 *  h start end bytestring
 */

static void hunt(void)
{
	uint32 adr, end_adr, len;
	uint8 str[256];
	int num = 0;

	if (!expression(&adr))
		return;
	if (!expression(&end_adr))
		return;
	if (!byte_string(str, len))
		return;

	while ((adr+len-1) <= end_adr && !aborted()) {
		int i;

		for (i=0; i<len; i++)
			if (read_byte(adr + i) != str[i])
				break;

		if (i == len) {
			fprintf(fout, "%08x ", use_real_mem ? adr : adr % mem_size);
			num++;
			if (num == 1)
				dot_address = adr;
			if (!(num & 7))
				fputc('\n', fout);
		}
		adr++;
	}

	if (num & 7)
		fputc('\n', fout);
	fprintf(fout, "Found %d occurrences\n", num);
}


/*
 *  Load data
 *  [ start "file"
 */

static void load_data(void)
{
	uint32 start_adr;
	FILE *file;
	int fc;

	if (!expression(&start_adr))
		return;
	if (the_token == T_END) {
		error("Missing file name");
		return;
	}
	if (the_token != T_STRING) {
		error("'\"' around file name expected");
		return;
	}
	get_token();
	if (the_token != T_END) {
		error("Too many arguments");
		return;
	}

	if (!(file = fopen(the_string, "rb")))
		error("Unable to open file");
	else {
		uint32 adr = start_adr;

		while ((fc = fgetc(file)) != EOF)
			write_byte(adr++, fc);
		fclose(file);

		fprintf(ferr, "%08x bytes read from %08x to %08x\n", adr - start_adr, use_real_mem ? start_adr : start_adr % mem_size, use_real_mem ? adr-1 : (adr-1) % mem_size);
		dot_address = adr;
	}
}


/*
 *  Save data
 *  ] start size "file"
 */

static void save_data(void)
{
	uint32 start_adr, size;
	FILE *file;

	if (!expression(&start_adr))
		return;
	if (!expression(&size))
		return;
	if (the_token == T_END) {
		error("Missing file name");
		return;
	}
	if (the_token != T_STRING) {
		error("'\"' around file name expected");
		return;
	}
	get_token();
	if (the_token != T_END) {
		error("Too many arguments");
		return;
	}

	if (!(file = fopen(the_string, "wb")))
		error("Unable to create file");
	else {
		uint32 adr = start_adr, end_adr = start_adr + size - 1;

		while (adr <= end_adr)
			fputc(read_byte(adr++), file);
		fclose(file);

		fprintf(ferr, "%08x bytes written from %08x to %08x\n", size, use_real_mem ? start_adr : start_adr % mem_size, use_real_mem ? end_adr : end_adr % mem_size);
	}
}


/*
 *  Set/clear/show variables
 *  set [var[=value]]
 */

static void set_var(void)
{
	if (the_token == T_END) {

		// Show all variables
		if (first_var == NULL)
			fprintf(fout, "No variables defined\n");
		else
			for (Variable *var=first_var; var; var=var->next)
				fprintf(fout, "%s = %08x\n", var->name, var->value);

	} else if (the_token == T_NAME) {
		char var_name[256];
		strcpy(var_name, the_name);
		get_token();
		if (the_token == T_ASSIGN) {

			// Set variable
			uint32 value;
			get_token();
			if (!expression(&value))
				return;
			if (the_token != T_END) {
				error("Too many arguments");
				return;
			}
			insert_var(var_name)->value = value;

		} else if (the_token == T_END) {

			// Clear variable
			remove_var(var_name);

		} else
			error("'=' expected");
	} else
		error("Variable name expected");
}


/*
 *  Clear all variables
 *  cv
 */

static void clear_vars(void)
{
	Variable *var, *next;
	for (var=first_var; var; var=next) {
		free(var->name);
		next = var->next;
		free(var);
	}
	first_var = NULL;
}
