// SeqGlobal.h (nanodot)
// Handles timing of sequence playback & hands off to
// Track objects for actual note generation
// e.moon apr98

// +++++ handle cloning

#pragma once
#include <MidiKit.h>
#include <KernelKit.h>
#include "Track.h"
#include "MidiThruText.h"
#include "NanoError.h"


class SeqGlobal : public BMidi {
public: // classes
	// helper class: represents one MIDI port
	struct PortEntry {
		PortEntry(char* pszName, BMidiPort* pPort) {
			m_pszName = strdup(pszName); m_pPort = pPort;
		}
		~PortEntry() { delete [] m_pszName; }
		
		char* m_pszName;
		BMidiPort* m_pPort;
	};
	
public: // interface
	SeqGlobal();
	~SeqGlobal();
	
	// copy/clone c'tor & operator:
	SeqGlobal(SeqGlobal& clone); // NYI
	SeqGlobal& operator=(SeqGlobal& clone); // NYI

	void addTrack(Track* pTrack);
	BList* /* of Track */ getTrackList() { return &m_trackList; }

	// tempo is measured in thousandths of a BPM
	void setTempo(int32 nTempo);
	int32 getTempo() { return m_nTempo; }
	
	void setPort(PortEntry* pPortEntry);
	PortEntry* getPort() { return m_pCurPortEntry; }
	
	void setChannel(int16 nChannel);
	int16 getChannel() { return m_nCurChannel; }
	
	// mute controller
	void setMuted(bool bMuted);
	bool isMuted() const { return m_bMuted; }
	
	// am I auto-starting:
	bool isStarting() const { return m_bStarting; }
	
	// reset all tracks' position pointers
	void rewind();
	
	// set all tracks' position pointers to the given value
	void setPos(uint16 nPos);
	
	// set the time the first note will be played (used for auto-start)
	void setStartTime(double fStartTime) { m_fStartTime = fStartTime; }
	
	// Start(): overridden to set m_bStarting to false
	status_t Start() {
		m_bStarting = false;
		return BMidi::Start();
	}
	
	// main loop:
	void Run();

	// returns all MIDI ports found:	
	BList* /* of PortEntry */ getPortList() { return &m_portList; }

protected: // helpers

	// calculate the delay for one 'tick' or sequencer step:
	double tempoDelay(int nDivision=4); // default division (16th notes)
	
	// build a list of available MIDI ports (and open each one)
	void enumeratePorts();
	
	// do basic c'tor initialization:
	void init();
	
	// clone self from given SeqGlobal:
	void initClone(SeqGlobal& clone);
	
	// create a default set of tracks:
	void initDefaultTracks();

	// register another SeqGlobal object to be automatically started
	// on the next tick
	// (super-primitive synchronization!)
	
	void registerClone(SeqGlobal* pClone); // NYI
	
	// convenience functions: lock & unlock tempo/parameters
	void lockTempo() {
		if(acquire_sem(m_lockTempo) != B_NO_ERROR)
			NanoError("SeqGlobal::AddTrack(): couldn't acquire m_lockTempo").
				displayAndQuit();
	}
	
	void unlockTempo() {
		if(release_sem(m_lockTempo) != B_NO_ERROR)
			NanoError("SeqGlobal::addTrack(): couldn't release m_lockTempo").
				displayAndQuit();
	}
	
protected: // STATIC MEMBERS
	static const int16 s_nNumDefaultTracks;

private: // implementation
	BList /* of PortEntry */ m_portList;
	
	// current port
	PortEntry* m_pCurPortEntry;
	
	// merge object (connects to current port)
	MidiThruText* m_pMidiText;

	// current channel (used for new tracks)
	int m_nCurChannel;

	// muted? (disconnects from output port)
	bool m_bMuted;
	
	// the tracks
	BList m_trackList;
	
	// clones to automatically start:
	BList m_registeredCloneList;
	
	// If this sequencer was cloned while its parent was running, m_bStarting
	// will be true until Start() is called, and fStartTime will be set
	// to the time at which the first note should be sent
	bool m_bStarting;
	double m_fStartTime;

	// current tempo:
	int32 m_nTempo;
	
	// tempo lock object
	sem_id m_lockTempo;
};