// Track.cpp (nanodot)
// e.moon apr98

#include <stdio.h>

#include "Track.h"

/* static */
const uint16 Track::s_nDefaultSeqLength = 16;

// c'tor, d'tor:

Track::Track() :
	m_nSeqLength(s_nDefaultSeqLength),
	m_nSeqPos(0),
	m_nChannel(1),
	m_nPitch(36), // C 1
	m_pOutPort(NULL) {
	
	init();
	
	// clear the pattern:
	for(uint16 nCount = 0; nCount < m_nSeqLength; nCount++)
		m_pbSeq[nCount] = false;
}

// copy c'tor
Track::Track(const Track& clone) :
	m_nSeqLength(clone.m_nSeqLength),
	m_nSeqPos(clone.m_nSeqPos),
	m_nChannel(clone.m_nChannel),
	m_nPitch(clone.m_nPitch),
	m_pOutPort(clone.m_pOutPort) {

	init();
	
	// copy pattern:
	memcpy(m_pbSeq, clone.m_pbSeq, m_nSeqLength*sizeof(bool));
	
	// connect to output port:
	Connect(m_pOutPort);
}

// copy operator
Track& Track::operator=(const Track& clone) {

	m_nSeqLength = clone.m_nSeqLength;
	m_nSeqPos = clone.m_nSeqPos;
	m_nChannel = clone.m_nChannel;
	m_nPitch = clone.m_nPitch;
	m_pOutPort = clone.m_pOutPort;
	
	// copy pattern:
	memcpy(m_pbSeq, clone.m_pbSeq, m_nSeqLength*sizeof(bool));

	// connect to output port:
	Connect(m_pOutPort);

	return *this;
}

Track::~Track() {
	// delete pattern:
	delete [] m_pbSeq;
	
	// delete semaphore
	delete_sem(m_lockParams);
}

// c'tor helper
void Track::init() {

	// alloc sequence:
	m_pbSeq = new bool[m_nSeqLength];

	// create semaphore:
	m_lockParams = create_sem(1, "m_lockParams");
	if(m_lockParams < B_NO_ERROR) {
		(new BAlert("LOCK ERROR", "Couldn't create m_lockParams",
			"Ouch", NULL, NULL,	B_WIDTH_AS_USUAL, B_STOP_ALERT))->Go();
		return;
	}
}

void Track::setPort(BMidi* pPort) {
	lockParams();
	
	// unplug the cable (if necessary)
	if(m_pOutPort != NULL)
		Disconnect(m_pOutPort);
		
	m_pOutPort = pPort;
	
	// plug it back in (if necessary)
	if(m_pOutPort != NULL)
		Connect(m_pOutPort);
	
	unlockParams();
}

//----------------------------------------------------------------------//

void Track::tickOn(ulong nWhen) {
	// play one sequence position's worth:
	if(m_pbSeq[m_nSeqPos]) {
		SprayNoteOn(m_nChannel, m_nPitch, 127, nWhen);
	}
}

void Track::tickOff(ulong nWhen) {
	if(m_pbSeq[m_nSeqPos]) {
		SprayNoteOff(m_nChannel, m_nPitch, 0, nWhen);
	}
		
	// increment position in sequence:
	if(++m_nSeqPos > m_nSeqLength-1)
		m_nSeqPos = 0;
}


void Track::setData(uint16 nPos, bool bValue) {

	lockParams();
	
	assert(nPos >= 0);
	assert(nPos < 16);
	m_pbSeq[nPos] = bValue;

	unlockParams();
}

// -- end Track.cpp --