//-----------------------------------------------------------------------------
// File and Version Information
//      $Id: parser.hh,v 1.1 1999/01/23 02:30:36 wglp09 Exp $
//
// Description:
//      parser.hh
//      Header file
//      A simple parser class.
// 
// Environment:
//	Software developed for the BaBar Detector at the SLAC B-Factory.
//
// Author List:
//      W.G.J. Langeveld, SLAC
//
// Modification History:
//      1998 - Langeveld : created
//-----------------------------------------------------------------------------
#ifndef PARSER_HH
#define PARSER_HH

#define TK_NULL    0x0000                  // null token
#define TK_INT     0x0001                  // integers
#define TK_FLOAT   0x0002                  // floating point number
#define TK_NUM     0x0003                  // numeric token
#define TK_ALPHA   0x0004                  // alphabetical token
#define TK_STRING  0x0008                  // character string
#define TK_ASCII   0x0010                  // other ascii token
#define TK_SUB     0x0020                  // sub-expression
#define TK_OP      0x0040                  // operator
#define TK_LB      0x0080                  // left brace
#define TK_RB      0x0100                  // right brace
#define TK_SD      0x0200                  // string delimiter
#define TK_COMMENT 0x0400                  // comment
#define TK_EOL     0x0800                  // end-of-line chars
#define TK_CONT    0x1000                  // continuation chars
#define TK_SEP     0x2000                  // separators
#define TK_ANY     0xFFFFFFFF              // match any token
#define TK_TOKEN   (TK_ALPHA | TK_NUM | TK_ASCII | TK_STRING | TK_SUB | \
                    TK_OP | TK_SEP)        // match a set of usual tokens.

class ParserToken {
public:
   ParserToken();
   ~ParserToken();

   char          *tostring(void);          // convert to string (expand escapes)
   char          *get(void);               // get the token string

   char          *typestring(void);        // get the string for a type.

   void           dump(FILE * = 0);        // print out the current token
   void           dump(char *);            // get printable info about current token

   long           match(char *);           // return true if token matches

   long           length;
   unsigned long  type;
   long           line_pos;
   long           line_num;
   long           file_pos;
                   
private:
   friend class ParsedStream;
   friend class BookMark;

   ParserToken(const ParserToken &);
   ParserToken &operator=(const ParserToken &);

   ParserToken   *insert(char *, long, long, long, long, long);
   void           remove(void);

   ParserToken   *_next;
   ParserToken   *_previous;

   char          *_token;
   char          *_escaped;
};

class BookMark {
public:
   BookMark();
   BookMark(const BookMark &);
   BookMark &operator=(const BookMark &);
   virtual ~BookMark();

   ParserToken   *first(unsigned long = TK_TOKEN);    // get the first token
   ParserToken   *current(void);                      // get the current token
   ParserToken   *next(unsigned long = TK_TOKEN);     // get the next token
   ParserToken   *previous(unsigned long = TK_TOKEN); // get the previous token;

   ParserToken   *firstline(void);           // get the first line
   ParserToken   *nextline(void);            // get the next line
   ParserToken   *previousline(void);        // get the previous line

private:
   friend class ParsedStream;
/*
*   The actual token list
*/
   ParserToken   *_head;
/*
*   Various pointers
*/
   ParserToken   *_current;
   ParserToken   *_tail;
};

class ParsedStream : public BookMark {
public:
   ParsedStream(FILE *fp = 0);
   ParsedStream(char *);
   ParsedStream(ParserToken *);
   ~ParsedStream();

   ParserToken   *insert(char *, long, long, long, long, long);
   ParserToken   *addtail(char *, long, long, long, long, long);
   ParserToken   *addhead(char *, long, long, long, long, long);

   long           setwhitespace(char *);
   long           setseparators(char *);
   long           seteolchars(char *);
   long           setcommentchars(char *);
   long           setcontinuationchars(char *);
   long           setoperators(char *);
   long           setleftbrackets(char *);
   long           setrightbrackets(char *);
   long           setstringdelimiters(char *);

   BookMark      *bookmark(void);               // set a bookmark at the current token
   ParserToken   *tobookmark(const BookMark *); // go to the specified bookmark.
   void           deletebookmark(BookMark *); // delete a bookmark

   long           parse(void);

private:
   ParsedStream(const ParsedStream &);
   ParsedStream &operator=(const ParsedStream &);
   void           init(void);

   char          *_cache;
/*
*   An automatically growing list of bookmarks
*/
   BookMark     **_bookmark;
   long           _bookmarkalloc;
   long           _nbookmarks;

   char          *_whitespace;
   char          *_separators;
   char          *_eolchars;
   char          *_commentchars;
   char          *_continuationchars;
   char          *_operators;
   char          *_leftbrackets;
   char          *_rightbrackets;
   char          *_stringdelimiters;

   void           addtoken(void);
   long           readtoken(void);
   void           findmatch(void);

   long           _cur_line_num;
   long           _cur_line_pos;
   long           _cur_file_pos;
   long           _cur_tok_len;
   unsigned long  _cur_tok_type;

   char          *_cur_cache_ptr;
   char          *_last_cache_ptr;
};

long axtol(char *);

#endif
