/*******************************************************************************
**
** (c) Copyright 1996 Novell, Inc.  All rights reserved.
**
**      File:  CgiMain.h : NetWare HTTP CGI Extension Interface Main File.
**
**  $Logfile:   J:/webserv/prodtree/disk1/web/samples/cginph/src/vcs/cgimain.c_v  $
** $Revision:   1.1  $
**     $Date:   30 Apr 1996 10:36:58  $
**   $Author:   PTANG  $
**
**  Comments:  This file contains the CGI Registration and Shutdown
**             routines necessary to interface with the NetWare HTTP
**             Server.
** 
** CGI Extention Routines Contained in this File:
**
**    main                 // System entry routine
**    RequestHandler       // CGI extension processor entry routine 
**    SigTermHandler       // Shutdown/deinitialization routine
**    SysMsgTblInit        // Initialize and register system message table
**    UnloadHandler        // Process unload requests from the HTTP server
**
**
**      $Log:   J:/webserv/prodtree/disk1/web/samples/cginph/src/vcs/cgimain.c_v  $
 * 
 *    Rev 1.1   30 Apr 1996 10:36:58   PTANG
 * G2
**
*******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <conio.h>
#include <time.h>
#include <signal.h>
#include <process.h>

#include "xxDecl.h"           // CGI Extension private definitions
#include "cgiApi.h"           // HTTP.NLM:  CGI Extension Prototype Definitions
#include "csSysMsg.h"         // CSSYSMSG.NLM:  SysMsg Prototype Definitions
#include "cgiNph.Fpd"         // PC-Lint Generated Prototype Definitions File.

/******************************************************************************/
/************************** Global Variables **********************************/
/******************************************************************************/
// The global status flag is currently used only to indicate when an unload
// request has been received from the HTTP Server.  The extension processor
// NLM may want to check this flag at various points of execution to determine
// whether a premature exit could or should be taken.  Other than that, the
// remaining bits are open for your definition and are defined in the private
// declaration file, xxDecl.h

LONG     xx_Status               =  0;       // Global status flag for this NLM

LONG     xx_NlmID                =  0;       // This modules NLM ID
LONG     xx_NlmHandle            =  0;       // This Modules NLM Handle
LONG     xx_ThreadID             =  0;       // This modules thread ID, used with suspend/resume thread
LONG     xx_ThreadGroupID        =  0;       // This modules thread group ID
LONG     xx_InRequestHandler     =  0;       // Count of number of Http threads executing in this NLM
LONG     xx_CgiXID               =  0;       // CGI Extension ID returned from CgiExtReg()

char     xx_NlmName     [16]     =  {"CgiNph"}; // change this to the name of your extension NLM
BYTE     xx_NlmPath     [_MAX_PATH];         // CGI Extension NLM's NetWare load path
BYTE  *  xx_HelpTable            =  NULL;    // ptr to loaded CGI NwsNut Help Table
BYTE  ** xx_MsgTable             =  NULL;    // ptr to loaded CGI NLS message table
LONG     xx_MsgCount             =  0;       // count of messages in message table
LONG     xx_LanguageID           =  CSSMC_DefaultLanguage;  // requests the NetWare default language


/*******************************************************************************
 *
 * Name:          main()
 *
 * This routine is called by the OS when the extension NLM is loaded.  It
 * serves to initialize the generic global viables, register its NLS enabled
 * messages with the csSysMsg.Nlm, initialize any extension processor specific
 * variables or structures, and finally, register itself with the NetWare HTTP
 * Server.
 *
 * Following registration, the main startup thread suspends itself and waits
 * for a shutdown event.  Since the startup thread will linger for the life of
 * NLM, it has been compiled with a stack size of only 1024 bytes.  Therefore,
 * if the CGI extension processor specific portion of the initialization
 * requires more stack space, you must increase the stack size as necessary.
 *
 * Once registered with the HTTP Server, this NLM must be ready to process
 * requests directed to it by the Server.  Note that request processing will
 * be done on an HTTP Server thread with a stack size of at least 16KB.  Ample
 * stack space is required for CGI processing and therefore, it is recommended
 * to extension developers to minimize their own use of the stack by using the
 * provided CgiMemGet and CgiMemFree routines, which provide complete memory
 * management in order to reduce the amount of memory fragmentation that
 * typically occurs on NetWare 4.1x Servers when memory is being constantly
 * allocated and freed.
 *
 * Parameters:    None.
 *
 * Returns:       Nothing.
 *
 ******************************************************************************/

void main (int argc, char * argv [])
{
   LONG     retCode;
// BYTE  *  tonto;

   
   ThreadSwitchWithDelay   ();                     // force the OS to fill in our PCB
   xx_NlmID                =  GetNLMID ();         // cache our NLM ID
   xx_NlmHandle            =  GetNLMHandle();      // cache our NLM Handle
   xx_ThreadID             =  GetThreadID ();      // used with suspend/resume thread
   xx_ThreadGroupID        =  GetThreadGroupID (); // used with set/get thread group ID

   // Before calling any of the auxilary initialization routines, we register
   // the necessary shutdown routines.  Once registered, if an error occurs
   // during initialization and this module does a simple return to the OS,
   // the shutdown routine will be called to release any allocated resources.
   
   signal                  (SIGTERM, SigTermHandler);
   atexit                  ((void (*) (void)) SigTermHandler);

   if ((retCode = SysMsgTblInit (argc, argv)) == SUCCESS)
   {
      xxDbgCmdProcInit     ();   // Installs this modules command line processor
      xxDbgHookSysMsg      ();   // Initializes a debug session with the CsSysMsg NLM

      // CGI extension processor specific initialization.
      
//    CgiMemGet   (&tonto, 10);  // rls -- used to test the unreleased resources feature in CsSysMsg.Nlm

      if ((retCode = CgiExtReg (RequestHandler, UnloadHandler, 0, &xx_CgiXID)) == SUCCESS)
      {
         // Once this module has been successfully initialized, we suspend 
         // the main thread and wait indefinitely for the shutdown event.  
         // This module may be unloaded from either the console, or when
         // the UnloadHander() is called.  Note that we MUST use this thread
         // for the UnloadHandler() to work in the NetWare environment.
         // Otherwise if we executed an "exit thread" command from the
         // calling Web Server's thread, we would kill the OS worker thread
         // and never get control back of the command line process.
   
         SuspendThread  (xx_ThreadID);
      }
   }
   
   if (retCode != SUCCESS)
   {
      // If an error occurred during initialization, we want a visual display
      // at the console.
      
      CSSM_DisplaySysMsg (CSSMC_LogOutput,      // option for logging
                          retCode,              // failure API error code
                          xx_NlmName);          // Insert our name in "compID2" field
   }
   ExitThread  (EXIT_NLM, 0);
}


/*******************************************************************************
 *
 * Name:          RequestHandler()
 *
 * Following CGI Extension Proccessor Registration, this routine will be called
 * on a NetWare HTTP Server thread with a minimum stack size of 16KB and will
 * execute in the thread context of the Server.
 *
 * Parameters:    cgiInfo     In/Out   
 *
 *                This structure is used to pass all necessary information 
 *                back and forth between the Server and the request processing
 *                thread.  With the exception of the Mutex Lock and unlock
 *                routines, all of the CGI API's require this structure.  Also,
 *                work pointer set and get APIs have been provided to allow
 *                the extension processor to set a pointer to its own extension
 *                specific information.
 *                
 * Returns:       SUCCESS, or a registered API error value.
 *
 ******************************************************************************/

LONG RequestHandler (struct Cgi_t * cgiInfo)
{
   BYTE  ** envList;

   
   xxDbgDispLoc ("RequestHandler");

   // On entry to this routine, we note the number of HTTP threads executing
   // in this NLM and then determine whether this NLM has been requested to
   // unload itself.  Note that it isn't really necessary to check this flag
   // here since the Server will not pass any more requests to this NLM once
   // it has determined that the NLM is unloading.  However, you may want to
   // check this flag in other areas of your code in order to determine
   // whether this NLM could or should do a premature exit.
   
   CgiMutexLock         (xx_CgiXID);
   xx_InRequestHandler  += 1;
   CgiMutexUnlock       (xx_CgiXID);

   if ((xx_Status & xxC_UnloadRequested) == 0)
   {
      if ((CgiStrmStdHdrSend (cgiInfo,       // cgi info block
                              "text/html",   // mine type
                              -1)            // -1 == don't send a content length
          ) == SUCCESS)
      {
         CgiStrmStrPut        (cgiInfo, NULL, "\r\n");
         CgiStrmPrintf        (cgiInfo, NULL, "%s", "<HTML><TITLE>This is a No-Parse Header Cgi Extension NLM</TITLE>");
         CgiStrmPrintf        (cgiInfo, NULL, "<BODY>");
         CgiStrmPrintf        (cgiInfo, NULL, "<H1>This is a No-Parse Header Cgi Extension NLM</H1>");
         CgiStrmPrintf        (cgiInfo, NULL, "</BODY>");
         CgiStrmPrintf        (cgiInfo, NULL, "</HTML>");
      }

      CgiEnvListCreate  (cgiInfo, &envList);
      CgiEnvListDestroy (cgiInfo, &envList);
   }
   CgiMutexLock         (xx_CgiXID);
   xx_InRequestHandler  -= 1;
   CgiMutexUnlock       (xx_CgiXID);

   return               (SUCCESS);
}


/*******************************************************************************
 *
 * Name:          SigTermHandler()
 *
 * This routine is called when either a NetWare "signal" is raised, the module
 * is being unloaded as a result of the console operator entering an unload
 * command, or when the module attempts to unload itself.  In either case,
 * this routine is used to cleanup all allocated resources and it must mark
 * them as "freed" since this routines may be called multiple times during
 * the shutdown process.
 *
 *
 * Parameters:    Standard signal handler parameters when called as a result
 *                of a raised signal.  When it's called as a result of being 
 *                registered as an "atexit" or "atUnload"
 *
 * Returns:       Nothing.
 *
 ******************************************************************************/

void SigTermHandler (int sig)
{
   static calledOnce    =  FALSE;
   time_t         startTime;
   
   (void)         sig;                 // unreferenced symbol
   xxDbgDispLoc   ("SigTermHandler");

   // Note that the SigTerm handler may be called more than once during the
   // shutdown process, so we use a flag to keep it simple.
   
   if (calledOnce == FALSE)
   {
      calledOnce  =  TRUE;

      CgiMutexLock      (xx_CgiXID);
      xx_Status         |= xxC_UnloadRequested;
      CgiMutexUnlock    (xx_CgiXID);

      startTime         =  time (NULL);

      while (xx_InRequestHandler != 0)
      {
         if (time (NULL) == startTime)
         {
            startTime            += 10;
            CSSM_DisplaySysMsg   (CSSMC_LogOutput, CgiE_WebInRequestHandler, xx_NlmName);
         }
         ThreadSwitchWithDelay   ();
      }
      
      if (xx_CgiXID)
      {
         // Deregister this CGI Extension NLM with the NetWare HTTP Server.
         // Once this module has deregistered, it will not receive any more
         // CGI requests.
      
         CgiExtDeReg    (xx_CgiXID);
         xx_CgiXID      =  NULL;
      }
      xxDbgCmdProcDeInit   ();
      CeFree               (&xx_MsgTable);      // free and clear the msg table ptr
   }
}


/*******************************************************************************
 *
 * Name:          SysMsgTblInit()
 *  
 *  This routine is called from main to initialize the message table for the 
 *  messages defined for this CGI extension module.  
 *  
 *  The table is initialized by setting the msgText pointer to the translated 
 *  text string associated with the .MDB index value that was compiled into the
 *  source code when the message text was extracted by the NLS tools.
 *  
 ******************************************************************************/

LONG  SysMsgTblInit (int argc, char * argv [])
{
   int      ii;
   LONG     retCode;
   char     pDrive      [48];
   char     pDirectory  [_MAX_PATH];
   char     loadName    [48];
   char     loadExt     [8];

   
   (void)      argc;                      // unreferenced variable
   _splitpath  (argv [0], pDrive, pDirectory, loadName, loadExt);
   strcpy      (xx_NlmPath, pDrive);
   strcat      (xx_NlmPath, pDirectory);

   retCode = CSSM_ReturnAuxMessageInformation (
                   xx_NlmID,        // Caller's NLN ID --> get name from load def struct
                   argv [0],        // NLM load path --> where to start looking
                   NULL,            // NULL --> default file name (.MSG & .HLP files)
                   &xx_MsgTable,    // return message structure
                   &xx_MsgCount,    // return message count
                   &xx_LanguageID,  // -1 --> default OS language
                   NULL,            // NULL --> don't return the version string
                   NULL);           // &xx_HelpTable);  // return NwsNut help structure

   if (retCode == SUCCESS || retCode == CSSME_ReqLangNotFound)
   {
      retCode = SUCCESS;      // return SUCCESS even if we didn't get the language we wanted
      
      for (ii = 0; xx_SysMsgTbl [ii]. msgNum != 0; ii++)
      {
         xx_SysMsgTbl [ii]. msgText = xx_MsgTable [xx_SysMsgTbl [ii]. msgIndex];
      }
   }

   if (retCode != SUCCESS)
   {
      CSSM_DisplayUserMsg (xx_NlmID,            //
                           CSSMC_WaitOnOutput,  // 
                           xxC_SysMsgBase,      // Display SysMsg base number (not registered anywhere)
                           NULL,                // severity
                           xx_NlmName,          // Nlm name
                           NULL,                // Null --> dynamically read
                           "CgiExt",            // compId # 1
                           NULL,                // compId # 2
                           "Unable to initialize the %s CGI Extension Module.  Refer to the manual for message number %X.", xx_NlmName, retCode);
   }  
   else
   {        
      retCode = CSSM_Register (xx_SysMsgTbl,    // system msg array to register 
                               xx_NlmPath,      // CGI NLM product directory
                               "",              // "" --> get message file name 
                               "",              // "" --> get NLM name 
                               "",              // "" --> get NLM version level
                               xx_NlmName,      // use the module name as the sub-process name 
                               xx_LanguageID,   // registered language ID
                               xxC_SysMsgBase,  // global msg base number
                               xx_NlmID);       // NLM ID
   }
   return (retCode);
}


/*******************************************************************************
 *
 * Name:          UnloadHandler()
 *
 * This routine will be called from the Http Web Server to request an unload
 * of this CGI Extension Processor.  It may be called under one of two
 * conditions; either the NetWare Web Server is being unloaded, in which case
 * the unload immediate flag will be set, or the web server has determined
 * that this NLM's services have not been requested for the configured timeout
 * period and is requesting this NLM to release its resources by unloading.
 *
 * Parameters:    
 *
 *    unloadFlags    In:   This bit-string is used to indicate the source of
 *                         the unload request.  The current release of the
 *                         NetWare Web Server defines the following bits:
 *
 *                         CgiC_UnloadImmediate
 *
 *                         This flag is used to indicate that the Web Server
 *                         has been requested to unload by the console
 *                         operator.  However, it remains the perogative of
 *                         the Extension NLM to honor the immediate unload
 *                         request, i.e., if database operations were not
 *                         complete, then it would be inappropriate for this
 *                         NLM to exit with an indeteminate state.
 *
 *                         CgiC_ServerUnload
 *
 *                         This bit will be set when the NetWare Web Server
 *                         is being unloaded.  In this case, the extension
 *                         still has the option of not unloading.
 *                         
 * Returns:       SUCCESS or an appropriate API error value.
 *    
 ******************************************************************************/

LONG UnloadHandler (LONG unloadFlags)
{
   int      curThreadGroupID;
   time_t   startTime;
   
   // Set the unload requested flag and check whether the request handler is
   // currently executing.  If not, wake up the main thread which will then
   // terminate the NLM.
   //
   // Note:  Since this process is running on the caller's thread we MUST NOT
   // kill (or EXIT_NLM) here, i.e., if the web server is being unloaded from
   // the console command line, we're running on the OS console thread and if
   // we kill that thread, we would hang the NetWare Server.

   curThreadGroupID  =  SetThreadGroupID (xx_ThreadGroupID);
   CgiMutexLock      (xx_CgiXID);
   xx_Status         |= xxC_UnloadRequested;
   CgiMutexUnlock    (xx_CgiXID);

   if ((unloadFlags & CgiC_UnloadImmediate) == 0)
   {
      startTime   =  time (NULL);
      
      while (xx_InRequestHandler != 0 && (time (NULL) - startTime) < 10)
      {
         ThreadSwitchWithDelay ();
      }
   }

   if (xx_InRequestHandler == 0)
   {
      // Whether or not a immediate unload was requested, if there are Web
      // Server threads processing requests in this module, we must not
      // resume the main thread and unload this NLM.  Although the NLM can
      // still be unloaded from the console at any time, the operator runs
      // the risk of crashing the server on an invalid access.
         
      ResumeThread   (xx_ThreadID);
   }
   SetThreadGroupID  (curThreadGroupID);
   return            (SUCCESS);
}

/******************************************************************************/
