//
// $Header$
//

// 32 bits Linux ext2 file system driver for OS/2 WARP - Allows OS/2 to
// access your Linux ext2fs partitions as normal drive letters.
// Copyright (C) 1995, 1996, 1997  Matthieu WILLM (willm@ibm.net)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#define INCL_DOS
#define INCL_DOSDEVIOCTL
#define INCL_DOSERRORS
#define INCL_NOPMAPI
#include <os2.h>

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <time.h>
#include <errno.h>
#include <linux/stat.h>
#include <os2/types.h>
#include <linux/ext2_fs.h>

#include "ext2fs/ext2fs.h"

#include "et/com_err.h"
#include "ext2fs/ext2_err.h"

#include "ext2fs/os2_io.h"

/*
 * For checking structure magic numbers...
 */

#define EXT2_CHECK_MAGIC(struct, code) \
          if ((struct)->magic != (code)) return (code)


static errcode_t os2_open(const char *name, int flags, io_channel *channel);
static errcode_t os2_close(io_channel channel);
static errcode_t os2_set_blksize(io_channel channel, int blksize);
static errcode_t os2_read_blk(io_channel channel, unsigned long block,
                               int count, void *data);
static errcode_t os2_write_blk(io_channel channel, unsigned long block,
                                int count, const void *data);
static errcode_t os2_flush(io_channel channel);

static struct struct_io_manager struct_os2_manager = {
        EXT2_ET_MAGIC_IO_MANAGER,
        "IBM OS/2 WARP I/O Manager",
        os2_open,
        os2_close,
        os2_set_blksize,
        os2_read_blk,
        os2_write_blk,
        os2_flush
};

io_manager os2_io_manager = &struct_os2_manager;


#define FLAGS_DASD   1          /* 1 : OPEN_FLAGS_DASD open         0 : normal open */
#define FLAGS_LOCKED 2          /* 1 : IOCtl DSK_LOCKDRIVE called   0 : not called  */
#define FLAGS_FORMAT 4          /* 1 : IOCtl DSK_BEGINFORMAT called 0 : not called  */

#define SECTOR_READ   1		/* IOCTL_DISK DSK_READTRACK 	*/
#define SECTOR_WRITE  2		/* IOCTL_DISK DSK_WRITETRACK	*/
#define SECTOR_VERIFY 3		/* IOCTL_DISK DSK_VERIFYTRACK	*/

#pragma pack(1)
struct parm {
    unsigned char command;
    unsigned char drive;
};

struct data {
    BIOSPARAMETERBLOCK bpb;
    unsigned short     nr_cyl;
    unsigned char      devtype;
    unsigned short     devattr;
};
#pragma pack(4)

static char *fs_name = "ext2";
int __os2_open(const char *name, int flags, struct os2_private_data *data) {
    APIRET rc;
    HFILE  f;
    int namelen;
    int open_flags;
    int rw;
    ULONG action;
    ULONG parmio;
    ULONG dataio;
    ULONG pio;
    ULONG dio;
    struct parm p;
    struct data d;

    namelen = strlen(name);
/*
    printf("\tOS2 : os2_open %s %d\n", name, namelen);
*/
    data->os2_flags = 0;
    /*
     * Case where "name" is a drive letter (block device name) : OS/2 has special requirements.
     */
    if ((namelen == 2) && (name[1] == ':') && (((name[0] >= 'A') && (name[0] <= 'Z')) || ((name[0] >= 'a') && (name[0] <= 'z')))) {
        rw = flags & IO_FLAG_RW ? OPEN_ACCESS_READWRITE | OPEN_SHARE_DENYWRITE : OPEN_ACCESS_READONLY | OPEN_SHARE_DENYNONE;
/*
        printf("\tOS2 : opening %s in DASD mode\n", name);
*/
        if ((rc = DosOpen(
                      name,
                      &f,
                      &action,
                      0,
                      0,
                      OPEN_ACTION_OPEN_IF_EXISTS,
                      OPEN_FLAGS_DASD | OPEN_FLAGS_NO_CACHE | OPEN_FLAGS_WRITE_THROUGH  | rw,
                      NULL
                     )) == NO_ERROR) {
            data->os2_flags = FLAGS_DASD;
/*
            printf("\tOS2 : DosOpen OK\n");
*/
            p.command = 0;
            p.drive   = 0;
            parmio = sizeof(p);
            dataio = sizeof(d);
            if ((rc = DosDevIOCtl(
                                  f,
                                  IOCTL_DISK,
                                  DSK_GETDEVICEPARAMS,
                                  &p, parmio, &parmio,
                                  &d, dataio, &dataio
                                 )) == NO_ERROR) {



        data->bytes_per_sector    = d.bpb.usBytesPerSector;
        data->cylinders  = d.bpb.cCylinders;
        data->heads      = d.bpb.cHeads;
        data->sectors_per_track = d.bpb.usSectorsPerTrack;
        data->hidden_sectors  = d.bpb.cHiddenSectors;
        if ((d.bpb.cSectors == 0) && (d.bpb.cLargeSectors)){
            data->total_sectors = d.bpb.cLargeSectors;
        } else if ((d.bpb.cSectors) && (d.bpb.cLargeSectors == 0)) {
            data->total_sectors = d.bpb.cSectors;
        } else {
            DosClose(f);
            f = -1;
            fprintf(stderr, "OS2 FATAL ERROR while retrieving device size\n");
            errno = -EACCES;
            return f;
        }
/*
        fprintf(stderr, "\tOS2 : total sectors     = %u\n", data->total_sectors);
        fprintf(stderr, "\t      bytes per sector  = %u\n", data->bytes_per_sector);
        fprintf(stderr, "\t      sectors per track = %u\n", data->sectors_per_track);
        fprintf(stderr, "\t      heads             = %u\n", data->heads);
        fprintf(stderr, "\t      hidden sectors    = %u\n", data->hidden_sectors);
        fprintf(stderr, "\t      cylinders         = %u\n", data->cylinders);
*/
            /*
             * In case the drive is mounted RW, we must lock the drive.
             */
            if (flags & IO_FLAG_RW) {
                parmio = 0;
                dataio = 0;
                pio    = 1;
                dio    = 1;
                if ((rc = DosDevIOCtl(
                                      f,
                                      IOCTL_DISK,
                                      DSK_LOCKDRIVE,
                                      &parmio, 1, &pio,
                                      &dataio, 1, &dio
                                     )) != NO_ERROR) {
                    os2_err(rc);
                    DosClose(f);
                    f = -1;
                    errno = EACCES;
                } else {
                    data->os2_flags |= FLAGS_LOCKED;

                    parmio = (ULONG)fs_name;
                    pio    = sizeof(char *);
                    dio    = 1;
                    dataio = 0;
                    if ((rc = DosDevIOCtl(
                                          f,
                                          IOCTL_DISK,
                                          DSK_BEGINFORMAT,
                                          &parmio, pio, &pio,
                                          &dataio, dio, &dio
                                         )) != NO_ERROR) {
                        os2_err(rc);
                        DosClose(f);
                        f = -1;
                        errno = EACCES;
                    } else {
                        data->os2_flags |= FLAGS_FORMAT;
                    }
                }
            }
            } else {
                /*
                 * DosDevIOCtl failed (DSK_GETDEVICEPARMS)
                 */
                os2_err(rc);
                DosClose(f);
                f = -1;
                errno = EACCES;
            }
        } else {
            /*
             * DosOpen failed
             */
            os2_err(rc);
            f = -1;
            errno = EACCES;
        }
        return f;
    }

    /*
     * Case where "name" is a regular file name or a character device name
     */
    return open(name, ((flags & IO_FLAG_RW) ? O_RDWR : O_RDONLY) | O_BINARY);

}

int __os2_close(int fd, struct os2_private_data *data) {
    APIRET rc;
    ULONG parmio;
    ULONG dataio;
    ULONG pio;
    ULONG dio;

    if ((data->os2_flags & FLAGS_DASD) && (data->os2_flags & FLAGS_FORMAT)) {
/*
        printf("\n\tOS2 : Redetermining media\n");
*/
        parmio = 0;
        dataio = 0;
        pio = 1;
        dio = 1;
        rc = DosDevIOCtl(
                         fd,
                         IOCTL_DISK,
                         DSK_REDETERMINEMEDIA,
                         &parmio, pio, &pio,
                         &dataio, dio, &dio
                        );
         if (rc != NO_ERROR) {
             os2_err(rc);
         }
    }

    if ((data->os2_flags & FLAGS_DASD) && (data->os2_flags & FLAGS_LOCKED)) {
/*
        printf("\tOS2 : Unlocking drive\n");
*/
        parmio = 0;
        dataio = 0;
        pio = 1;
        dio = 1;
        rc = DosDevIOCtl(
                         fd,
                         IOCTL_DISK,
                         DSK_UNLOCKDRIVE,
                         &parmio, pio, &pio,
                         &dataio, dio, &dio
                        );
         if (rc != NO_ERROR) {
             os2_err(rc);
         }
    }
/*
    printf("\tOS2 : Closing drive\n");
*/
    return close(fd);
}


#pragma pack(1)


//
// Extended Boot record structure and extended BPB (from the IBM IFS document)
//
struct Extended_BPB {
    unsigned short BytePerSector;
    unsigned char SectorPerCluster;
    unsigned short ReservedSectors;
    unsigned char NumberOfFats;
    unsigned short RootEntries;
    unsigned short TotalSectors;
    unsigned char MediaDescriptor;
    unsigned short SectorsPerFat;
    unsigned short SectorsPerTrack;
    unsigned short Heads;
    unsigned long HiddenSectors;
    unsigned long Ext_TotalSectors;
};

struct Extended_Boot {
    unsigned char Boot_jmp[3];
    unsigned char Boot_OEM[8];
    struct Extended_BPB Boot_BPB;
    unsigned char Boot_DriveNumber;
    unsigned char Boot_CurrentHead;
    unsigned char Boot_Sig;
    unsigned long Boot_Serial;
    unsigned char Boot_Vol_Label[11];
    unsigned char Boot_System_ID[8];
};

struct boot_sector {
    struct Extended_Boot boot;
    char                 pad1[512 - sizeof(struct Extended_Boot) - sizeof(unsigned short)];
    unsigned short       sig;       // 0xAA55
    unsigned char        media;     // media type
    char                 pad2[511];
};

#pragma pack(4)

int make_dos_compatible_boot_sector(ext2_filsys fs) {
    APIRET                    rc;
    struct boot_sector        boot;
    int                       retval;
    struct os2_private_data *data;

    data = (struct os2_private_data *)(fs->io->private_data);


    if (data->os2_flags & FLAGS_DASD) {
        /*
         * This is a drive
         */
        struct parm p;
        struct data d;
        unsigned long plen = sizeof(p);
        unsigned long dlen = sizeof(d);

        p.command = 0;
        p.drive   = 0;
        /*
         * Retrieves the recommended BPB for the drive.
         */
        if ((rc = DosDevIOCtl(
                              (HFILE)(data->dev),
                              IOCTL_DISK,
                              DSK_GETDEVICEPARAMS,
                              &p, plen, &plen,
                              &d, dlen, &dlen
                             )) != NO_ERROR) {
            fprintf(stderr, "DosDevIOCtl failed with rc = %d in file %s at line %d\n", rc, __FILE__, __LINE__);
            os2_err(rc);
            return -1;
        }

        memset(&boot, 0, sizeof(boot));

        /*
         * We put the recommended BPB in sector 0. ext2 doesn't care, but this
         * makes OS/2 happy. (there's a bug in the OS/2 kernel that prevents FLOPPIES
         * with a non DOS boot sector to be mounted by an IFS).
         */

        boot.boot.Boot_BPB.BytePerSector        = d.bpb.usBytesPerSector;
        boot.boot.Boot_BPB.SectorPerCluster     = d.bpb.bSectorsPerCluster;
        boot.boot.Boot_BPB.ReservedSectors      = d.bpb.usReservedSectors;
        boot.boot.Boot_BPB.NumberOfFats         = d.bpb.cFATs;
        boot.boot.Boot_BPB.RootEntries          = d.bpb.cRootEntries;
        boot.boot.Boot_BPB.TotalSectors         = d.bpb.cSectors;
        boot.boot.Boot_BPB.MediaDescriptor      = d.bpb.bMedia;
        boot.boot.Boot_BPB.SectorsPerFat        = d.bpb.usSectorsPerFAT;
        boot.boot.Boot_BPB.SectorsPerTrack      = d.bpb.usSectorsPerTrack;
        boot.boot.Boot_BPB.Heads                = d.bpb.cHeads;
        boot.boot.Boot_BPB.HiddenSectors        = d.bpb.cHiddenSectors;
        boot.boot.Boot_BPB.Ext_TotalSectors     = d.bpb.cLargeSectors;

        strncpy(boot.boot.Boot_System_ID, "ext2    ", 8);
        strncpy(boot.boot.Boot_OEM, "OS2 WARP", 8);
        strncpy(boot.boot.Boot_Vol_Label, "NO LABEL   ", 11);
        boot.boot.Boot_Sig    = 41;
        boot.boot.Boot_Serial = 0xABCDDCBA;
        boot.sig              = 0xAA55;
        boot.media            = d.bpb.bMedia;


        retval = io_channel_write_blk(fs->io, 0, -1024, &boot);
        if (retval)
            printf("Warning: could not write boot block 0: %s\n",
                   error_message(retval));

    } else {
        /*
         * This is NOT a drive -> don't bother.
         */
        retval = 0;
    }
    return retval;

}


static errcode_t os2_open(const char *name, int flags, io_channel *channel)
{
        io_channel      io = NULL;
        struct os2_private_data *data = NULL;
        errcode_t       retval;

        if (name == 0)
                return EXT2_ET_BAD_DEVICE_NAME;
        io = (io_channel) malloc(sizeof(struct struct_io_channel));
        if (!io)
                return ENOMEM;
        memset(io, 0, sizeof(struct struct_io_channel));
        io->magic = EXT2_ET_MAGIC_IO_CHANNEL;
        data = (struct os2_private_data *)
                malloc(sizeof(struct os2_private_data));
        if (!data) {
                retval = ENOMEM;
                goto cleanup;
        }
        io->manager = os2_io_manager;
        io->name = malloc(strlen(name)+1);
        if (!io->name) {
                retval = ENOMEM;
                goto cleanup;
        }
        strcpy(io->name, name);
        io->private_data = data;
        io->block_size = 1024;
        io->read_error = 0;
        io->write_error = 0;
	io->refcount = 1;

        memset(data, 0, sizeof(struct os2_private_data));
        data->magic = EXT2_ET_MAGIC_UNIX_IO_CHANNEL;
        data->buf = malloc(io->block_size);
        data->buf_block_nr = -1;
        if (!data->buf) {
                retval = ENOMEM;
                goto cleanup;
        }
        data->dev = __os2_open(name, flags, data);
        if (data->dev < 0) {
                retval = errno;
                goto cleanup;
        }
        *channel = io;
        return 0;

cleanup:
        if (io)
                free(io);
        if (data) {
                if (data->buf)
                        free(data->buf);
                free(data);
        }
        return retval;
}

static errcode_t os2_close(io_channel channel)
{
        struct os2_private_data *data;
        errcode_t       retval = 0;

        EXT2_CHECK_MAGIC(channel, EXT2_ET_MAGIC_IO_CHANNEL);
        data = (struct os2_private_data *) channel->private_data;
        EXT2_CHECK_MAGIC(data, EXT2_ET_MAGIC_UNIX_IO_CHANNEL);


	if (--channel->refcount > 0)
		return 0;

        if (__os2_close(data->dev, data) < 0)
                retval = errno;
        if (data->buf)
                free(data->buf);
        if (channel->private_data)
                free(channel->private_data);
        if (channel->name)
                free(channel->name);
        free(channel);
        return retval;
}

static errcode_t os2_set_blksize(io_channel channel, int blksize)
{
        struct os2_private_data *data;

        EXT2_CHECK_MAGIC(channel, EXT2_ET_MAGIC_IO_CHANNEL);
        data = (struct os2_private_data *) channel->private_data;
        EXT2_CHECK_MAGIC(data, EXT2_ET_MAGIC_UNIX_IO_CHANNEL);

        if (channel->block_size != blksize) {
                channel->block_size = blksize;
                free(data->buf);
                data->buf = malloc(blksize);
                if (!data->buf)
                        return ENOMEM;
                data->buf_block_nr = -1;
        }
        return 0;
}

/*
 * Reads or writes n sectors on a logical drive. Uses CSH addressing to handle
 * > 2 Gb devices
 */
int os2_sector_io(struct os2_private_data *data, const char *buf, unsigned long sector, int operation, unsigned long nr) {
    APIRET        rc;
    ULONG         pio;
    ULONG         dio;
    int fnt;
    TRACKLAYOUT  *tracklayout;
    size_t actual = 0;
    unsigned short c, s, h;
    unsigned long last_sector;
    unsigned short c_end, s_end, h_end;
    unsigned short begin_h, end_h;
    unsigned short begin, end;
    unsigned short m, n, o, i;

    switch(operation) {
	case SECTOR_READ :
	    fnt = DSK_READTRACK;
	    break;
	case SECTOR_WRITE :
	    fnt = DSK_WRITETRACK;
	    break;
	default :
	    errno = -EINVAL;
	    return 0;
    }

    tracklayout = malloc(sizeof(TRACKLAYOUT) + 4 * (data->sectors_per_track - 1));
    if (tracklayout) {

    sector += data->hidden_sectors;
    last_sector = sector + nr - 1;

    c = (unsigned short)(sector / ((unsigned long)(data->sectors_per_track) * (unsigned long)(data->heads)));
    h = (unsigned short)(((sector % ((unsigned long)(data->sectors_per_track) * (unsigned long)(data->heads)))) / (unsigned long)(data->sectors_per_track));
    s = (unsigned short)(((sector % ((unsigned long)(data->sectors_per_track) * (unsigned long)(data->heads)))) % (unsigned long)(data->sectors_per_track));

    c_end = (unsigned short)(last_sector / ((unsigned long)(data->sectors_per_track) * (unsigned long)(data->heads)));
    h_end = (unsigned short)(((last_sector % ((unsigned long)(data->sectors_per_track) * (unsigned long)(data->heads)))) / (unsigned long)(data->sectors_per_track));
    s_end = (unsigned short)(((last_sector % ((unsigned long)(data->sectors_per_track) * (unsigned long)(data->heads)))) % (unsigned long)(data->sectors_per_track));
/*
    printf("lba = %lu nr = %lu, (c, h, s) = (%u, %u, %u), (c_end, h_end, s_end) = (%u, %u, %u)\n", sector, nr, c, h, s, c_end, h_end, s_end);
*/


    /*
     * loop on cylinders
     */
    for (m = c ; m <= c_end ; m++) {
	if (m == c)
	    begin_h = h;
	else
	    begin_h = 0;
        if (m == c_end)
	    end_h = h_end;
	else
	    end_h = data->heads - 1;

        /*
         * loop on heads
         */
        for (n = begin_h ; n <= end_h ; n++) {
	    if ((n == h) && (m == c))
		begin = s;
	    else
	        begin = 0;
            if ((n == h_end) && (m == c_end))
		end = s_end;
	    else
		end = data->sectors_per_track - 1;
/*
	    printf("\t\t\tsector I/O : (c,h)=(%d,%d) begin = %d end = %d\n", m, n, begin, end);
*/
	    tracklayout->bCommand      = 0;
	    tracklayout->usHead        = n;
	    tracklayout->usCylinder    = m;
	    tracklayout->usFirstSector = 0;
	    tracklayout->cSectors      = end - begin + 1;
	    for (i = 0 ; i < end - begin + 1 ; i++) {
    		tracklayout->TrackTable[i].usSectorNumber = begin + i + 1;
    		tracklayout->TrackTable[i].usSectorSize   = data->bytes_per_sector;
            }
	    pio = sizeof(TRACKLAYOUT) + 4 * (end - begin);
	    dio = data->bytes_per_sector * (end - begin + 1);

	    rc = DosDevIOCtl(
            	            data->dev,
                     	    IOCTL_DISK,
                    	    fnt,
                     	    tracklayout, pio, &pio,
                     	    buf + actual, dio, &dio
                           );
            if (rc == NO_ERROR) {
                actual += data->bytes_per_sector * (end - begin + 1);
            } else {
                fprintf(stderr, "Error while doing sector I/O %lu (c,h,s)=(%u,%u,%u)\n", sector, c, h, s);
                fprintf(stderr, "lba = %lu nr = %lu, (c, h, s) = (%u, %u, %u), (c_end, h_end, s_end) = (%u, %u, %u)\n", sector, nr, c, h, s, c_end, h_end, s_end);
                os2_err(rc);
	        errno  = -EIO;
	        break;
            }    

        }
    }
        free(tracklayout);   
    } else {
        errno = -ENOMEM;
    }
    return actual;
}

/*
 * Reads blocks from disk. Uses os2_sector_io instead of read() to handle drives 
 * larger than 2 Gb (CHS addressing)
 */
size_t __os2_read(io_channel channel, void *buf, size_t size, ext2_loff_t location) {
        struct os2_private_data *data;
        size_t actual = 0;
        int            nr_sectors;
	unsigned long  sector;
        int            partial;
        int i;

        EXT2_CHECK_MAGIC(channel, EXT2_ET_MAGIC_IO_CHANNEL);
        data = (struct os2_private_data *) channel->private_data;
        EXT2_CHECK_MAGIC(data, EXT2_ET_MAGIC_UNIX_IO_CHANNEL);

        if (! ((location) & (data->bytes_per_sector - 1))) {
            nr_sectors = size / data->bytes_per_sector;
            partial    = size % data->bytes_per_sector;

	    /*
             * Reads the full sectors if any
             */
	    if (nr_sectors) {
                actual += os2_sector_io(data, buf, location / data->bytes_per_sector, SECTOR_READ, nr_sectors);
            }

	    /*
             * Reads the last partial sector if any
             */
            if (partial) {
		char *partial_buf = malloc(data->bytes_per_sector);
                if (partial_buf) {
                    int bytes_read = os2_sector_io(data, partial_buf, location / data->bytes_per_sector + nr_sectors, SECTOR_READ, 1);
                    if (bytes_read == data->bytes_per_sector) {
			memcpy(buf + nr_sectors * data->bytes_per_sector, partial_buf, partial);
			actual += partial;
                    } else {
			errno = -EIO;
                    }
		    free(partial_buf);
                } else {
                    errno = -ENOMEM;
                }
            }
        } else {
            fprintf(stderr, "__os2_read : location %d not a multiple of sector size %d", size, data->bytes_per_sector);
            errno = -EIO;
        }        
        return actual;
}

/*
 * Reads buffer or blocks from disk. Uses os2_sector_io instead of read() 
 * to handle drives larger than 2 Gb (CHS addressing).
 */
static errcode_t os2_read_blk(io_channel channel, unsigned long block,
                               int count, void *buf)
{
        struct os2_private_data *data;
        errcode_t       retval;
        size_t          size;
        ext2_loff_t     location;
        int             actual = 0;


        EXT2_CHECK_MAGIC(channel, EXT2_ET_MAGIC_IO_CHANNEL);
        data = (struct os2_private_data *) channel->private_data;
        EXT2_CHECK_MAGIC(data, EXT2_ET_MAGIC_UNIX_IO_CHANNEL);

        /*
         * If it's in the cache, use it!
         */
        if ((count == 1) && (block == data->buf_block_nr)) {
                memcpy(buf, data->buf, channel->block_size);
                return 0;
        }
#if 0
        printf("read_block %lu (%d)\n", block, count);
#endif
        size = (count < 0) ? -count : count * channel->block_size;
        location = (ext2_loff_t) block * channel->block_size;

        if (data->os2_flags & FLAGS_DASD) {
            actual = __os2_read(channel, buf, size, location);
        } else {
            if (lseek(data->dev, location, SEEK_SET) != location) {
                    retval = errno;
                    goto error_out;
            }
            actual = read(data->dev, buf, size);
        }
        if (actual != size) {
                if (actual < 0)
                        actual = 0;
                retval = EXT2_ET_SHORT_READ;
                goto error_out;
        }
        if (count == 1) {
                data->buf_block_nr = block;
                memcpy(data->buf, buf, size);   /* Update the cache */
        }
        return 0;

error_out:
        memset((char *) buf+actual, 0, size-actual);
        if (channel->read_error)
                retval = (channel->read_error)(channel, block, count, buf,
                                               size, actual, retval);
        return retval;
}


size_t __os2_write(io_channel channel, const void *buf, size_t size, ext2_loff_t location) {
        struct os2_private_data *data;
        size_t actual = 0;
        int            nr_sectors;
	unsigned long  sector;
        int            partial;
        int i;

        EXT2_CHECK_MAGIC(channel, EXT2_ET_MAGIC_IO_CHANNEL);
        data = (struct os2_private_data *) channel->private_data;
        EXT2_CHECK_MAGIC(data, EXT2_ET_MAGIC_UNIX_IO_CHANNEL);

        if (! ((location) & (data->bytes_per_sector - 1))) {
            nr_sectors = size / data->bytes_per_sector;
            partial    = size % data->bytes_per_sector;

	    /*
             * Writes the full sectors if any
             */
	    if (nr_sectors) {
                actual += os2_sector_io(data, buf, location / data->bytes_per_sector, SECTOR_WRITE, nr_sectors);
            }

	    if (actual != size - partial)
		fprintf(stderr, __FUNCTION__ " actual = %d size = %d\n", actual, size);
	    /*
             * Writes the last partial sector if any
             */
            if (partial) {
		char *partial_buf = malloc(data->bytes_per_sector);
                if (partial_buf) {
		    int bytes_written;

		    memcpy(partial_buf, buf + nr_sectors * data->bytes_per_sector, partial);
                    bytes_written = os2_sector_io(data, partial_buf, location / data->bytes_per_sector + nr_sectors, SECTOR_WRITE, 1);
                    if (bytes_written == data->bytes_per_sector) {
			actual += partial;
                    } else {
			fprintf(stderr, __FUNCTION__ " Partial write error\n");
			errno = -EIO;
                    }
		    free(partial_buf);
                } else {
                    errno = -ENOMEM;
                }
            }
        } else {
            fprintf(stderr, "__os2_write : location %d not a multiple of sector size %d", size, data->bytes_per_sector);
            errno = -EIO;
        }        
        return actual;
}

static errcode_t os2_write_blk(io_channel channel, unsigned long block,
                                int count, const void *buf)
{
        struct os2_private_data *data;
        size_t          size;
        ext2_loff_t     location;
        int             actual = 0;
        errcode_t       retval;

        EXT2_CHECK_MAGIC(channel, EXT2_ET_MAGIC_IO_CHANNEL);
        data = (struct os2_private_data *) channel->private_data;
        EXT2_CHECK_MAGIC(data, EXT2_ET_MAGIC_UNIX_IO_CHANNEL);

        if (count == 1)
                size = channel->block_size;
        else {
                data->buf_block_nr = -1;        /* Invalidate the cache */
                if (count < 0)
                        size = -count;
                else
                        size = count * channel->block_size;
        }

        location = (ext2_loff_t) block * channel->block_size;

        if (data->os2_flags & FLAGS_DASD) {
            actual = __os2_write(channel, buf, size, location);
        } else {
            if (lseek(data->dev, location, SEEK_SET) != location) {
                retval = errno;
                goto error_out;
            }
            actual = write(data->dev, buf, size);
        }
        if (actual != size) {
                retval = EXT2_ET_SHORT_WRITE;
                goto error_out;
        }

        if ((count == 1) && (block == data->buf_block_nr))
                memcpy(data->buf, buf, size); /* Update the cache */

        return 0;

error_out:
        if (channel->write_error)
                retval = (channel->write_error)(channel, block, count, buf,
                                                size, actual, retval);
        return retval;
}

/*
 * Flush data buffers to disk.  Since we are currently using a
 * write-through cache, this is a no-op.
 */
static errcode_t os2_flush(io_channel channel)
{
        struct os2_private_data *data;

        EXT2_CHECK_MAGIC(channel, EXT2_ET_MAGIC_IO_CHANNEL);
        data = (struct os2_private_data *) channel->private_data;
        EXT2_CHECK_MAGIC(data, EXT2_ET_MAGIC_UNIX_IO_CHANNEL);

        return 0;
}

