/** ColEm: portable Coleco emulator **************************/
/**                                                         **/
/**                           Unix.c                        **/
/**                                                         **/
/** This file contains Unix/X-dependent subroutines and     **/
/** drivers. It includes common drivers from Common.h.      **/
/**                                                         **/
/** Copyright (C) Marat Fayzullin 1994-1998                 **/
/**     You are not allowed to distribute this software     **/
/**     commercially. Please, notify me, if you make any    **/
/**     changes to this file.                               **/
/*************************************************************/
#ifdef UNIX

/** Private #includes ****************************************/
#include "Coleco.h"
#include "LibUnix.h"

#ifdef SOUND
#include "SndUnix.h"
#endif

/** Standard Unix/X #includes ********************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>

/** Public parameters ****************************************/
char *Title="ColEm Unix/X 1.0"; /* Window/command line title */
int SaveCPU  = 1;               /* 1 = freeze when focus out */
int UseSHM   = 1;               /* 1 = use MITSHM            */
int UseSound = 0;               /* 1 = use sound             */

/** Various variables ****************************************/
#define WIDTH  272
#define HEIGHT 208

static byte *XBuf,XPal[16],XPal0;
static Image Img;

/** Various X-related variables ******************************/
static Display *Dsp;
static Window Wnd;
static Colormap DefaultCMap;
static GC DefaultGC;
static unsigned long White,Black;

/** Sound-related definitions ********************************/
#ifdef SOUND
static int SndSwitch = 0x0F;
static int SndVolume = 200;
#endif

/** This function is called on signals ***********************/
static void OnBreak(int Arg) { ExitNow=1;signal(Arg,OnBreak); }

/** InitMachine() ********************************************/
/** Allocate resources needed by Unix/X-dependent code.     **/
/*************************************************************/
int InitMachine(void)
{
  Screen *Scr;
  int J,I;

  /* Open display */
  if(Verbose) printf("Initializing Unix/X drivers:\n  Opening display...");
  if(!(Dsp=XOpenDisplay(NULL))) { if(Verbose) puts("FAILED");return(0); }

  /* Set internal variables */
  Scr=DefaultScreenOfDisplay(Dsp);
  White=WhitePixelOfScreen(Scr);
  Black=BlackPixelOfScreen(Scr);
  DefaultGC=DefaultGCOfScreen(Scr);
  DefaultCMap=DefaultColormapOfScreen(Scr);

  /* Reset the palette */
  for(XPal0=Black,J=0;J<16;J++) XPal[J]=Black;

  /* Keys do not autorepeat */
  XAutoRepeatOff(Dsp);

  /* Initialize LibUnix toolkit */
  if(Verbose) printf("OK\n  Initializing LibUnix...");
  if(!InitLibUnix(Dsp)) { if(Verbose) puts("FAILED");return(0); }

  /* Create a window */
  if(Verbose) printf("OK\n  Opening window...");
  if(!(Wnd=X11Window(Title,WIDTH,HEIGHT)))
  { if(Verbose) puts("FAILED");return(0); }

  /* Create an image */
  if(Verbose) printf("OK\n  Allocating screen buffer...");
  J=X11NewImage(&Img,WIDTH,HEIGHT,UseSHM? USE_SHM:0);
  if(Verbose) puts(J? "OK":"FAILED");
  if(!J) return(0);
  XBuf=Img.Data;

#ifdef SOUND
  /* Initialize sound */
  if(InitSound(UseSound,Verbose))
  {
    SetChannels(SndVolume,SndSwitch);
    SetSound(3,SND_NOISE);
  }
#endif SOUND

  /* Catch all signals */
  signal(SIGHUP,OnBreak);signal(SIGINT,OnBreak);
  signal(SIGQUIT,OnBreak);signal(SIGTERM,OnBreak);

  return(1);
}

/** TrashMachine() *******************************************/
/** Deallocate all resources taken by InitMachine().        **/
/*************************************************************/
void TrashMachine(void)
{
  unsigned long L;
  int J;

  if(Verbose) printf("Shutting down...\n");

  if(Dsp)
  {
    for(XPal[0]=XPal0,J=0;J<16;J++)
      if(XPal[J]!=Black)
      { L=XPal[J];XFreeColors(Dsp,DefaultCMap,&L,1,0); }

    X11FreeImage(&Img);
    XAutoRepeatOn(Dsp);
    XCloseDisplay(Dsp);
  }

#ifdef SOUND
  TrashSound();
#endif SOUND
}

/** SetColor() ***********************************************/
/** Allocate a given color.                                 **/
/*************************************************************/
void SetColor(byte N,byte R,byte G,byte B)
{
  XColor Color;
  unsigned long L;

  if(XPal[N]!=Black)
  { L=XPal[N];XFreeColors(Dsp,DefaultCMap,&L,1,0); }

  if(!R&&!G&&!B) XPal[N]=Black;
  else
  {
    Color.flags=DoRed|DoGreen|DoBlue;
    Color.red=(int)R<<8;
    Color.green=(int)G<<8;
    Color.blue=(int)B<<8;
    XPal[N]=XAllocColor(Dsp,DefaultCMap,&Color)? Color.pixel:Black;
  }

  XPal0=XPal[0];
}

/** PutImage() ***********************************************/
/** Put an image on the screen.                             **/
/*************************************************************/
void PutImage(void)
{ X11PutImage(Wnd,&Img,0,0,0,0,WIDTH,HEIGHT); }

/** Joysticks ************************************************/
/** Check for keyboard events, parse them, and modify       **/
/** joystick controller status                              **/
/*************************************************************/
void Joysticks(void)
{
  static word JS[2] = { 0xFFFF,0xFFFF };
  static byte N=0;
  XEvent E;
  word J;

  if(XCheckWindowEvent(Dsp,Wnd,KeyPressMask|KeyReleaseMask,&E))
  {
    J=XLookupKeysym((XKeyEvent *)&E,0);
    if(E.type==KeyPress)
      switch(J)
      {
        case XK_F2:  LogSnd=!LogSnd;break;
        case XK_Escape:
        case XK_F12: ExitNow=1;break;
#ifdef DEBUG
        case XK_F1:  CPU.Trace=!CPU.Trace;break;
#endif
#ifdef SOUND
        case XK_F6:  SetChannels(SndVolume,SndSwitch^=0x01);break;
        case XK_F7:  SetChannels(SndVolume,SndSwitch^=0x02);break;
        case XK_F8:  SetChannels(SndVolume,SndSwitch^=0x04);break;
        case XK_F9:  SetChannels(SndVolume,SndSwitch^=0x08);break;
        case XK_F10: SndSwitch=SndSwitch? 0x00:0x0F;
                     SetChannels(SndVolume,SndSwitch);break;
        case XK_Page_Up:
          if(SndVolume<250) SndVolume+=10;
          SetChannels(SndVolume,SndSwitch);
          break;
        case XK_Page_Down:
          if(SndVolume>0) SndVolume-=10;    
          SetChannels(SndVolume,SndSwitch);
          break;
#endif
        case XK_0: case XK_1: case XK_2: case XK_3: case XK_4:
        case XK_5: case XK_6: case XK_7: case XK_8: case XK_9:
          JS[N]=(JS[N]&0xFFF0)|(J-XK_0);break; 

        case XK_z: case XK_x: case XK_c: case XK_v:
        case XK_b: case XK_n: case XK_m: 
        case XK_Z: case XK_X: case XK_C: case XK_V:
        case XK_B: case XK_N: case XK_M:
        case XK_Control_L:
          JS[N]&=0xBFFF;break;

        case XK_a: case XK_s: case XK_d: case XK_f:
        case XK_g: case XK_h: case XK_j:
        case XK_A: case XK_S: case XK_D: case XK_F:
        case XK_G: case XK_H: case XK_J:
        case XK_space: case XK_Alt_L:
          JS[N]&=0xFFBF;break;

        case XK_Down:   JS[N]&=0xFBFF;break;
        case XK_Up:     JS[N]&=0xFEFF;break;
        case XK_Left:   JS[N]&=0xF7FF;break;
        case XK_Right:  JS[N]&=0xFDFF;break;
        case XK_minus:  JS[N]=(JS[N]&0xFFF0)|10;break; 
        case XK_equal:  JS[N]=(JS[N]&0xFFF0)|11;break; 

        case XK_q: case XK_e: case XK_t: case XK_u: case XK_o:
        case XK_Q: case XK_E: case XK_T: case XK_U: case XK_O:
        case XK_bracketleft:
          JS[N]=(JS[N]&0xFFF0)|12;break;

        case XK_w: case XK_r: case XK_y: case XK_i: case XK_p:
        case XK_W: case XK_R: case XK_Y: case XK_I: case XK_P:
        case XK_bracketright:
          JS[N]=(JS[N]&0xFFF0)|13;break; 

        case XK_Caps_Lock: case XK_Shift_L: case XK_Shift_R: 
          JS[0]=JS[1]=0xFFFF;N=1;break;
      }
    else
      switch(J)
      {
        case XK_0: case XK_1: case XK_2: case XK_3: case XK_4:
        case XK_5: case XK_6: case XK_7: case XK_8: case XK_9:
          if((JS[N]&0x000F)==(J-XK_0)) JS[N]|=0x000F;break; 

        case XK_z: case XK_x: case XK_c: case XK_v:
        case XK_b: case XK_n: case XK_m: 
        case XK_Z: case XK_X: case XK_C: case XK_V:
        case XK_B: case XK_N: case XK_M: 
        case XK_Control_L:
          JS[N]|=0x4000;break;

        case XK_a: case XK_s: case XK_d: case XK_f:
        case XK_g: case XK_h: case XK_j:
        case XK_A: case XK_S: case XK_D: case XK_F:
        case XK_G: case XK_H: case XK_J:
        case XK_space: case XK_Alt_L:
          JS[N]|=0x0040;break;

        case XK_Down:   JS[N]|=0x0400;break;
        case XK_Up:     JS[N]|=0x0100;break;
        case XK_Left:   JS[N]|=0x0800;break;
        case XK_Right:  JS[N]|=0x0200;break;
        case XK_minus:  if((JS[N]&0x000F)==10) JS[N]|=0x000F;break;
        case XK_equal:  if((JS[N]&0x000F)==11) JS[N]|=0x000F;break;

        case XK_q: case XK_e: case XK_t: case XK_u: case XK_o:
        case XK_Q: case XK_E: case XK_T: case XK_U: case XK_O:
        case XK_bracketleft:
          if((JS[N]&0x000F)==12) JS[N]|=0x000F;break;

        case XK_w: case XK_r: case XK_y: case XK_i: case XK_p:
        case XK_W: case XK_R: case XK_Y: case XK_I: case XK_P:
        case XK_bracketright:
          if((JS[N]&0x000F)==13) JS[N]|=0x000F;break;

        case XK_Caps_Lock: case XK_Shift_L: case XK_Shift_R:
          JS[0]=JS[1]=0xFFFF;N=0;break;
      }
  }

  /* Export joystick states into ColEm */
  JoyState[0]=JS[0];JoyState[1]=JS[1];

  /* If saving CPU and focus is out, sleep */
  for(E.type=0;XCheckWindowEvent(Dsp,Wnd,FocusChangeMask,&E););
  if(SaveCPU&&(E.type==FocusOut))
  {
    XAutoRepeatOn(Dsp);
#ifdef SOUND
    StopSound();
#endif

    do
      while(!XCheckWindowEvent(Dsp,Wnd,FocusChangeMask,&E)&&!ExitNow)
        sleep(1);
    while((E.type!=FocusIn)&&!ExitNow);

    XAutoRepeatOff(Dsp);
#ifdef SOUND
    ResumeSound();
#endif  
  }
}

/** Part of the code common for Unix/X and MSDOS drivers ********/
#include "Common.h"

#endif /* UNIX */
