// ---------------------------------------------------------------------------
//  M88 - PC88 emulator
//  Copyright (c) cisc 1998. 1999.
// ---------------------------------------------------------------------------
//  PC88 Keyboard Interface Emulation for Win32/106 key (Rev. 3)
// ---------------------------------------------------------------------------
//  $Id: WinKeyIF.cpp,v 1.3 1999/07/22 15:57:27 cisc Exp $

#include "headers.h"
#include "WinKeyIF.h"
#include "messages.h"
#include "pc88/config.h"

using namespace PC8801;

// ---------------------------------------------------------------------------
//  Construct/Destruct
//
#ifdef __OS2__
WinKeyIF::WinKeyIF()
: Device(0)
{
    hwnd = 0;

    for (int i=0; i<16; i++)
    {
        keyport[i] = -1;
    }
    memset(keyboard, 0, 256);
    memset(keystate, 0, 512);
    usearrow = false;

    disable = false;
}
#else
WinKeyIF::WinKeyIF()
: Device(0)
{
    hwnd = 0;
    hevent = 0;
    for (int i=0; i<16; i++)
    {
        keyport[i] = -1;
    }
    memset(keyboard, 0, 256);
    memset(keystate, 0, 512);
    usearrow = false;

    disable = false;
}
#endif

#ifdef __OS2__
WinKeyIF::~WinKeyIF()
{
}
#else
WinKeyIF::~WinKeyIF()
{
    if (hevent)
        CloseHandle(hevent);
}
#endif

// ---------------------------------------------------------------------------
//  
//
#ifdef __OS2__
bool WinKeyIF::Init(HWND hwndmsg)
{
    hwnd = hwndmsg;
    keytable = KeyTable106[0];
    return TRUE;
}
#else
bool WinKeyIF::Init(HWND hwndmsg)
{
    hwnd = hwndmsg;
    hevent = CreateEvent(0, 0, 0, 0);
    keytable = KeyTable106[0];
    return hevent != 0;
}
#endif

// ---------------------------------------------------------------------------
//  ZbgiƂABASIC [h̕ύXj
//
void WinKeyIF::Reset(uint, uint)
{
    pc80mode = (basicmode & 2) != 0;
}

// ---------------------------------------------------------------------------
//  ݒ蔽f
//
void WinKeyIF::ApplyConfig(const Config* config)
{
    usearrow = 0 != (config->flags & Config::usearrowfor10);
    basicmode = config->basicmode;

    switch (config->keytype)
    {
    case Config::PC98:
        keytable = KeyTable98[0];
        break;

    case Config::AT106:
    default:
        keytable = KeyTable106[0];
        break;
    }
}

 // ---------------------------------------------------------------------------
//  WM_KEYDOWN
//
void WinKeyIF::KeyDown(uint vkcode, uint32 keydata)
{
    uint keyindex = keydata;
    keystate[keyindex] = 1;
}

// ---------------------------------------------------------------------------
//  WM_KEYUP
//
#ifdef __OS2__
#define VK_NUMPAD0 500
#define VK_NUMPAD1 501
#define VK_NUMPAD2 502
#define VK_NUMPAD3 503
#define VK_NUMPAD4 504
#define VK_NUMPAD5 505
#define VK_NUMPAD6 506
#define VK_NUMPAD7 507
#define VK_NUMPAD8 508
#define VK_NUMPAD9 509
#define VK_NEXT    510
#define VK_PRIOR   511
#define VK_MULTIPLY 512
#define VK_ADD      513
#define VK_SEPARATOR 514
#define VK_DECIMAL   515
#define VK_RETURN    516
#define VK_BACK      517
#define VK_CONTROL   518
#define VK_SCROLL    519
#define VK_ESCAPE    520
#define VK_CONVERT   521
#define VK_CAPITAL   522
#define VK_NONCONVERT 523
#define VK_LSHIFT     524
#define VK_HELP       525
#define VK_RSHIFT     526
#define VK_ACCEPT     527
#endif
void WinKeyIF::KeyUp(uint vkcode, uint32 keydata)
{
    uint keyindex = keydata & 0xff;
    keystate[keyindex] = 0;

    return ;

    // SHIFT + eL[ɂ鉟ςȂۑ΍

    if (keytable == KeyTable106[0])
    {
        switch (keyindex)
        {
        case VK_NUMPAD0: case VK_INSERT:
            keystate[VK_NUMPAD0] = keystate[VK_INSERT] = 0;
            break;
        case VK_NUMPAD1: case VK_END:
            keystate[VK_NUMPAD1] = keystate[VK_END] = 0;
            break;
        case VK_NUMPAD2: case VK_DOWN:
            keystate[VK_NUMPAD2] = keystate[VK_DOWN] = 0;
            break;
        case VK_NUMPAD3: case VK_NEXT:
            keystate[VK_NUMPAD3] = keystate[VK_NEXT] = 0;
            break;
        case VK_NUMPAD4: case VK_LEFT:
            keystate[VK_NUMPAD4] = keystate[VK_LEFT] = 0;
            break;
        case VK_NUMPAD5: case VK_CLEAR:
            keystate[VK_NUMPAD5] = keystate[VK_CLEAR] = 0;
            break;
        case VK_NUMPAD6: case VK_RIGHT:
            keystate[VK_NUMPAD6] = keystate[VK_RIGHT] = 0;
            break;
        case VK_NUMPAD7: case VK_HOME:
            keystate[VK_NUMPAD7] = keystate[VK_HOME] = 0;
            break;
        case VK_NUMPAD8: case VK_UP:
            keystate[VK_NUMPAD8] = keystate[VK_UP] = 0;
            break;
        case VK_NUMPAD9: case VK_PRIOR:
            keystate[VK_NUMPAD9] = keystate[VK_PRIOR] = 0;
            break;
        }
    }
}

// ---------------------------------------------------------------------------
//  Key
//  keyboard ɂL[`FbN͔݂mꂸ
//
uint WinKeyIF::GetKey(const Key* key)
{
    uint i;

    for (i=0; i<8 && key->k; i++, key++)
    {
        switch (key->f)
        {
        case lock:
            if (keyboard[key->k] & 0x01)
                return 0;
            break;

        case keyb:
            if (keyboard[key->k] & 0x80)
                return 0;
            break;

        case nex:
            if (keystate[key->k])
                return 0;
            break;

        case ext:
            if (keystate[key->k | 0x100])
                return 0;
            break;

        case arrowten:
            if (usearrow && (keyboard[key->k] & 0x80))
                return 0;
            break;

        case noarrowten:
            if (!usearrow && (keyboard[key->k] & 0x80))
                return 0;
            break;

        case noarrowtenex:
            if (!usearrow && keystate[key->k | 0x100])
                return 0;
            break;

        case pc80sft:
            if (pc80mode && ((keyboard[VK_DOWN] & 0x80) || (keyboard[VK_LEFT] & 0x80)))
                return 0;
            break;

        case pc80key:
            if (pc80mode && (keyboard[key->k] & 0x80))
                return 0;
            break;

        default:
            if (keystate[key->k] | keystate[key->k | 0x100]) // & 0x80)
                return 0;
            break;
        }
    }
    return 1;
}

// ---------------------------------------------------------------------------
//  VSync 
//
#ifdef __OS2__
void WinKeyIF::VSync(uint,uint d)
{
    if (d && active)
    {
        if (hwnd)
        {
//            WinPostMsg(hwnd, WM_M88_SENDKEYSTATE, (MPARAM)(keyboard), 0);
//            WaitForSingleObject(hevent, 10);
              WinSetKeyboardStateTable( HWND_DESKTOP, (char *)keyboard, FALSE );
//            DosSleep( 10 );
        }

        for (int i=0; i<16; i++)
        {
            keyport[i] = -1;
        }
    }
}
#else
void WinKeyIF::VSync(uint,uint d)
{
    if (d && active)
    {
        if (hwnd)
        {
            PostMessage(hwnd, WM_M88_SENDKEYSTATE, reinterpret_cast<DWORD>(keyboard), (DWORD) hevent);
            WaitForSingleObject(hevent, 10);
        }

        for (int i=0; i<16; i++)
        {
            keyport[i] = -1;
        }
    }
}
#endif

void WinKeyIF::Activate(bool yes)
{
    active = yes;
    if (active)
    {
        memset(keystate, 0, 512);
        for (int i=0; i<16; i++)
        {
            keyport[i] = -1;
        }
    }
}

void WinKeyIF::Disable(bool yes)
{
    disable = yes;
}

// ---------------------------------------------------------------------------
//  L[
//
uint WinKeyIF::In(uint port)
{
    port &= 0x0f;

    if (active)
    {
        int r = keyport[port];
        if (r == -1)
        {
            const Key* key = keytable + port * 64 + 56;
            r=0;
            for (int i=0; i<8; i++)
            {
                r = r * 2 + GetKey(key);
                key -= 8;
            }
            keyport[port] = r;
        }
        return uint8(r);
    }
    else
        return 0xff;
}

// ---------------------------------------------------------------------------
//  L[Ή\
//  ЂƂ̃L[ɏGg͂W܂ŁB
//  W̏ꍇ͍Ō TERM t邱ƁB
//
//  KEYF  f ͎̂ǂꂩB
//  nex     WM_KEYxxx  extended tO 0 ̃L[̂
//  ext     WM_KEYxxx  extended tO 1 ̃L[̂
//  lock    bN@\L[ (ʂ CAPS LOCK Jî悤ɕI
//                                bN@\ĂKv͖͂)
//  arrowten L[eL[ɑΉꍇ̂
//

#define KEY(k)     { k, 0 }
#define KEYF(k,f)  { k, f }
#define TERM       { 0, 0 }

// ---------------------------------------------------------------------------
//  L[Ή\ for { 106 L[{[h
//
#ifdef __OS2__
const WinKeyIF::Key WinKeyIF::KeyTable106[16 * 8][8] =
{
    // 00
    { KEY(0x52), TERM, },   // num 0 @@@
    { KEY(0x4f), TERM, },   // num 1 @@@
    { KEY(0x50), TERM, },   // num 2 @@@
    { KEY(0x51), TERM, },   // num 3 @@@
    { KEY(0x4b), TERM, },   // num 4 @@@
    { KEY(0x4c), TERM, },   // num 5 @@@
    { KEY(0x4d), TERM, },   // num 6 @@@
    { KEY(0x47), TERM, },   // num 7 @@@

    // 01
    { KEY(0x48), TERM, },   // num 8 @@@
    { KEY(0x49), TERM, },   // num 9 @@@
    { KEY(0x37), TERM, },   // num * @@@
    { KEY(0x4e), TERM, },   // num + @@@
    { TERM, },              // num =
    { TERM, },              // num ,
    { KEY(0x53),  TERM, },  // num . @@@
    { KEYF(VK_ENTER, keyb), KEYF(VK_NEWLINE, keyb) /* KEY(0x1c) */ ,  TERM, },  // RET @@@

    // 02
    { KEY(0x1a), TERM }, // @ @@@
    { KEY(0x1e), TERM }, // A @@@
    { KEY(0x30), TERM }, // B @@@
    { KEY(0x2e), TERM }, // C @@@
    { KEY(0x20), TERM }, // D @@@
    { KEY(0x12), TERM }, // E @@@
    { KEY(0x21), TERM }, // F @@@
    { KEY(0x22), TERM }, // G @@@

    // 03
    { KEY(0x23), TERM }, // H @@@
    { KEY(0x17), TERM }, // I @@@
    { KEY(0x24), TERM }, // J @@@
    { KEY(0x25), TERM }, // K @@@
    { KEY(0x26), TERM }, // L @@@
    { KEY(0x32), TERM }, // M @@@
    { KEY(0x31), TERM }, // N @@@
    { KEY(0x18), TERM }, // O @@@

    // 04
    { KEY(0x19), TERM }, // P @@@
    { KEY(0x10), TERM }, // Q @@@
    { KEY(0x13), TERM }, // R @@@
    { KEY(0x1f), TERM }, // S @@@
    { KEY(0x14), TERM }, // T @@@
    { KEY(0x16), TERM }, // U @@@
    { KEY(0x2f), TERM }, // V @@@
    { KEY(0x11), TERM }, // W @@@

    // 05
    { KEY(0x2d), TERM }, // X @@@
    { KEY(0x15), TERM }, // Y @@@
    { KEY(0x2c), TERM }, // Z @@@
    { KEY(0x1b), TERM }, // [ @@@
    { KEY(0x7d), TERM }, // \ ___ @@@
    { KEY(0x2b), TERM }, // ] @@@
    { KEY(0x0d), TERM }, // ^ @@@
    { KEY(0x0c), TERM }, // - @@@

    // 06
    { KEY(0x0b), TERM }, // 0 @@@
    { KEY(0x02), TERM }, // 1 @@@
    { KEY(0x03), TERM }, // 2 @@@
    { KEY(0x04), TERM }, // 3 @@@
    { KEY(0x05), TERM }, // 4 @@@
    { KEY(0x06), TERM }, // 5 @@@
    { KEY(0x07), TERM }, // 6 @@@
    { KEY(0x08), TERM }, // 7 @@@

    // 07
    { KEY(0x09), TERM }, // 8 @@@
    { KEY(0x0a), TERM }, // 9 @@@
    { KEY(0x28), TERM }, // : @@@
    { KEY(0x27), TERM }, // ; @@@
    { KEY(0x33), TERM }, // , @@@
    { KEY(0x34), TERM }, // . @@@
    { KEY(0x35), TERM }, // / @@@
    { KEY(0xdf), TERM }, // _

    // 08
    { TERM }, // CLR
    { KEYF(VK_UP, keyb), TERM },        //  @@@
    { KEYF(VK_RIGHT, keyb), TERM },     //  @@@
    { KEY(0x0e), TERM }, // BS @@@
    { TERM }, // GRPH
    { KEYF(VK_SCRLLOCK, lock),TERM },   // Ji  @@@
    { KEYF(VK_SHIFT, keyb), TERM },     // SHIFT @@@
    { KEYF(VK_CTRL,  keyb), TERM  },    // CTRL  @@@

    // 09
    { TERM }, // STOP
    { KEYF(VK_F1, keyb), TERM },        // F1 @@@
    { KEYF(VK_F2, keyb), TERM },        // F2 @@@
    { KEYF(VK_F3, keyb), TERM },        // F3 @@@
    { KEYF(VK_F4, keyb), TERM },        // F4 @@@
    { KEYF(VK_F5, keyb), TERM },        // F5 @@@
    { KEYF(VK_SPACE, keyb), TERM },     // SPACE @@@
    { KEY(0x01), TERM },                // ESC @@@

    // 0a
    { KEYF(VK_TAB,  keyb),  TERM },     // TAB @@@
    { KEYF(VK_DOWN, keyb),  TERM },     //  @@@
    { KEYF(VK_LEFT, keyb),  TERM },     //  @@@
    { KEYF(VK_END,  keyb),  TERM },     // HELP @@@
    { TERM }, // COPY
    { TERM }, // -
    { TERM }, // /
    { KEYF(VK_CAPSLOCK, lock), TERM },  // CAPS LOCK @@@

    // 0b
    { KEYF(VK_PAGEDOWN, keyb), TERM },  // ROLL DOWN @@@
    { KEYF(VK_PAGEUP, keyb), TERM },    // ROLL UP   @@@
    { TERM, },
    { TERM, },
    { TERM, },
    { TERM, },
    { TERM, },
    { TERM, },

    // 0c
    { KEYF(VK_F6, keyb), TERM },        // F6
    { KEYF(VK_F7, keyb), TERM },        // F7
    { KEYF(VK_F8, keyb), TERM },        // F8
    { KEYF(VK_F9, keyb), TERM },        // F9
    { KEYF(VK_F10, keyb), TERM },       // F10
    { KEYF(VK_BACKSPACE, keyb), TERM }, // BS
    { KEYF(VK_INSERT, keyb), TERM },    // INS
    { KEYF(VK_DELETE, keyb), TERM },    // DEL

    // 0d
    { TERM },      // ϊ
    { TERM }, // 
    { TERM },                       // PC
    { TERM },            // Sp
    { TERM },
    { TERM },
    { TERM },
    { TERM },

    // 0e
    { KEYF(VK_ENTER, keyb), KEYF(VK_NEWLINE, keyb), TERM },     // RET FK
    { KEYF(VK_ENTER, keyb), KEYF(VK_NEWLINE, keyb), TERM },     // RET 10
    { KEY(0x96), TERM },       // SHIFT L @@@
    { KEY(0x95), TERM },       // SHIFT R @@@
    { TERM },
    { TERM },
    { TERM },
    { TERM },

    // 0f
    { TERM },
    { TERM },
    { TERM },
    { TERM },
    { TERM },
    { TERM },
    { TERM },
    { TERM },
};

// ---------------------------------------------------------------------------
//  L[Ή\ for 9801 key
//
const WinKeyIF::Key WinKeyIF::KeyTable98[16 * 8][8] =
{
    // 00
    { KEY(VK_NUMPAD0), TERM, },     // num 0
    { KEY(VK_NUMPAD1), TERM, },     // num 1
    { KEY(VK_NUMPAD2), KEYF(VK_DOWN, arrowten), TERM, },        // num 2
    { KEY(VK_NUMPAD3), TERM, },     // num 3
    { KEY(VK_NUMPAD4), KEYF(VK_LEFT, arrowten), TERM, },        // num 4
    { KEY(VK_NUMPAD5), TERM, },     // num 5
    { KEY(VK_NUMPAD6), KEYF(VK_RIGHT,arrowten), TERM, },        // num 6
    { KEY(VK_NUMPAD7), TERM, },     // num 7

    // 01
    { KEY(VK_NUMPAD8), KEYF(VK_UP  , arrowten), TERM, },        // num 8
    { KEY(VK_NUMPAD9), TERM, },     // num 9
    { KEY(VK_MULTIPLY), TERM, },    // num *
    { KEY(VK_ADD),      TERM, },    // num +
    { KEY(0x92),        TERM, },    // num =
    { KEY(VK_SEPARATOR), TERM, },   // num ,
    { KEY(VK_DECIMAL),  TERM, },    // num .
    { KEY(0x1c),   TERM, },    // RET @@@

    // 02
    { KEY(0xc0),TERM }, // @
    { KEY('A'), TERM }, // A
    { KEY('B'), TERM }, // B
    { KEY('C'), TERM }, // C
    { KEY('D'), TERM }, // D
    { KEY('E'), TERM }, // E
    { KEY('F'), TERM }, // F
    { KEY('G'), TERM }, // G

    // 03
    { KEY('H'), TERM }, // H
    { KEY('I'), TERM }, // I
    { KEY('J'), TERM }, // J
    { KEY('K'), TERM }, // K
    { KEY('L'), TERM }, // L
    { KEY('M'), TERM }, // M
    { KEY('N'), TERM }, // N
    { KEY('O'), TERM }, // O

    // 04
    { KEY('P'), TERM }, // P
    { KEY('Q'), TERM }, // Q
    { KEY('R'), TERM }, // R
    { KEY('S'), TERM }, // S
    { KEY('T'), TERM }, // T
    { KEY('U'), TERM }, // U
    { KEY('V'), TERM }, // V
    { KEY('W'), TERM }, // W

    // 05
    { KEY('X'), TERM }, // X
    { KEY('Y'), TERM }, // Y
    { KEY('Z'), TERM }, // Z
    { KEY(0xdb),TERM }, // [
    { KEY(0xdc),TERM }, // \ ___
    { KEY(0xdd),TERM }, // ]
    { KEY(0xde),TERM }, // ^
    { KEY(0xbd),TERM }, // -

    // 06
    { KEY(0x0b), TERM }, // 0 @@@
    { KEY(0x02), TERM }, // 1 @@@
    { KEY(0x03), TERM }, // 2 @@@
    { KEY(0x04), TERM }, // 3 @@@
    { KEY(0x05), TERM }, // 4 @@@
    { KEY(0x06), TERM }, // 5 @@@
    { KEY(0x07), TERM }, // 6 @@@
    { KEY(0x08), TERM }, // 7 @@@

    // 07
    { KEY(0x09), TERM }, // 8 @@@
    { KEY(0x0a), TERM }, // 9 @@@
    { KEY(0xba),TERM }, // :
    { KEY(0x27),TERM }, // ; @@@
    { KEY(0x33),TERM }, // , @@@
    { KEY(0x34),TERM }, // . @@@
    { KEY(0xbf),TERM }, // /
    { KEY(0xdf),TERM }, // _

    // 08
    { KEY(VK_HOME),     TERM }, // CLR
    { KEYF(VK_UP, noarrowten),      TERM }, // 
    { KEYF(VK_RIGHT, noarrowten),   TERM }, // 
    { KEY(VK_BACK),     TERM }, // BS
    { KEY(VK_MENU),     TERM }, // GRPH
    { KEYF(0x15, lock), TERM }, // Ji
    { KEY(VK_SHIFT), KEY(VK_F6), KEY(VK_F7), KEY(VK_F8), KEY(VK_F9), KEY(VK_F10), TERM }, // SHIFT
    { KEY(VK_CONTROL),      TERM }, // CTRL

    // 09
    { KEY(VK_PAUSE),KEY(VK_F11),KEY(VK_F15),TERM }, // STOP
    { KEY(VK_F1),   KEY(VK_F6), TERM }, // F1
    { KEY(VK_F2),   KEY(VK_F7), TERM }, // F2
    { KEY(VK_F3),   KEY(VK_F8), TERM }, // F3
    { KEY(VK_F4),   KEY(VK_F9), TERM }, // F4
    { KEY(VK_F5),   KEY(VK_F10),TERM }, // F5
    { KEY(VK_SPACE), KEY(0x1d), KEY(0x19), TERM }, // SPACE
    { KEY(VK_ESCAPE),   TERM }, // ESC

    // 0a
    { KEY(VK_TAB),      TERM }, // TAB
    { KEYF(VK_DOWN, noarrowten),        TERM }, // 
    { KEYF(VK_LEFT, noarrowten),        TERM }, // 
    { KEY(VK_END),      TERM }, // HELP
    { KEY(VK_F14),  KEY(VK_F12),    TERM }, // COPY
    { KEY(0x6d),        TERM }, // -
    { KEY(0x6f),        TERM }, // /
    { KEYF(VK_CAPITAL, lock), TERM }, // CAPS LOCK

    // 0b
    { KEY(VK_NEXT),     TERM }, // ROLL DOWN
    { KEY(VK_PRIOR),    TERM }, // ROLL UP
    { TERM, },
    { TERM, },
    { TERM, },
    { TERM, },
    { TERM, },
    { TERM, },

    // 0c
    { KEY(VK_F1), TERM }, // F1
    { KEY(VK_F2), TERM }, // F2
    { KEY(VK_F3), TERM }, // F3
    { KEY(VK_F4), TERM }, // F4
    { KEY(VK_F5), TERM }, // F5
    { KEY(VK_BACK), TERM }, // BS
    { KEY(VK_INSERT), TERM }, // INS
    { KEY(VK_DELETE), TERM }, // DEL

    // 0d
    { KEY(VK_F6),   TERM }, // F6
    { KEY(VK_F7),   TERM }, // F7
    { KEY(VK_F8),   TERM }, // F8
    { KEY(VK_F9),   TERM }, // F9
    { KEY(VK_F10),  TERM }, // F10
    { KEY(0x1d),    TERM }, // ϊ
    { KEY(0x19),    TERM }, // 
    { KEY(VK_SPACE), TERM }, // SPACE

    // 0e
    { KEY(VK_RETURN), TERM }, // RET FK
    { KEY(VK_RETURN), TERM }, // RET 10
    { KEY(VK_LSHIFT), TERM }, // SHIFT L
    { KEY(VK_RSHIFT), TERM }, // SHIFT R
    { TERM }, // PC
    { TERM }, // Sp
    { TERM },
    { TERM },

    // 0f
    { TERM },
    { TERM },
    { TERM },
    { TERM },
    { TERM },
    { TERM },
    { TERM },
    { TERM },
};
#else
const WinKeyIF::Key WinKeyIF::KeyTable106[16 * 8][8] =
{
    // 00
    { KEY(VK_NUMPAD0), KEYF(VK_INSERT, nex), TERM, },   // num 0
    { KEY(VK_NUMPAD1), KEYF(VK_END,    nex), TERM, },   // num 1
    { KEY(VK_NUMPAD2), KEYF(VK_DOWN,   nex), KEYF(VK_DOWN, arrowten), TERM, },
    { KEY(VK_NUMPAD3), KEYF(VK_NEXT,   nex), TERM, },   // num 3
    { KEY(VK_NUMPAD4), KEYF(VK_LEFT,   nex), KEYF(VK_LEFT, arrowten), TERM, },
    { KEY(VK_NUMPAD5), KEYF(VK_CLEAR,  nex), TERM, },   // num 5
    { KEY(VK_NUMPAD6), KEYF(VK_RIGHT,  nex), KEYF(VK_RIGHT,arrowten), TERM, },
    { KEY(VK_NUMPAD7), KEYF(VK_HOME,   nex), TERM, },   // num 7

    // 01
    { KEY(VK_NUMPAD8), KEYF(VK_UP,     nex), KEYF(VK_UP  , arrowten), TERM, },
    { KEY(VK_NUMPAD9), KEYF(VK_PRIOR,  nex), TERM, },   // num 9
    { KEY(VK_MULTIPLY), TERM, },                        // num *
    { KEY(VK_ADD),      TERM, },                        // num +
    { TERM, },                                          // num =
    { KEY(VK_SEPARATOR), KEYF(VK_DELETE, nex), TERM, }, // num ,
    { KEY(VK_DECIMAL),  TERM, },                        // num .
    { KEY(VK_RETURN),   TERM, },                        // RET

    // 02
    { KEY(0xc0),TERM }, // @
    { KEY('A'), TERM }, // A
    { KEY('B'), TERM }, // B
    { KEY('C'), TERM }, // C
    { KEY('D'), TERM }, // D
    { KEY('E'), TERM }, // E
    { KEY('F'), TERM }, // F
    { KEY('G'), TERM }, // G

    // 03
    { KEY('H'), TERM }, // H
    { KEY('I'), TERM }, // I
    { KEY('J'), TERM }, // J
    { KEY('K'), TERM }, // K
    { KEY('L'), TERM }, // L
    { KEY('M'), TERM }, // M
    { KEY('N'), TERM }, // N
    { KEY('O'), TERM }, // O

    // 04
    { KEY('P'), TERM }, // P
    { KEY('Q'), TERM }, // Q
    { KEY('R'), TERM }, // R
    { KEY('S'), TERM }, // S
    { KEY('T'), TERM }, // T
    { KEY('U'), TERM }, // U
    { KEY('V'), TERM }, // V
    { KEY('W'), TERM }, // W

    // 05
    { KEY('X'), TERM }, // X
    { KEY('Y'), TERM }, // Y
    { KEY('Z'), TERM }, // Z
    { KEY(0xdb),TERM }, // [
    { KEY(0xdc),TERM }, // \
    { KEY(0xdd),TERM }, // ]
    { KEY(0xde),TERM }, // ^
    { KEY(0xbd),TERM }, // -

    // 06
    { KEY('0'), TERM }, // 0
    { KEY('1'), TERM }, // 1
    { KEY('2'), TERM }, // 2
    { KEY('3'), TERM }, // 3
    { KEY('4'), TERM }, // 4
    { KEY('5'), TERM }, // 5
    { KEY('6'), TERM }, // 6
    { KEY('7'), TERM }, // 7

    // 07
    { KEY('8'), TERM }, // 8
    { KEY('9'), TERM }, // 9
    { KEY(0xba),TERM }, // :
    { KEY(0xbb),TERM }, // ;
    { KEY(0xbc),TERM }, // ,
    { KEY(0xbe),TERM }, // .
    { KEY(0xbf),TERM }, // /
    { KEY(0xe2),TERM }, // _

    // 08
    { KEYF(VK_HOME, ext),   TERM }, // CLR
    { KEYF(VK_UP, noarrowtenex),    KEYF(VK_DOWN, pc80key), TERM }, // 
    { KEYF(VK_RIGHT, noarrowtenex), KEYF(VK_LEFT, pc80key), TERM }, // 
    { KEY(VK_BACK), KEYF(VK_INSERT, ext), KEYF(VK_DELETE, ext), TERM }, // BS
    { KEY(VK_MENU),         TERM }, // GRPH
    { KEYF(VK_SCROLL, lock),TERM }, // Ji
    { KEY(VK_SHIFT), KEY(VK_F6), KEY(VK_F7), KEY(VK_F8), KEY(VK_F9), KEY(VK_F10), KEYF(VK_INSERT, ext), KEYF(1, pc80sft) }, // SHIFT
    { KEY(VK_CONTROL),      TERM }, // CTRL

    // 09
    { KEY(VK_F11),KEY(VK_PAUSE), TERM }, // STOP
    { KEY(VK_F1), KEY(VK_F6),   TERM }, // F1
    { KEY(VK_F2), KEY(VK_F7),   TERM }, // F2
    { KEY(VK_F3), KEY(VK_F8),   TERM }, // F3
    { KEY(VK_F4), KEY(VK_F9),   TERM }, // F4
    { KEY(VK_F5), KEY(VK_F10),  TERM }, // F5
    { KEY(VK_SPACE), KEY(VK_CONVERT), KEY(VK_NONCONVERT), TERM }, // SPACE
    { KEY(VK_ESCAPE),   TERM }, // ESC

    // 0a
    { KEY(VK_TAB),          TERM }, // TAB
    { KEYF(VK_DOWN, noarrowtenex),  TERM }, // 
    { KEYF(VK_LEFT, noarrowtenex),  TERM }, // 
    { KEYF(VK_END, ext), KEY(VK_HELP), TERM }, // HELP
    { KEY(VK_F12), TERM }, // COPY
    { KEY(0x6d),            TERM }, // -
    { KEY(0x6f),            TERM }, // /
    { KEYF(VK_CAPITAL, lock), TERM }, // CAPS LOCK

    // 0b
    { KEYF(VK_NEXT, ext),   TERM }, // ROLL DOWN
    { KEYF(VK_PRIOR, ext),  TERM }, // ROLL UP
    { TERM, },
    { TERM, },
    { TERM, },
    { TERM, },
    { TERM, },
    { TERM, },

    // 0c
    { KEY(VK_F6), TERM },           // F6
    { KEY(VK_F7), TERM },           // F7
    { KEY(VK_F8), TERM },           // F8
    { KEY(VK_F9), TERM },           // F9
    { KEY(VK_F10), TERM },          // F10
    { KEY(VK_BACK), TERM },         // BS
    { KEYF(VK_INSERT, ext), TERM }, // INS
    { KEYF(VK_DELETE, ext), TERM }, // DEL

    // 0d
    { KEY(VK_CONVERT), TERM },      // ϊ
    { KEY(VK_NONCONVERT), KEY(VK_ACCEPT), TERM }, // 
    { TERM },                       // PC
    { KEY(0xf4), TERM },            // Sp
    { TERM },
    { TERM },
    { TERM },
    { TERM },

    // 0e
    { KEYF(VK_RETURN, nex), TERM },     // RET FK
    { KEYF(VK_RETURN, ext), TERM },     // RET 10
    { KEY(VK_LSHIFT), TERM },       // SHIFT L
    { KEY(VK_RSHIFT), TERM },       // SHIFT R
    { TERM },
    { TERM },
    { TERM },
    { TERM },

    // 0f
    { TERM },
    { TERM },
    { TERM },
    { TERM },
    { TERM },
    { TERM },
    { TERM },
    { TERM },
};

// ---------------------------------------------------------------------------
//  L[Ή\ for 9801 key
//
const WinKeyIF::Key WinKeyIF::KeyTable98[16 * 8][8] =
{
    // 00
    { KEY(VK_NUMPAD0), TERM, },     // num 0
    { KEY(VK_NUMPAD1), TERM, },     // num 1
    { KEY(VK_NUMPAD2), KEYF(VK_DOWN, arrowten), TERM, },        // num 2
    { KEY(VK_NUMPAD3), TERM, },     // num 3
    { KEY(VK_NUMPAD4), KEYF(VK_LEFT, arrowten), TERM, },        // num 4
    { KEY(VK_NUMPAD5), TERM, },     // num 5
    { KEY(VK_NUMPAD6), KEYF(VK_RIGHT,arrowten), TERM, },        // num 6
    { KEY(VK_NUMPAD7), TERM, },     // num 7

    // 01
    { KEY(VK_NUMPAD8), KEYF(VK_UP  , arrowten), TERM, },        // num 8
    { KEY(VK_NUMPAD9), TERM, },     // num 9
    { KEY(VK_MULTIPLY), TERM, },    // num *
    { KEY(VK_ADD),      TERM, },    // num +
    { KEY(0x92),        TERM, },    // num =
    { KEY(VK_SEPARATOR), TERM, },   // num ,
    { KEY(VK_DECIMAL),  TERM, },    // num .
    { KEY(VK_RETURN),   TERM, },    // RET

    // 02
    { KEY(0xc0),TERM }, // @
    { KEY('A'), TERM }, // A
    { KEY('B'), TERM }, // B
    { KEY('C'), TERM }, // C
    { KEY('D'), TERM }, // D
    { KEY('E'), TERM }, // E
    { KEY('F'), TERM }, // F
    { KEY('G'), TERM }, // G

    // 03
    { KEY('H'), TERM }, // H
    { KEY('I'), TERM }, // I
    { KEY('J'), TERM }, // J
    { KEY('K'), TERM }, // K
    { KEY('L'), TERM }, // L
    { KEY('M'), TERM }, // M
    { KEY('N'), TERM }, // N
    { KEY('O'), TERM }, // O

    // 04
    { KEY('P'), TERM }, // P
    { KEY('Q'), TERM }, // Q
    { KEY('R'), TERM }, // R
    { KEY('S'), TERM }, // S
    { KEY('T'), TERM }, // T
    { KEY('U'), TERM }, // U
    { KEY('V'), TERM }, // V
    { KEY('W'), TERM }, // W

    // 05
    { KEY('X'), TERM }, // X
    { KEY('Y'), TERM }, // Y
    { KEY('Z'), TERM }, // Z
    { KEY(0xdb),TERM }, // [
    { KEY(0xdc),TERM }, // \
    { KEY(0xdd),TERM }, // ]
    { KEY(0xde),TERM }, // ^
    { KEY(0xbd),TERM }, // -

    // 06
    { KEY('0'), TERM }, // 0
    { KEY('1'), TERM }, // 1
    { KEY('2'), TERM }, // 2
    { KEY('3'), TERM }, // 3
    { KEY('4'), TERM }, // 4
    { KEY('5'), TERM }, // 5
    { KEY('6'), TERM }, // 6
    { KEY('7'), TERM }, // 7

    // 07
    { KEY('8'), TERM }, // 8
    { KEY('9'), TERM }, // 9
    { KEY(0xba),TERM }, // :
    { KEY(0xbb),TERM }, // ;
    { KEY(0xbc),TERM }, // ,
    { KEY(0xbe),TERM }, // .
    { KEY(0xbf),TERM }, // /
    { KEY(0xdf),TERM }, // _

    // 08
    { KEY(VK_HOME),     TERM }, // CLR
    { KEYF(VK_UP, noarrowten),      TERM }, // 
    { KEYF(VK_RIGHT, noarrowten),   TERM }, // 
    { KEY(VK_BACK),     TERM }, // BS
    { KEY(VK_MENU),     TERM }, // GRPH
    { KEYF(0x15, lock), TERM }, // Ji
    { KEY(VK_SHIFT), KEY(VK_F6), KEY(VK_F7), KEY(VK_F8), KEY(VK_F9), KEY(VK_F10), KEYF(0, pc80sft) }, // SHIFT
    { KEY(VK_CONTROL),      TERM }, // CTRL

    // 09
    { KEY(VK_PAUSE),KEY(VK_F11),KEY(VK_F15),TERM }, // STOP
    { KEY(VK_F1),   KEY(VK_F6), TERM }, // F1
    { KEY(VK_F2),   KEY(VK_F7), TERM }, // F2
    { KEY(VK_F3),   KEY(VK_F8), TERM }, // F3
    { KEY(VK_F4),   KEY(VK_F9), TERM }, // F4
    { KEY(VK_F5),   KEY(VK_F10),TERM }, // F5
    { KEY(VK_SPACE), KEY(0x1d), KEY(0x19), TERM }, // SPACE
    { KEY(VK_ESCAPE),   TERM }, // ESC

    // 0a
    { KEY(VK_TAB),      TERM }, // TAB
    { KEYF(VK_DOWN, noarrowten),        TERM }, // 
    { KEYF(VK_LEFT, noarrowten),        TERM }, // 
    { KEY(VK_END),      TERM }, // HELP
    { KEY(VK_F14),  KEY(VK_F12),    TERM }, // COPY
    { KEY(0x6d),        TERM }, // -
    { KEY(0x6f),        TERM }, // /
    { KEYF(VK_CAPITAL, lock), TERM }, // CAPS LOCK

    // 0b
    { KEY(VK_NEXT),     TERM }, // ROLL DOWN
    { KEY(VK_PRIOR),    TERM }, // ROLL UP
    { TERM, },
    { TERM, },
    { TERM, },
    { TERM, },
    { TERM, },
    { TERM, },

    // 0c
    { KEY(VK_F1), TERM }, // F1
    { KEY(VK_F2), TERM }, // F2
    { KEY(VK_F3), TERM }, // F3
    { KEY(VK_F4), TERM }, // F4
    { KEY(VK_F5), TERM }, // F5
    { KEY(VK_BACK), TERM }, // BS
    { KEY(VK_INSERT), TERM }, // INS
    { KEY(VK_DELETE), TERM }, // DEL

    // 0d
    { KEY(VK_F6),   TERM }, // F6
    { KEY(VK_F7),   TERM }, // F7
    { KEY(VK_F8),   TERM }, // F8
    { KEY(VK_F9),   TERM }, // F9
    { KEY(VK_F10),  TERM }, // F10
    { KEY(0x1d),    TERM }, // ϊ
    { KEY(0x19),    TERM }, // 
    { KEY(VK_SPACE), TERM }, // SPACE

    // 0e
    { KEY(VK_RETURN), TERM }, // RET FK
    { KEY(VK_RETURN), TERM }, // RET 10
    { KEY(VK_LSHIFT), TERM }, // SHIFT L
    { KEY(VK_RSHIFT), TERM }, // SHIFT R
    { TERM }, // PC
    { TERM }, // Sp
    { TERM },
    { TERM },

    // 0f
    { TERM },
    { TERM },
    { TERM },
    { TERM },
    { TERM },
    { TERM },
    { TERM },
    { TERM },
};
#endif

// ---------------------------------------------------------------------------
//  device description
//
const Device::Descriptor WinKeyIF::descriptor = { indef, outdef };

const Device::OutFuncPtr WinKeyIF::outdef[] =
{
#ifndef __OS2__
    static_cast<OutFuncPtr> (Reset),
    static_cast<OutFuncPtr> (VSync),
#else
    (Device::OutFuncPtr) (Reset),
    (Device::OutFuncPtr) (VSync),
#endif
};

const Device::InFuncPtr WinKeyIF::indef[] =
{
#ifndef __OS2__
    static_cast<InFuncPtr> (In),
#else
    (Device::InFuncPtr) (In),
#endif
};
