#!/bin/sh

# Solaris platform specific script to start the Tomcat web server instance.
#
# wcstart [username] [groupname]

USER=$1
GROUP=$2

# Check that user identity is a valid user.  If not specified as an option,
# use the default configuration user identity.
if [ -z "${USER}" ]; then
    USER="%%console_user%%"
fi
line=`id -a ${USER}`
if [ $? -ne 0 ]; then
    echo "WCERR: Startup failed; invalid user identity \"${USER}\"." 1>&2
    exit 1
fi

# Get group identity.  If specified as an option, check that the user
# is a member of this group.  If not specified, use the primary group.
if [ -z "${GROUP}" ]; then
    GROUP=`/usr/xpg4/bin/id -gn ${USER}`
else
    chk=`echo "${line}" | awk '{print $2 $3}' | grep "(${GROUP})"`
    if [ -z "${chk}" ]; then
	echo "WCERR: Startup failed; user \"${USER}\" is not a member of group \"${GROUP}\"." 1>&2
	exit 1
    fi
fi

###################################################################
#
# Solaris 10 and later...
# Update the execution profiles database for the web console.
# We only update values for the "uid" and "gid" attributes.
#
# $USER      = userName or userID
# $GROUP     = groupName or groupID
# $START_CMD = entry command name
#
###################################################################

UpdateExecProfs() {

    EXEC_ATTR=/etc/security/exec_attr
    PROF_NAME="Web Console Management"
    PRIVS=proc_audit

    # Find the line based upon unique command name.
    line=`grep "$START_CMD" $EXEC_ATTR`
    if [ $? -ne 0 ]; then
        # Entry does not exist.  Create new one with specified uid and
        # gid values
        newLine="$PROF_NAME:solaris:cmd:::${START_CMD}:uid=$USER;gid=$GROUP;privs=${PRIVS}"
        echo "$newLine" >> $EXEC_ATTR

    else
        # Entry already exists.

        # Extract the `attr' field.  This is an optional list of
        # semicolon-separated (;)  name=value pairs.  We also replace
        # the semicolon with a space so we can loop over the pairs.
        #
        attrs=`echo $line | awk -F: '{print $NF}' | sed -e "s/;/ /g"`

        # Loop over the attrs, replace the values for `uid' and `gid' with
        # the specified new values respectively, building up a new attr field.
        #
        newAttr=""
	SWUP=0
        for a in $attrs
        do
            # Extract name and value
            name=`echo $a | awk -F= '{print $1}'`
            value=`echo $a | awk -F= '{print $2}'`

            # If uid field, replace with specified user name.
            if [ "$name" = "uid" ]; then
                if [ -n "$USER" -a "$USER" != "$value" ]; then
                    value=$USER
		    SWUP=1
                fi
            fi
            # If gid field, replace with specified group name.
            if [ "$name" = "gid" ]; then
                if [ -n "$GROUP" -a "$GROUP" != "$value" ]; then
                    value=$GROUP
		    SWUP=1
                fi
            fi

            # Rebuild new attr field, stripping off any leading semi-colon.
            newAttr="$newAttr;$name=$value"
            newAttr=`echo $newAttr | sed -e "s/^;//"`

        done

        # Replacement line preserves all fields but the new attr field.
        newLine="`echo $line | cut -d: -f1-6`:$newAttr"

        # Replace the line only if user or group actually changed.
	if [ "${SWUP}" = "1" ]; then
            tmp=/tmp/exec_attr.$$
            sed -e "\@${START_CMD}@c\\
$newLine" $EXEC_ATTR > $tmp
            cp $tmp $EXEC_ATTR
            rm -f $tmp
	fi

    fi

} # UpdateExecProfs

###################################################################
#
# Main
#
###################################################################

# Make sure Tomcat and log directories are writable by user
chown -R ${USER} %%console_dombase%%/%%console_instance%%/webapps
chown -R ${USER} %%console_dombase%%/%%console_instance%%/work
chmod 775 %%console_log%%/%%console_instance%%
chown ${USER} %%console_log%%/%%console_instance%%

# Make sure log file exists and is writable by user
LOG_FILE=%%console_log%%/%%console_instance%%/console_debug_log
touch ${LOG_FILE}
chown ${USER} ${LOG_FILE}

CMD=%%console_dombase%%/%%console_instance%%/conf/start_tomcat
chmod 755 ${CMD}

OSREL=`uname -r`
if [ "${OSREL}" = "5.8" -o "${OSREL}" = "5.9" ]; then

    # For 5.9 and earlier, use su command.
    su ${USER} -c "pwd" >/dev/null 2>&1
    if [ $? -ne 0 ]; then
	echo "WCERR: Startup failed: unable to become the user identity \"${USER}\"." 1>&2
	echo "This may be due to one of the following:" 1>&2
	echo "  - No permissions to read the ${USER} home directory." 1>&2
	echo "  - No permissions to read the root (/) home directory." 1>&2
	echo "  - Invalid login shell for the user ${USER}." 1>&2
	exit 1
    fi
    su ${USER} -c "nohup ${CMD}" >/dev/null 2>&1
    STAT=$?

else

    # For 5.10 and later, use pfexec command.
    START_CMD=%%console_home%%/private/bin/smcwebstart
    UpdateExecProfs
    pfexec ${START_CMD} ${CMD}
    STAT=$?
fi

if [ ${STAT} -ne 0 ]; then
    echo "WCERR: Startup failed with error ${STAT}" 1>&2
fi

exit ${STAT}
