//
// my_window.cc
//


#include <Window.h>
#include <FilePanel.h>
#include <ScrollBar.h>
#include <ScrollView.h>
#include <MenuBar.h>
#include <MenuItem.h>
#include <StringView.h>
#include <Alert.h>
#include <Path.h>
#include <stdio.h>
#include <string.h>

#include "my_view.h"
#include "my_window.h"
#include "my_app.h"


// (Im/Ex)port these functions to beio.c
extern "C" {
	// Old entry point in jzip.c
	extern int old_main(int argc, char **argv);

	void Be_scroll_line(void);
	void Be_display_char(int c);
	void Be_select_status_window(void);
	void Be_select_text_window(void);
	void Be_clear_screen(void);
	int Be_input_line(int buflen, char *buffer, int timeout, int *read_size);
	int Be_input_character(int timeout);
	void Be_move_cursor(int row, int col);
}

// Stuff to handle output
const int BUFFER_LEN				= 256;
static int buffer_idx					= 0;
static int status_idx					= 0;
static int input_offset				= 0;
static char output_buffer[BUFFER_LEN + 1]	= "";
static char status_buffer[BUFFER_LEN + 1]	= "                                                             ";
static bool in_status_window			= false;


my_window::my_window(BRect frame, char *title) :
			BWindow((BRect) frame, (const char *) "BeJZip", B_TITLED_WINDOW, uint32(NULL)),
			file_requester(0), text_view(0), sleep_type(I_WANT_NOWT)
{
	BRect view_rect = Bounds();
	view_rect.right -= B_V_SCROLL_BAR_WIDTH;
//	view_rect.bottom -= B_H_SCROLL_BAR_HEIGHT;

	// Add a menu...
	BMenuBar *menu_bar	= new BMenuBar(Bounds(), "mb");
	BMenu *option_menu	= new BMenu("File");
	option_menu->AddItem(new BMenuItem("About", new BMessage(MSG_ABOUT)));
	option_menu->AddItem(new BMenuItem("Play New Game...", new BMessage(MSG_NEW)));
	option_menu->AddItem(new BSeparatorItem);
	option_menu->AddItem(new BMenuItem("Quit", new BMessage(MSG_QUIT)));
	menu_bar->AddItem(option_menu);

	BMenu *font_menu		= new BMenu("Font");
	BMenu *font_family_menu	= new BMenu("Family");
	font_family_menu->SetRadioMode(true);
	font_menu->AddItem(font_family_menu);

	BMenu *font_size_menu	= new BMenu("Size");
	font_size_menu->SetRadioMode(true);
	font_menu->AddItem(font_size_menu);
	menu_bar->AddItem(font_menu);

	// Populate the font menus...
	BFont default_font(be_plain_font);
	font_family default_family;
	default_font.GetFamilyAndStyle(&default_family, NULL);

	int32 num_families = count_font_families();
	for (int32 i=0; i<num_families; i++) {
		font_family family;
		uint32 flags;
		if (get_font_family(i, &family, &flags) == B_OK) {
			BMessage *msg = new BMessage(MSG_FONT_FAMILY);
			msg->AddString("family", family);
			BMenuItem *new_item = new BMenuItem(family, msg);
			font_family_menu->	AddItem(new_item);
			if (!strcmp(family, default_family)) {
				new_item->SetMarked(true);	// Mark current font family...
			}
		}
	}
	for (int i=10; i<25; i++) {
		char buffer[20];
		sprintf(buffer, "%d", i);
		BMessage *msg = new BMessage(MSG_FONT_SIZE);
		msg->AddInt32("size", i);
		BMenuItem *new_item = new BMenuItem(buffer, msg);
		font_size_menu->AddItem(new_item);
		if (i == default_font.Size()) {			// Mark current font size...
			new_item->SetMarked(true);
		}
	}
	AddChild(menu_bar);

	// Add a mini-view for status line...
	float status_height = menu_bar->Bounds().Height() + 1;
	BRect status_rect = Bounds();
	status_rect.bottom = status_height;
	status_rect.OffsetBy(0.0, status_height);
	status_view = new BStringView(status_rect, "Status", "", B_FOLLOW_NONE, B_WILL_DRAW);
	AddChild(status_view);

	// Make a listview with scrollbars...
	float menu_bar_height = menu_bar->Bounds().Height() + 1;
	view_rect.top += menu_bar_height;
	view_rect.top += menu_bar_height;
	text_view = new my_view(view_rect, (const char *) "List", this);

	BScrollView *scrollers = new BScrollView(
			"Scrollers",
			text_view,				// Child view
			B_FOLLOW_ALL,
			B_WILL_DRAW|B_FRAME_EVENTS,
			false, true,
			B_FANCY_BORDER);
	AddChild(scrollers);
	text_view->MakeFocus();

	// Set display fonts to defaults...
	status_font = BFont(be_bold_font);
	status_view->SetFont(&status_font);
	text_font = BFont(be_plain_font);

	BFont def;
	text_view->GetFont(&def);
	font_family family;
	text_font.GetFamilyAndStyle(&family, NULL);
	//cout << "family = " << be_plain_font << endl;;
	//cout << "size = " << def.Size() << endl;

	// Handle private parts...
	the_thread		= 0;
	file_requester	= 0;

	// Display everything...
	Show();

	// Jump straight in if user specified game datafile
	if (title) {
		run_jzip(title);
	}
}


static int my_argc;
static char *my_argv[2];

//
// Thread spawning functions...
//
void my_window::run_jzip(char *name)
{
	my_argc = 2;
	//char **test = new *char [1];
	my_argv[0] = new char [10];
	my_argv[0][0] = 0;
	my_argv[1] = new char [strlen(name) + 1];
	strcpy(my_argv[1], name);

	// Stop current thread is any...
	if (the_thread) {
		kill_thread(the_thread);
	}

	the_thread = spawn_thread(thread_func, "BeJZip", B_NORMAL_PRIORITY, this);
	resume_thread(the_thread);
}

long my_window::thread_func(void *  /*obj*/ )
{
	old_main(my_argc, my_argv);
	return 0;
}


//
// Add a line to the bottom of the list, removing the top item if there
//  are already MAX_LINES displayed.
// On entry		: 'text' is what's to be added
//			  'input_offset' is how far into a row to insert text
//			 'replace' removes last line if true
//
void my_window::add_line(const char *text, bool replace)
{
	Lock();
	// If I leave out the next line, any changes I make to the font family/size *after*
	//  starting a game are not permanent 8^/  No idea why.
	text_view->SetFontAndColor(&text_font);

	// Aw! This doesn't seem to work
	if (text_view->CountLines() > MAX_LINES) {
		int32 start	= text_view->OffsetAt(0);
		int32 end	= text_view->OffsetAt(1);
		//text_view->Delete(start, end);
	}

	// Are we dealing with user editing?
	if (replace) {
		int32 start	= input_offset + text_view->OffsetAt(text_view->CountLines()-1);
		int32 end	= strlen(text_view->Text());
		if (text_view->ByteAt(start) == '>') {
			start++;
		}
		text_view->Delete(start, end);
	}
	text_view->ScrollToOffset(text_view->OffsetAt(text_view->CountLines()));	// Scroll up a line
	text_view->Insert(text);
	text_view->ScrollToOffset(text_view->OffsetAt(text_view->CountLines()));	// Scroll up a line

	Unlock();
}

void my_window::update_status(const char *text)
{
	Lock();
//	status_view->SetFont(&status_font);
	status_view->SetText(text);
	Unlock();
}

void my_window::MessageReceived(BMessage *message)
{
	switch (message->what) {
		case MSG_ABOUT:
			{
			char str[256];
			sprintf(str, "                           BeJZip                   \n"
				     	 "                   JZip by John Holder              \n"
				     	 "             BeOS front end by J. Belson            ");
			BAlert *the_alert = new BAlert("", str, "OK");
			the_alert->Go();
			}
			break;
		case MSG_NEW:
			do_new_game_requester();
			break;
		case MSG_LOAD:
			cout << "Load saved game" << endl;
			break;
		case MSG_SAVE:
			cout << "Save game" << endl;
			break;
		case MSG_QUIT:
			be_app->PostMessage(B_QUIT_REQUESTED);
			break;
		case MSG_FONT_FAMILY:
			const char *name;
			if (message->FindString("family", &name) == B_NO_ERROR) {
				if (name) {
					font_family family;
					strcpy(family, name);
					text_font.SetFamilyAndStyle(family, NULL);
					text_view->SetFontAndColor(&text_font);
					text_view->Invalidate();
				} else {
					cout << "Font had no name??" << endl;
				}
			}
			break;
		case MSG_FONT_SIZE:
			{
			int32 size = 0;
			if (message->FindInt32("size", &size) == B_NO_ERROR) {
				if (size) {
					text_font.SetSize(float (size));
					text_view->SetFontAndColor(&text_font);
					text_view->Invalidate();
				} else {
					cout << "Got font size zero??" << endl;
				}
			}
			}
			break;
		case MSG_PICKED_FILE:
			handle_selected_file(message);
			break;
		case MSG_PAUSE_ZIP:
			// Initialise text collection...
			input_count = 0;
			input_buffer[0] = '\0';
			break;
		case MSG_RESUME_ZIP:
			//cout << "Resume jzip message" << endl;
			break;
		case MSG_KEY_PRESSED:
			handle_key_pressed(message);
			break;
		case MSG_KEY_RELEASED:
			break;
		default:
			BWindow::MessageReceived(message);
			break;
	}
}


//
// Handle KEYDOWN message from our TextView
//
void my_window::handle_key_pressed(BMessage *message)
{
	if (sleep_type == I_WANT_CHAR) {
		int8 key;
		message->FindInt8("key", &key);
		input_buffer[0] = key;
		sleep_type = I_WANT_NOWT;
		resume_thread(the_thread);
	} else if (sleep_type == I_WANT_LINE) {
		int8 key;
		message->FindInt8("key", &key);
		switch (key) {
			case B_RETURN:
				input_buffer[input_count] = '\0';
				sleep_type = I_WANT_NOWT;
				resume_thread(the_thread);
				break;
			case B_DELETE: case B_BACKSPACE:
				if (input_count > 0) {
					input_buffer[--input_count] = '\0';
				}
				break;
			default:
				if (input_count < MAX_INPUT_LEN) {
					input_buffer[input_count++] = key;
					input_buffer[input_count] = '\0';
				}
				break;
		}
		// Display current buffer contents...
		win->add_line(input_buffer, 1);
	}
}


//
// Show game select requester...
//
void my_window::do_new_game_requester(void)
{
	if (file_requester) {
		file_requester->Show();
	} else {
		file_requester = new BFilePanel(
				B_OPEN_PANEL,
				NULL,
				NULL,
				B_FILE_NODE,
				false,		// Multiple selection
				new BMessage(MSG_PICKED_FILE),
				NULL,		// Filter
				false,		// Modal
				true);		// Hide when done
		file_requester->Window()->SetTitle("Select game...");
		file_requester->SetPanelDirectory(".");
		file_requester->SetTarget(this);
		file_requester->Show();
	}
}


//
// Extract filename from received message
//
void my_window::handle_selected_file(BMessage *message)
{
	entry_ref ref;
	BEntry entry;
	entry_ref open_dir;
	BPath my_path;
	if (message->HasRef("refs")) {
		char dirname[512];
		message->FindRef("refs", &ref);
		entry.SetTo(&ref);			// Get filename..
		entry.GetParent(&entry);
		entry.GetRef(&open_dir);
		entry.GetPath(&my_path);		// Get path..
		if ((strlen(my_path.Path()) + 1 + strlen(ref.name)) < sizeof(dirname)) {
			strcpy(dirname, my_path.Path());
			strcat(dirname, "/");
			strcat(dirname, ref.name);
			in_status_window = false;
			buffer_idx = 0;
//			status_idx = 0;
			
			run_jzip(dirname);		// Try to run the game...
		} else {
			puts("Error - dirpath name too long");
		}
	}
}



bool my_window::QuitRequested(void)
{
	be_app->PostMessage(B_QUIT_REQUESTED);
	return true;
}



//
// Signal main thread to start collecting input from user, then suspend.
//  When a line of text has been collected, the main thread restarts the jzip thread.
//
int my_window::input_line(int buflen, char *buffer, int /*timeout*/, int *read_size)
{
	// Announce that we're about to about to sleep until
	//  a line of text is available...
	sleep_type = I_WANT_LINE;
	PostMessage(new BMessage(MSG_PAUSE_ZIP), this);

	// Sleep until a line has been entered...
	suspend_thread(the_thread);

	// Start new line
	add_line("\n");

	strncpy(buffer, input_buffer, (buflen < input_count) ? buflen : input_count);

	*read_size += (buflen < input_count) ? buflen : input_count;

	return '\n';
}



//
// Signal main thread to put us to sleep until a key has been
//  pressed, then wake us up.
//
int my_window::input_character(int /*timeout*/)
{
	sleep_type = I_WANT_CHAR;
	PostMessage(new BMessage(MSG_PAUSE_ZIP), this);

	// Sleep until a key has been pressed...
	suspend_thread(the_thread);

	// <yawn>
	return input_buffer[0];
}



//
// Update cursor position, only used for status bar
//
void my_window::move_cursor(int row, int col)
{
	status_idx	= col;
	x_pos		= col;
	y_pos		= row;
}



//
// Display output buffer and status buffer
//
void Be_scroll_line(void)
{
	output_buffer[buffer_idx++] = '\n';
	output_buffer[buffer_idx] = '\0';

	if (!in_status_window) {
		win->add_line(output_buffer);
		output_buffer[0] = '\0';
		buffer_idx = 0;
	} else {
		win->update_status(status_buffer);
//		status_buffer[0] = '\0';
		in_status_window = false;
	}
}


//
// Write character to a buffer for later display
//
void Be_display_char(int c)
{
	if (in_status_window) {
		if (status_idx < BUFFER_LEN) {
			status_buffer[status_idx++] = c;
		}
	} else {
		if (buffer_idx < BUFFER_LEN) {
			output_buffer[buffer_idx++] = c;
		}
	}
}



//
// Display whatever text we have, then
//  switch to status window
//
void Be_select_status_window(void)
{
	in_status_window = true;

	// Display whatever's left in output buffer
	output_buffer[buffer_idx] = '\0';
	win->add_line(output_buffer);
	output_buffer[0] = '\0';
	buffer_idx = 0;
}

//
// Display whatever text we have, then
//  switch to text window
//
void Be_select_text_window(void)
{
	in_status_window = false;

	// Display whatever's left in status buffer
//	status_buffer[status_idx] = '\0';
	win->update_status(status_buffer);
//	status_buffer[0] = '\0';
//	status_idx = 0;
}


//
// Clear all text from window
//
void Be_clear_screen(void)
{
	if (win) {
		win->clear_text();
	}
}


//
// Allow user to input a line of text
// On exit	: -1 if timeout occurred
//		 '\n' if return pressed
//
int Be_input_line(int buflen, char *buffer, int timeout, int *read_size)
{
	if (*read_size!=0) {
		cout << "*read_size != 0" << endl;
	}
	if (timeout != 0) {
		cout << "Ignoring timeout 8^(" << endl;
	}

	// Take note of where typed characters should appear
	input_offset = buffer_idx;

	// Dump current contents of output buffer...
	if (buffer_idx) {
		char buf[BUFFER_LEN];
		strncpy(buf, output_buffer, buffer_idx);
		buf[buffer_idx] = '\0';
		win->add_line(buf);
		output_buffer[0] = 0;
		input_offset = buffer_idx;
		buffer_idx = 0;
	}
	return win->input_line(buflen, buffer, timeout, read_size);
}


//
// Wait up to timeout seconds for a keypress
//
int Be_input_character(int timeout)
{
	// Ignore [MORE] prompts, we have scrollback 8^)
	if (strcmp(output_buffer, "[MORE]")) {
		output_buffer[0] = '\0';
		buffer_idx = 0;
		return '\n';
	}

	return win->input_character(timeout);
}


//
// Move current `cursor` position, used for printing in the status bar
//
void Be_move_cursor(int row, int col)
{
	win->move_cursor(row, col);
}


