/*********************************************************************
HyperTTT - Tic tac toe in higher dimensions
Copyright (c) 1998 Brian Nenninger

This program is free software; you can redistribute it and/or modify it under 
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 2 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
PARTICULAR PURPOSE. See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along with 
this program; if not, write to the Free Software Foundation, Inc., 675 Mass 
Ave., Cambridge, MA 02139, USA.

You may contact me at bwn@kreative.net
************************************************************************/


#include "4dBoard.h"

extern int N, D;

// ThreatLine *v1;
//Board scratchBoard;

// int created, destroyed;

///////////////////////////////////////////////////////////////////////////////////
Vector::Vector()
{ // allocates memory for vector
//  x = new signed char[D];
//  ++created;
};

Vector::~Vector()
{ // frees memory
//  delete [] x;
//  ++destroyed;
/*  if (this==(Vector *)v1)
  {
    SysBeep(10);
  } */
};

void Vector::Set(int n, int val)
{ // sets nth element of vector to val
  x[n] = (signed char) val;
};

void Vector::Set0()
{ // sets vector to all 0's
  int i;
  for(i=0; i<D; i++)
    x[i] = 0;
};

int Vector::Get(int n)
{ // gets nth element of vector
  return x[n];
};

int Vector::operator[](int n)
{ // same as above, allows v[i] instead of v.Get(i)
  return x[n];
};

void Vector::Inc()
{ // adds 1 to rightmost element of vector, carries if neccesary
  int i;
  i=D-1;
  x[i]++;
  while (x[i]>=N && i>0)
  { // carry until no overflow or until end of vector
    x[i]=0;
    x[--i]++;
  }
};

void Vector::operator++()
{ // same as above
  Inc();
};

void Vector::operator++(int n)
{ // same as above
  Inc();
};

Vector Vector::operator+(Vector &v)
{ // returns sum of two vectors
  int i;
  Vector ans;
  for(i=0; i<D; i++)
    ans.Set(i, x[i]+v[i]);
  return ans;
};

Vector Vector::operator-(Vector &v)
{ // returns difference of two vectors
  int i;
  Vector ans;
  for(i=0; i<D; i++)
    ans.Set(i, x[i]-v[i]);
  return ans;
};

Vector Vector::operator-()
{ // returns negative of vector
  int i;
  Vector ans;
  for(i=0; i<D; i++)
    ans.Set(i, -x[i]);
  return ans;
};

void Vector::operator+=(Vector &v)
{ // increments vector by another vector
  int i;
  for(i=0; i<D; i++)
    x[i] += v[i];
};

void Vector::operator-=(Vector &v)
{ // decrements vector by another vector
  int i;
  for(i=0; i<D; i++)
    x[i] -= v[i];
};

Vector Vector::operator*(int k)
{ // returns product of vector and scalar
  int i;
  Vector ans;
  for(i=0; i<D; i++)
   // ans.Set(i, k*x[i]);
   ans.x[i] = k*x[i];
  return ans;
};

Vector operator*(int k, Vector &v)
{ // returns product of vector and scalar
  int i;
  Vector ans;
  for(i=0; i<D; i++)
    ans.Set(i, k*v.Get(i));
  return ans;
};

/*
Vector Vector::operator+(DirVec &dv)
{ // returns sum of vector and direction vector
  int i;
  Vector ans;
  for(i=0; i<D; i++)
    ans.Set(i, x[i]+dv[i]);
  return ans;
};

Vector Vector::operator-(DirVec &dv)
{ // returns difference of vector and direction vector
  int i;
  Vector ans;
  for(i=0; i<D; i++)
    ans.Set(i, x[i]-dv[i]);
  return ans;
};
*/
Vector Vector::operator=(Vector &v)
{ // copies contents of v
  int i;
  for(i=0; i<D; i++)
    x[i] = v.x[i];
  return *this;
};

int Vector::operator==(Vector &v)
{ // returns 1 if all elements equal to those of v
  int i, good=1;
  for(i=0; i<D && good; i++)
    if (x[i] != v.Get(i))
      good = 0;
  return good;
}

int Vector::IsMax()
{ // returns true if vector is equal to <N-1,N-1,N-1...,N-1>
  int i, good=1;
  for(i=0; i<D && good; i++)
    if (x[i]!=N-1)
      good=0;
  return good;
}

int Vector::IsEdge()
{ // returns true if vector represents a cell on at least one edge on the board
  // true if at least one element equals 0 or N-1
  int i;
  for(i=0; i<D; i++)
    if (x[i]==0 || x[i]==N-1)
      return 1;
  return 0;
};

int Vector::IsLegal()
{ // returns true if vector is within board boundaries(all elements between 0 and N-1)
  int i;
  for(i=0; i<D; i++)
    if (x[i]<0 || x[i]>=N)
      return 0;
  return 1;
};

int Vector::Normalize()
{ // returns true if vector is nonzero multiple of vector containing only -1, 0, and 1
  // if true, changes all pos. elements to 1 and neg. elements to -1
  int i, good=1, non0val=0;
  for(i=0; i<D && good; i++)
    if (x[i]!=0)
      if (non0val==0)
        non0val=x[i];
      else if (x[i]!=non0val && x[i]!=-non0val)
        good=0;
  if (good && non0val!=0)
  { // normalize to -1,0,1
    for(i=0; i<D; i++)
      x[i] = (x[i]>0 ? 1 : (x[i]==0 ? 0 : -1));
    return 1;
  }
  return 0;
};

int Vector::LegalDir(Vector &dv)
{ // returns true if ThreatLine from v to v+(N-1)dv is a valid path
  return (IsLegal() && (*this + ((N-1)*dv)).IsLegal());
};

///////////////////////////////////////////////////////////////////////////////////
/*
DirVec::DirVec()
{ // allocate memory(3^4=81<256, so 4 members per byte)
  x = new unsigned char[1+(D-1)/4];
};

DirVec::~DirVec()
{ // release memory
  delete [] x;
};

void DirVec::Set(int n, int val)
{ // sets nth element of DirVec to val, -1<=val<=1
  int i, mask=255;
  // clear proper 2 bits of mask
  mask ^= ((1 << n) + (1 << (n+1)));
  x[n/4] &= mask;
  // store val in bits just cleared(actually store val+1)
  x[n/4] += (1 << (2*n))*(val+1);
};

int DirVec::Get(int n)
{ // gets nth element of DirVec
  int mask=0, val;
  // use mask to isolate the 2 bits
  mask += ((1 << (2*n)) + (1 << (2*n+1)));
  val = (x[n/4] & mask)>>(2*n);
  return val-1;
};

int DirVec::operator[](int n)
{ // same as above
  return Get(n);
};

DirVec DirVec::operator=(DirVec &dv)
{ // copies contents of v
  int i;
  for(i=0; i<1+(D-1)/4; i++)
    x[i] = dv.x[i];
  return *this;
};
*/
//////////////////////////////////////////////////////////////////////////////////////////

Board::Board()
{ // this is icky, N**D cells, 4 cells per byte
  int i;
  for(i=0, numCells=1; i<D; i++)
    numCells*=N;
  size = 1+(numCells-1)/4;
  x = new unsigned char[size];
};

Board::~Board()
{ // release memory
  delete [] x;
};

void Board::Set(Vector &v, int val)
{ // sets cell described by v to val(0<=val<=3)
  int i, index, placeVal, bitNum, mask;
  for(i=D-1, index=0, placeVal=1; i>=0; i--)
  {
    index += v[i]*placeVal;
    placeVal*=N;
  }
  // set 2 bits of x[index/4]
  bitNum = 2*(index%4);
  mask = 255^((1 << bitNum) + (1 << (bitNum+1)));
  // clear the 2 bits and store new value
  x[index/4] &= mask;
  x[index/4] += val*(1 << bitNum);
};

void Board::Set0()
{ // sets all cells to 0
  int i;
  for(i=0; i<size; i++)
    x[i] = 0;
}; 

int Board::IsEmpty()
{ // returns 1 if all cells are empty(equal to 0)
  int i;
  for(i=0; i<size; i++)
    if (x[i]!=0)
      return 0;
  return 1;
};

int Board::Get(Vector &v)
{ // sets cell described by v to val(0<=val<=3)
  int i, index, placeVal, bitNum, mask;
  for(i=D-1, index=0, placeVal=1; i>=0; i--)
  {
    index += v[i]*placeVal;
    placeVal*=N;
  }
  // get 2 bits of x[index/4]
  bitNum = 2*(index%4);
  mask = (1 << bitNum) + (1 << (bitNum+1));
  return (x[index/4] & mask) >> bitNum;
};

int Board::operator[](Vector &v)
{ // same as above
  return Get(v);
};

int Board::ThreatLen(Vector &v, Vector &dv, int threatVal)
{ // returns number of cells in path determined by v and v+dv containing threatVal
  // returns 0 if a cell in the path contains a nonzero value other than threatVal(already blocked)
  int i, len, val;
  for(i=len=0; i<N; i++)
  {
    val = Get(v + (i*dv));
    if (val!=0 && val!=threatVal) // path is blocked
      return 0;
    if (val==threatVal) // another mark in the path
      ++len;
  }
  return len;
};

void Board::GetThreats(Vector &threatPos, int threatVal, 
                        VectorPtr &threatList, int *listSize, int *threatSize)
{ // creates list of unblocked paths with value threatVal passing through threatPos
  int len, maxLen, i, done, numThreats, tempVal;
  Vector v, dv;
  static Vector threats[170]; // almost 1K
  Board temp;
  v.Set0();
  temp.Set0();
  // assume threatPos is filled
  tempVal = Get(threatPos);
  Set(threatPos, threatVal);
  maxLen = done = numThreats = 0;
  while (!done)
  { // go through every cell
    if (v.IsEdge()) // if it's an edge
      if (temp[v]==0) // and a previous path hasn't ended on it
      { // get displacement to threatVal
        dv = threatPos - v;
        if (dv.Normalize() && v.LegalDir(dv)) // if there is a path from v to threatVal
        {
          // determine existence and magnitude of threat along path defined by v and v+u
          len = ThreatLen(v, dv, threatVal);
          temp.Set(v, 1);
          temp.Set(v+((N-1)*dv), 1);
          if (len > maxLen)
          { // new maximum threat
            maxLen = len;
          }
          if (len > 0) 
          { // valid threat
            threats[numThreats] = v;
            numThreats++;
          }
        }
      }
    // move to next cell unless done
    if (v.IsMax())
      done=1;
    else
      v++;      
  }
  // create list of max threats
  *listSize = numThreats;
  if (numThreats==0)
    threatList = NULL;
  else
  {
	  threatList = new Vector[numThreats];
	  for(i=0; i<numThreats; i++)
	    threatList[i] = threats[i];
	}
	*threatSize = maxLen;
	// restore board to original state
	Set(threatPos, tempVal);
}

///////////////////////////////////////////////////////////////////////////////////

void ThreatLine::operator=(ThreatLine &t)
{
  v  = t.v;
  dv = t.dv;
};

int ThreatLine::operator==(ThreatLine &t)
{ // returns true if two lines cover same cells
  if (v==t.v && dv==t.dv)
    return 1; // lines are identical
  if (v+((N-1)*dv) == t.v && dv == -t.dv)
    return 1; // lines cover same cells in opposite directions
  return 0;
};

int ThreatLine::HitsCell(Vector &target)
{ // returns true if target is along ThreatLine
  Vector diff;
  int i, mag=0;
  if (target == v)
    return 1; // target is initial point of ThreatLine
  diff = v - target;
  // get 1st nonzero of difference vector
  for(i=0; !(mag=diff[i]); i++);
//  if (mag<0)
//    mag = -mag;
  // compare diff to corresponding point on ThreatLine
  if (diff == mag*dv || diff == -mag*dv)
    return 1;
  else return 0; 
};


ThreatList::ThreatList()
{
  lineList = NULL;
  maxThreatLen = numMaxThreats = 0;
};

ThreatList::~ThreatList()
{
  if (lineList)
    delete [] lineList;
};

Player::Player()
{
  winLines.listLen = loseLines.listLen = 0;
}

void Player::ComputeBestDefenseMove(Board &b, int val, Vector *move)
{
  ThreatLine path, path2;
  int linesBlocked, maxLB, numNewThreats, maxNT, newThreatLen, maxTL, i, j, k;
  Vector v;
  VectorPtr newThreatList=NULL;
  Board scratchBoard;
  int numSamePower;

  maxLB = 0;
  scratchBoard.Set0();
  for(i=0; i<loseLines.numMaxThreats; i++)
  {
    path = loseLines.lineList[i];
    // try every empty cell in the ThreatLine
    for(k=0, v=path.v; k<N; k++, v+=path.dv)
    {
//      bval = b[v]; sval = scratchBoard[v];
      if (b[v]==0 && scratchBoard[v]==0)
      {
        scratchBoard.Set(v,1); // mark cell as checked
        // count number of threats this blocks(will always be >0)
        for(j=linesBlocked=0; j<loseLines.numMaxThreats; j++)
          if (loseLines.lineList[j].HitsCell(v))
            ++linesBlocked;
        if (linesBlocked > maxLB)
        { // best defensive move so far; calculate offensive power
          b.GetThreats(v, val, newThreatList, &numNewThreats, &newThreatLen);
          if (newThreatList != NULL)
            delete [] newThreatList;
          *move = v; // current best move
          maxLB = linesBlocked;
          maxTL = newThreatLen;
          maxNT = numNewThreats;
        } 
        else if (linesBlocked == maxLB)
        {
          // compare offensive power of this move to previous best
          b.GetThreats(v, val, newThreatList, &numNewThreats, &newThreatLen);
          if (newThreatList != NULL)
            delete [] newThreatList;
          if ((newThreatLen > maxTL) || (newThreatLen==maxTL && numNewThreats > maxNT))
          {
            *move = v; // current best move
	          maxTL = newThreatLen;
	          maxNT = numNewThreats;
	          numSamePower = 1;
          }
          else if (newThreatLen==maxTL && numNewThreats==maxNT)
          { // this move has the same strength as the previous best, so use Randomizer to
            // choose whether to accept this move or retain previous move
            numSamePower++;
            if (Rand(numSamePower)==0)
            {
            //SysBeep(1);
	            *move = v; // current best move
	            maxLB = linesBlocked;
	            numSamePower = 1;
            }
          }
        } // end (linesBlocked == maxLB)     
      }
    }
  }
}

void Player::ComputeBestOffenseMove(Board &b, int val, Vector *move)
{ 
  ThreatLine path, path2;
  int linesBlocked, maxLB, numNewThreats, maxNT, i, j, k;
  Vector v, *newThreatList=NULL;
  Board scratchBoard;
  int numSamePower;

  maxNT = 0;
  scratchBoard.Set0();
  for(i=0; i<winLines.numMaxThreats; i++)
  {
    path = winLines.lineList[i];
    // try every empty cell in the ThreatLine
    for(j=0, v=path.v; j<N; j++, v+=path.dv)
    {
      if (b[v]==0 && scratchBoard[v]==0)
      {
        scratchBoard.Set(v, 1);
        // count number of attacks this adds to(will always be >0)
        for(k=numNewThreats=0; k<winLines.numMaxThreats; k++)
          if (winLines.lineList[k].HitsCell(v))
            ++numNewThreats;
        if (numNewThreats > maxNT)
        { // best offensive move so far; calculate defensive power
	        for(k=linesBlocked=0; k<loseLines.numMaxThreats; k++)
	          if (loseLines.lineList[k].HitsCell(v))
	            ++linesBlocked;
          *move = v; // current best move
          maxNT = numNewThreats;
          maxLB = linesBlocked;
        } 
        else if (numNewThreats == maxNT)
        {
          // compare defensive power of this move to previous best
	        for(k=linesBlocked=0; k<loseLines.numMaxThreats; k++)
	          if (loseLines.lineList[k].HitsCell(v))
	            ++linesBlocked;
          if (linesBlocked > maxLB)
          {
            *move = v; // current best move
            maxLB = linesBlocked;
            numSamePower = 1;
          }
          else if (linesBlocked==maxLB)
          { // this move has the same strength as the previous best, so use randomizer to
            // choose whether to accept this move or retain previous move
            numSamePower++;
            if (Rand(numSamePower)==0)
            {
            //SysBeep(9);
	            *move = v; // current best move
	            maxLB = linesBlocked;
	            numSamePower = 1;
            }
          }
        } // end (numNewThreats == maxNT)       
      }
    }
  } 
}

int Rand(int n)
{
  return (abs(rand()) % n);
  //return n-1;
}

int Player::BalancedMove(Board &b, int val, Vector *move)
{ // stores best play in move
  ThreatLine path, path2;
  int i;
  // check for win
  if (winLines.maxThreatLen == N-1)
  {
    path = winLines.lineList[0];
    for(*move = path.v; b[*move] != EMPTY; *move += path.dv); // finish the line
    return WIN;
  }                                                                                                                                                                                                                                                                                                                                                                                                             
  else if (winLines.maxThreatLen == 0 && loseLines.maxThreatLen == 0)
  { // nobody has any existing threats; the game is going to be a tie, so move randomly
    // get random vector
    for(i=0; i<D; i++)
      move->Set(i, Rand(N));
    while (b[*move]!=0)
    { // increment vector until empty space found
      (*move)++;
      if (!(move->IsLegal())) // "wrap around" if the vector gets out of range
        move->Set0();
    }
    return TIE;
  }
  else if (winLines.maxThreatLen >= loseLines.maxThreatLen)
  { // longer offensive lines than threats; make best offensive move    
//    SysBeep(100);    
    ComputeBestOffenseMove(b, val, move);
    return 0;
  }
  else
  { // make best defensive move
    ComputeBestDefenseMove(b, val, move);
    return 0;
  }
};


int Player::AttackMove(Board &b, int val, Vector *move)
{ // stores best play in move
  ThreatLine path, path2;
  int i;
  // check for win
  if (winLines.maxThreatLen == N-1)
  { // Win!
    path = winLines.lineList[0];
    for(*move = path.v; b[*move] != EMPTY; *move += path.dv); // finish the line
    return WIN;
  }
  else if (loseLines.maxThreatLen == N-1)
  { // Don't lose
    path = loseLines.lineList[0];
    for(*move = path.v; b[*move] != EMPTY; *move += path.dv); // finish the line
    return 0;  
  }                                                                                                                                                                                                                                                                                                                                                                                                            
  else if (winLines.maxThreatLen == 0 && loseLines.maxThreatLen == 0)
  { // nobody has any existing threats; the game is going to be a tie, so move randomly
    // get random vector
    for(i=0; i<D; i++)
      move->Set(i, Rand(N));
    while (b[*move]!=0)
    { // increment vector until empty space found
      (*move)++;
      if (!(move->IsLegal())) // "wrap around" if the vector gets out of range
        move->Set0();
    }
    return TIE;
  }
  else if (winLines.maxThreatLen > 0)
  { // attack if have any existing threats
//    SysBeep(100);    
    ComputeBestOffenseMove(b, val, move);
    return 0;
  }
  else
  { // make best defensive move
    ComputeBestDefenseMove(b, val, move);
    return 0;
  }
};


int Player::DefendMove(Board &b, int val, Vector *move)
{ // stores best play in move
  ThreatLine path, path2;
  int i;
  // check for win
  if (winLines.maxThreatLen == N-1)
  {
    path = winLines.lineList[0];
    for(*move = path.v; b[*move] != EMPTY; *move += path.dv); // finish the line
    return WIN;
  }                                                                                                                                                                                                                                                                                                                                                                                                             
  else if (winLines.maxThreatLen == 0 && loseLines.maxThreatLen == 0)
  { // nobody has any existing threats; the game is going to be a tie, so move randomly
    // get random vector
    for(i=0; i<D; i++)
      move->Set(i, Rand(N));
    while (b[*move]!=0)
    { // increment vector until empty space found
      (*move)++;
      if (!(move->IsLegal())) // "wrap around" if the vector gets out of range
        move->Set0();
    }
    return TIE;
  }
  else if (loseLines.maxThreatLen == 0)
  { // no enemy threats; make best offensive move    
//    SysBeep(100);    
    ComputeBestOffenseMove(b, val, move);
    return 0;
  }
  else
  { // make best defensive move
    ComputeBestDefenseMove(b, val, move);
    return 0;
  }
};

void Player::MakeFirstMove(Board &b, Vector *move)
{
  int i;
  if (Rand(2)==0)
  { // choose a random corner
    for(i=0; i<D; i++)
      move->Set(i, (Rand(2)==0) ? 0 : N-1);
  }
  else
  { // choose a random center square(only 1 possible if odd # of squares)
    if (N%2==1)
      for(i=0; i<D; i++) // exact center
        move->Set(i, N/2);
    else
      for(i=0; i<D; i++) // no exact center; choose a mostly centered square
        move->Set(i, (Rand(2)==0) ? N/2 : N/2-1);
  }
}

int Player::ComputeBestMove(Board &b, int val, Vector *move)
{
  if (b.IsEmpty())
  {
    MakeFirstMove(b, move);
    return 0;
  }
  else
  {
	  switch (strategy)
	  {
	    case DEFEND : return DefendMove(b, val, move);
	    case ATTACK : return AttackMove(b, val, move);
	    default : return BalancedMove(b, val, move);
	  }
	}
};


void SwapUC(unsigned char *a, unsigned char *b)
{
  unsigned char c;
  c  = *a;
  *a = *b;
  *b = c;
};

void SwapTL(ThreatLine *a, ThreatLine *b)
{
  ThreatLine c;
  c  = *a;
  *a = *b;
  *b = c;
};

void SortThreatList(unsigned char *lenList, ThreatLine *threatList, int len)
{ // radix sort arrays using lenList as keys
  unsigned char *count;
  int i, index, added, j;
  count = new unsigned char[N];
  for(i=0; i<N; i++)
    count[i] = 0;
  // count number of elements with each possible value(1 to N, so use lenList[i]-1)
  for(i=0; i<len; i++)
    ++(count[lenList[i] - 1]);
  for(index=0, i=N; i>0; i--)
  {
    for(added=0; added<count[i-1]; added++)
    { // find next element with length i
      for(j=index; lenList[j]!=i; j++);
      if (j!=index)
      {
        SwapUC(lenList+j, lenList+index);
        SwapTL(threatList+j, threatList+index);
      }
      index++;
    }
  }
  delete [] count;
}

int Player::AddNewThreats(Board &b, ThreatList &tList, int val, Vector &move)
{
  Vector temp;
  VectorPtr newThreatList;
  int newThreatListLen, newThreatLen, found, numRemoved, len, i, j, index;
  ThreatLine *newLineList, *combinedList;
  unsigned char *combinedLenList;
  
  b.GetThreats(move, val, newThreatList, &newThreatListLen, &newThreatLen);
  // create new offensive threat list by combining new and old threats
  // remove all existing threats that are in new threat list
  newLineList = new ThreatLine[newThreatListLen];
  for(i=0; i<newThreatListLen; i++) // create ThreatList objects from vector list
  {
    newLineList[i].v  = newThreatList[i];
    temp = move - newThreatList[i];
    temp.Normalize();
    newLineList[i].dv = temp;
  }
  // remove duplicates from original threat list
  for(numRemoved=i=0; i<newThreatListLen; i++)
  {
    for(found=j=0; j<tList.listLen && tList.lenList[j]>0 && !found; j++)
      if (newLineList[i] == tList.lineList[j])
      {
        found = 1;
        ++numRemoved;
        tList.lenList[j] = 0; // mark line as duplicated
      }
  }
  // merge old and new threats into single list
  len = tList.listLen + newThreatListLen - numRemoved;
  combinedList = new ThreatLine[len];
  combinedLenList = new unsigned char[len];
  // copy to new list, skipping duplicates in old list
  for(i=0; i<newThreatListLen; i++)
  {
    combinedList[i] = newLineList[i];
    // determine length of threat
    combinedLenList[i] = b.ThreatLen(combinedList[i].v, combinedList[i].dv, val);
  }
  for(index=newThreatListLen, i=0; index<len; index++, i++)
  {
    while(tList.lenList[i]==0) // get next nonduplicate line
      i++;
    combinedList[index] = tList.lineList[i];
    combinedLenList[index] = tList.lenList[i];
  }
  // release memory and sort new threat list
  if (newLineList)
    delete [] newLineList;
  if (tList.listLen > 0)
  {
    delete [] tList.lineList;   
    delete [] tList.lenList;
//    v1 = NULL;
  }
  SortThreatList(combinedLenList, combinedList, len);
  tList.lineList = combinedList;
  
//  v1 = tList.lineList;
  
  tList.lenList  = combinedLenList;
  tList.listLen  = len;
  // get number of threats with longest length
  if (len>0)
  {
    for(tList.numMaxThreats=i=1; i<len; i++)
      if (tList.lenList[i] == tList.lenList[0])
        ++(tList.numMaxThreats);
    tList.maxThreatLen = tList.lenList[0];
  }
  else
  {
    tList.numMaxThreats = 0;
    tList.maxThreatLen  = 0;
  }
//  delete [] combinedList;
//  delete [] combinedLenList;
// delete [] newLineList;
  if (tList.maxThreatLen==N)
    return WIN;
  else return 0;
};


void Player::RemoveBlockedThreats(Board &b, ThreatList &tList, int val, Vector &move)
{
  ThreatLine *filterList;
  unsigned char *filterLenList;
  int i, index, removed;
  // mark all lines that have just been blocked
  for(removed=i=0; i<tList.listLen; i++)
    if (tList.lineList[i].HitsCell(move))
    {
      tList.lenList[i] = 0;
      ++removed;
    }
  // copy remaining lines to new list and free memory
  filterList = new ThreatLine[tList.listLen - removed];
  filterLenList = new unsigned char[tList.listLen - removed];
  for(index=i=0; i<tList.listLen; i++)
    if (tList.lenList[i] > 0)
    {
      filterList[index] = tList.lineList[i];
      filterLenList[index] = tList.lenList[i];
      ++index;
    }
  if (tList.listLen > 0)
  {
    delete [] tList.lineList;
    delete [] tList.lenList;
  }
  tList.lineList = filterList;
  tList.lenList  = filterLenList;
  tList.listLen -= removed;
  // get number of threats with longest length
  if (tList.listLen>0)
  {
    for(tList.numMaxThreats=i=1; i<tList.listLen && tList.lenList[i]==tList.lenList[0]; i++)
      ++(tList.numMaxThreats);
    tList.maxThreatLen = tList.lenList[0];
  }
  else
  {
    tList.numMaxThreats = 0;
    tList.maxThreatLen  = 0;
  }
}; 

int Player::MakeMove(Board &b, int val, Vector *move)
{
  int result = ComputeBestMove(b, val, move);
  b.Set(*move, val);
  AddNewThreats(b, winLines, val, *move);
  RemoveBlockedThreats(b, loseLines, val, *move);
  return result;
};

int Player::UpdateThreats(Board &b, int val, Vector *move)
{ // same as MakeMove but move is given instead of computed
  b.Set(*move, val);
  AddNewThreats(b, winLines, val, *move);
  RemoveBlockedThreats(b, loseLines, val, *move);
  if (winLines.maxThreatLen==N)
    return WIN;
  else return 0;
}

int Player::RespondToMove(Board &b, int val, Vector *move)
{
  int result = AddNewThreats(b, loseLines, val, *move);
  RemoveBlockedThreats(b, winLines, val, *move);
  return result;
};
