//Name:		TetrisWindow.cpp
//Author:	Brian Tietz
//Copyright 1999
//Conventions:
//	Global constants (declared with const) and #defines - begin with "c_" followed by lowercase
//		words separated by underscores.
//		(E.G., #define c_my_constant 5).
//		(E.G., const int c_my_constant = 5;).
//	Global variables - begin with "g_" followed by lowercase words separated by underscores.
//		(E.G., int g_my_global;).
//	New data types (classes, structs, typedefs, etc.) - begin with an uppercase letter followed by
//		lowercase words separated by uppercase letters.  Enumerated constants contain a prefix
//		associating them with a particular enumerated set.
//		(E.G., typedef int MyTypedef;).
//		(E.G., enum MyEnumConst {c_mec_one, c_mec_two};)
//	Private member variables - begin with "m_" followed by lowercase words separated by underscores.
//		(E.G., int m_my_member;).
//	Public or friend-accessible member variables - all lowercase words separated by underscores.
//		(E.G., int public_member;).
//	Argument and local variables - begin with a lowercase letter followed by
//		lowercase words separated by underscores.  If the name is already taken by a public member
//		variable, prefix with a_ or l_
//		(E.G., int my_local; int a_my_arg, int l_my_local).
//	Functions (member or global) - begin with an uppercase letter followed by lowercase words
//		separated by uppercase letters.
//		(E.G., void MyFunction(void);).
//License:
//  Obviously, the idea for Tetris isn't mine, so I hold no copyright on the idea.  I do, however,
//  reserve all rights with regard to the source code for this BeOS version of Tetris.  The executable,
//  is freely distributable.  The source code is likewise freely distributable, provided that the
//  license and copyright are retained in these source code files and in the about box of the
//  executable.


//******************************************************************************************************
//**** System header files
//******************************************************************************************************
#include <stdlib.h>
#include <MenuBar.h>
#include <MenuItem.h>
#include <Screen.h>
#include <MessageRunner.h>


//******************************************************************************************************
//**** Project header files
//******************************************************************************************************
#include "TetrisWindow.h"
#include "Tetris.h"
#include "TetrisResources.h"
#include "NewStrings.h"
#include "Colors.h"


//******************************************************************************************************
//**** TetrisWindow
//******************************************************************************************************
TetrisWindow::TetrisWindow()
: BWindow(BRect(0,0,0,0),g_tetris_string,B_TITLED_WINDOW,B_NOT_RESIZABLE|B_ASYNCHRONOUS_CONTROLS),
m_self_messenger(NULL,this)
{
	BMenuBar* menu_bar = GenerateMenuBar();
	AddChild(menu_bar);
	float menu_bar_height = menu_bar->Frame().Height();

	TetrisPrefs* prefs = ((TetrisApplication*)be_app)->Prefs();
	prefs->Lock();
	BRect window_position = prefs->WindowPosition();
	prefs->Unlock();

	uint8 grid_width = 10;
	uint8 grid_height = 20;
	m_key_left = '4';
	m_key_right = '6';
	m_key_rotate = '5';
	m_key_down = '2';
	m_key_drop = B_SPACE;

	BRect screen_limits = BScreen().Frame();
	screen_limits.InsetBy(3.0,3.0);
	if(!screen_limits.Intersects(window_position))
		window_position.OffsetTo(screen_limits.LeftTop()+BPoint(47,47));

	BFont font = *be_plain_font;
	font.SetSize(18);
	struct font_height font_ht;
	font.GetHeight(&font_ht);
	m_font_ascent = ceil(font_ht.ascent);
	float font_height = m_font_ascent + ceil(font_ht.descent);

	const char* strings[3] = {g_score_label,g_lines_label,g_next_block_label};
	float string_widths[3];
	float boxes_width = GetStringsMaxWidth(strings,3,&font,string_widths);
	float max_score_width = font.StringWidth("0000000");
	if(boxes_width < max_score_width + 10)
		boxes_width = max_score_width + 10;
	if(boxes_width < 16*3+10)
		boxes_width = 16*3+10;

	float boxes_start = 10+2+grid_width*16+2+20;
	m_score_label_rect.Set(boxes_start,10,boxes_start+string_widths[0],10+font_height);
	BRect score_rect(boxes_start,m_score_label_rect.bottom+5,boxes_start+boxes_width,
		m_score_label_rect.bottom+5+2+font_height+2);
	m_score_view = new NumberView(score_rect,&font);
	m_lines_label_rect.Set(boxes_start,score_rect.bottom+font_height,boxes_start+string_widths[1],
		score_rect.bottom+font_height+font_height);
	BRect lines_rect(boxes_start,m_lines_label_rect.bottom+5,boxes_start+boxes_width,
		m_lines_label_rect.bottom+5+2+font_height+2);
	m_lines_view = new NumberView(lines_rect,&font);
	m_next_block_label_rect.Set(boxes_start,lines_rect.bottom+font_height,boxes_start+string_widths[1],
		lines_rect.bottom+font_height+font_height);
	BRect nect_block_rect(boxes_start,m_next_block_label_rect.bottom+5,boxes_start+boxes_width,
		m_next_block_label_rect.bottom+5+2+5+16*4+5+2);
	m_next_block_view = new NextBlockView(nect_block_rect);

	float window_bottom = menu_bar_height+1+10+2+grid_height*16+1+10;
	if(window_bottom < nect_block_rect.bottom+10)
		window_bottom = nect_block_rect.bottom+10;

	m_background_color = ui_color(B_PANEL_BACKGROUND_COLOR);
	m_dark_1_color = tint_color(m_background_color,B_DARKEN_1_TINT);
	m_dark_2_color = tint_color(m_background_color,B_DARKEN_4_TINT);

	BRect view_rect(0,menu_bar_height+1,boxes_start+boxes_width+10,window_bottom);	
	m_window_background_view = new TetrisWindowView(view_rect,this);
	m_window_background_view->SetFont(&font);
	m_window_background_view->SetViewColor(m_background_color);
	m_window_background_view->AddChild(m_score_view);
	m_window_background_view->AddChild(m_lines_view);
	m_window_background_view->AddChild(m_next_block_view);

	m_tetris_view_rect.Set(10,10,10+2+grid_width*16+1,10+2+grid_height*16+1);
	m_tetris_view = new TetrisView(m_tetris_view_rect.InsetByCopy(2,2),grid_width,grid_height,&font);
	m_window_background_view->AddChild(m_tetris_view);
	
	MoveTo(window_position.LeftTop());
	ResizeTo(view_rect.Width(),menu_bar_height+view_rect.Height()+1);
	AddChild(m_window_background_view);
	m_score = 0;
	m_lines = 0;
	m_next_block_type = 0;
	m_down_timer = NULL;
	m_ignore_downs = true;
	m_user_down_timer = NULL;
	m_game_over = true;

	AddCommonFilter(new BMessageFilter(B_KEY_DOWN,KeyFilterStatic));

	Show();
}


BMenuBar* TetrisWindow::GenerateMenuBar()
{
	BMenuBar* menu_bar = new BMenuBar(BRect(0,0,0,0),"Menu");
	BMenu* menu;
	BMenuItem* item;

	//File Menu
	menu = new BMenu(g_file_string);
		//Begin new game
		menu->AddItem(new BMenuItem(g_begin_new_game_string,new BMessage(c_begin_new_game),'N'));

		//Pause game
		m_pause_item = new BMenuItem(g_pause_string,new BMessage(c_pause_game),'P');
		menu->AddItem(m_pause_item);

		#if 0
			//No preferences window needed
			item = new BMenuItem(g_preferences_string,new BMessage(c_edit_preferences));
			item->SetTarget(be_app);
			menu->AddItem(item);
		#endif

		//About
		item = new BMenuItem(g_about_string,new BMessage(B_ABOUT_REQUESTED));
		item->SetTarget(be_app);
		menu->AddItem(item);

		//Quit
		item = new BMenuItem(g_quit_string,new BMessage(B_QUIT_REQUESTED),'Q');
		item->SetTarget(be_app);
		menu->AddItem(item);

	menu_bar->AddItem(menu);
	return menu_bar;
}


TetrisWindow::~TetrisWindow()
{
	TetrisPrefs* prefs = ((TetrisApplication*)be_app)->Prefs();
	prefs->Lock();
	prefs->SetWindowPosition(Frame());
	prefs->Unlock();
	if(m_down_timer)
		delete m_down_timer;
	if(m_user_down_timer)
		delete m_user_down_timer;
	be_app->PostMessage(B_QUIT_REQUESTED);
}


void TetrisWindow::MessageReceived(BMessage *message)
{
	switch(message->what)
	{
		case c_begin_new_game:
		{
			srand((int)real_time_clock_usecs());
			m_tetris_view->Reset();
			m_score = 0;
			m_lines = 0;
			m_next_block_type = GetRandomBlock();
			m_score_view->SetNumber(m_score);
			m_lines_view->SetNumber(m_lines);
			m_next_block_view->SetType(m_next_block_type);
			m_tetris_view->AddFallingBlock(GetRandomBlock());
			m_down_interval = c_initial_down_interval;
			m_lines_until_next_speed_up = c_lines_between_speedups;
			if(m_down_timer)
				delete m_down_timer;
			BMessage down_message(c_down_block);
			m_down_timer = new BMessageRunner(m_self_messenger,&down_message,m_down_interval);
			m_ignore_downs = false;
			m_game_over = false;
			break;
		}
		case c_pause_game:
			if(m_down_timer)
				delete m_down_timer;
			m_down_timer = NULL;
			m_pause_item->SetLabel(g_resume_string);
			m_pause_item->SetMessage(new BMessage(c_resume_game));
			m_ignore_downs = true;
			break;
		case c_resume_game:
			m_pause_item->SetLabel(g_pause_string);
			m_pause_item->SetMessage(new BMessage(c_pause_game));
			if(m_down_timer == NULL)
			{
				BMessage down_message(c_down_block);
				m_down_timer = new BMessageRunner(m_self_messenger,&down_message,m_down_interval);
			}
			m_ignore_downs = false;
			break;
		case c_edit_preferences:
			//No preferences window needed
			break;
		case c_down_block:
			if(m_ignore_downs || m_game_over)
				break;
			BlockDown();
			break;
	}
}


bool TetrisWindow::BlockDown()
{
	if(!m_tetris_view->MoveFallingBlockDown())
	{
		//Couldn't take the block down...either it's at the bottom, or the game is over
		bool game_over = false;
		if(m_tetris_view->FallingBlockY() < 0)
			//Didn't make it into the grid...Game over
			game_over = true;
		else
		{
			int8 lines = m_tetris_view->AddFallingBlockToGrid();
			if(lines > 0)
			{
				m_lines_until_next_speed_up -= lines;
				if(m_lines_until_next_speed_up <= 0)
				{
					m_lines_until_next_speed_up = c_lines_between_speedups;
					m_down_interval = (bigtime_t)((float)m_down_interval * (1.0-c_speedup));
					if(m_down_timer)
						m_down_timer->SetInterval(m_down_interval);
				}
				m_lines += lines;
				if(lines == 1)
					m_score += 10;
				else if(lines == 2)
					m_score += 23;
				else if(lines == 3)
					m_score += 39;
				else if(lines == 4)
					m_score += 80;
				m_score_view->SetNumber(m_score);
				m_lines_view->SetNumber(m_lines);
			}
			if(!m_tetris_view->AddFallingBlock(m_next_block_type))
				//Couldn't add a new block...game over
				game_over = true;
			m_next_block_type = GetRandomBlock();
			m_next_block_view->SetType(m_next_block_type);
		}
		if(game_over)
		{
			delete m_down_timer;
			m_down_timer = NULL;
			m_ignore_downs = true;
			m_tetris_view->SetGameOver();
			m_game_over = true;
		}
		return false;
	}
	return true;
}


filter_result TetrisWindow::KeyFilterStatic(BMessage *message, BHandler **target,
	BMessageFilter *filter)
{
	return ((TetrisWindow*)filter->Looper())->KeyFilter(message);
}


filter_result TetrisWindow::KeyFilter(BMessage *message)
{
	if((!m_game_over) && (message->what == B_KEY_DOWN || message->what == B_KEY_UP))
	{
		char byte;
		if(message->FindInt8("byte",(int8*)&byte) == B_NO_ERROR && !message->HasInt8("byte",1))
		{
			if(byte == m_key_left && message->what == B_KEY_DOWN)
			{
				m_tetris_view->MoveFallingBlockLeft();
				return B_SKIP_MESSAGE;
			}
			else if(byte == m_key_right && message->what == B_KEY_DOWN)
			{
				m_tetris_view->MoveFallingBlockRight();
				return B_SKIP_MESSAGE;
			}
			else if(byte == m_key_rotate && message->what == B_KEY_DOWN)
			{
				m_tetris_view->RotateFallingBlock();
				return B_SKIP_MESSAGE;
			}
			else if(byte == m_key_down)
			{
				if(message->what == B_KEY_DOWN && m_down_timer)
				{
					bigtime_t down_interval = m_down_interval/2;
					if(down_interval > c_user_down_interval)
						down_interval = c_user_down_interval;
					m_down_timer->SetInterval(down_interval);
				}
				else if(message->what == B_KEY_UP && m_down_timer)
					m_down_timer->SetInterval(m_down_interval);
				return B_SKIP_MESSAGE;
			}
			else if(byte == m_key_drop && message->what == B_KEY_DOWN)
			{
				while(BlockDown())
				{}
				return B_SKIP_MESSAGE;
			}
		}
	}
	return B_DISPATCH_MESSAGE;
}


void TetrisWindow::DrawContent(BView* view,BRect update_rect)
{
	if(m_tetris_view_rect.Intersects(update_rect) &&
		!m_tetris_view_rect.InsetByCopy(2,2).Contains(update_rect))
	{
		view->SetHighColor(m_dark_1_color);
		view->StrokeLine(BPoint(m_tetris_view_rect.left,m_tetris_view_rect.top),
			BPoint(m_tetris_view_rect.right,m_tetris_view_rect.top));
		view->StrokeLine(BPoint(m_tetris_view_rect.left,m_tetris_view_rect.top+1),
			BPoint(m_tetris_view_rect.left,m_tetris_view_rect.bottom));
		view->SetHighColor(White);
		view->StrokeLine(BPoint(m_tetris_view_rect.right,m_tetris_view_rect.top+1),
			BPoint(m_tetris_view_rect.right,m_tetris_view_rect.bottom-1));
		view->StrokeLine(BPoint(m_tetris_view_rect.left+1,m_tetris_view_rect.bottom),
			BPoint(m_tetris_view_rect.right,m_tetris_view_rect.bottom));
		view->SetHighColor(m_dark_2_color);
		view->StrokeLine(BPoint(m_tetris_view_rect.left+1,m_tetris_view_rect.top+1),
			BPoint(m_tetris_view_rect.right-1,m_tetris_view_rect.top+1));
		view->StrokeLine(BPoint(m_tetris_view_rect.left+1,m_tetris_view_rect.top+2),
			BPoint(m_tetris_view_rect.left+1,m_tetris_view_rect.bottom-1));
	}
	if(m_score_label_rect.Intersects(update_rect))
	{
		view->SetHighColor(Black);
		view->SetLowColor(m_background_color);
		view->DrawString(g_score_label,BPoint(m_score_label_rect.left,m_score_label_rect.top+
			m_font_ascent));
	}
	if(m_lines_label_rect.Intersects(update_rect))
	{
		view->SetHighColor(Black);
		view->SetLowColor(m_background_color);
		view->DrawString(g_lines_label,BPoint(m_lines_label_rect.left,m_lines_label_rect.top+
			m_font_ascent));
	}
	if(m_next_block_label_rect.Intersects(update_rect))
	{
		view->SetHighColor(Black);
		view->SetLowColor(m_background_color);
		view->DrawString(g_next_block_label,BPoint(m_next_block_label_rect.left,
			m_next_block_label_rect.top+m_font_ascent));
	}
}


//******************************************************************************************************
//**** TetrisWindowView
//******************************************************************************************************
TetrisWindowView::TetrisWindowView(BRect rect, TetrisWindow *parent)
: BView(rect,NULL,B_FOLLOW_LEFT|B_FOLLOW_TOP,B_WILL_DRAW)
{
	m_parent = parent;
}


void TetrisWindowView::Draw(BRect update_rect)
{
	m_parent->DrawContent(this,update_rect);
}

