//
// Definition of Disc, DiscTowerView and TowerWindow classes
//
// this belongs to the 'Tower of Benoi' Project
// Copyright 1998
//
// Written by: Marco Zinn
//

#ifndef	TOWER_WINDOW_H
#define	TOWER_WINDOW_H

//#include <InterfaceKit.h>
#include "Towers.h"

// Forward declarations:
class Disc; 
class DiscTowerView;
class HistoryWindow;

// Message-Constants
//  ... for moving the discs (drag-and-drop and auto-solving)
//      OT means 'Offering Tower', which is the tower the disc is stacked upon before the move
//      RT means 'Receiving Tower', which is the tower the disc should be stacked upon.
const uint32 OFFER_DISC = 'OfDs';
	// An OFFER_DISC message is send from OT to RT in order to 'offer' the disc ('Hey, RT, do you want that?')
const uint32 DISC_DENIED = 'DnDs';
	// An DISC_DENIED message is send back (as a reply), when the RT can't take the disc
	//  This happens, when the disc, that is moved, is bigger than the upper disc on the RT
const uint32 DISC_ACCEPTED = 'AcDs';
	// An DISC_ACCEPTED message is send back (as a reply), when the RT will take the disc
	//  (In fact, when this message is send, the RT already has put the disc on it's top.
	//   When the OT gets this reply message, it deletes the disc. Thus, there is always
	//   some time when the disc exists on both towers. But you won't see that (normally)!
	//
	//   You can try to synchronize the stuff. But that means another reply-message back to
	//   the RT. Or it means sending an synchronous message (which will wait for the reply).
	//   But this didn't work for me (at Version 1.0) and the it worked and crashed my system
	//   (at Version 1.x), cause the Window get's dead-locked AFAIK! But forget about that.)
	

// ... and the Message-constants for the Menu are
//      Note that the receiver of the menu messages is the application (see TowerWindow.h),
//      By Default, the receiver would be the window the menu is attached to!
const uint32 RESET_GAME = 'Rest';   // Request to reset the game
const uint32 SOLVE_GAME  = 'Solv';   // Request to solve the game

// Enumeration for the Towers (DiscTowerView)
//  Tower A is a START-Tower, Tower B and C are GOAL-Towers
//  That means, that you don't win by stacking the whole tower on Tower A (guess why...)
//  You have to move it to B or C. That's all.
enum TowerKind { START, GOAL };

// Definition of Disc class
//  A Disc is one of these cute little things you move are the screen while playing.
//  Just kidding... 
//  if you're wondering, why this class is named 'Disc' and not 'Rect' or 'Bar':
//  the original game is made of three wooden pillars. The Discs have different size 
//  (diameters), have a hole in the center and are stacked onto these pillars.... thats it.
//  Technically, the Discs are derived from BStringItem (as we want to print at least the
//  number of the disc), which is derived from BListItem. Thanks to that, we can Add and Remove
//  the Discs on any kind of BListView, which we will use to display the Discs as a 'Tower'
//  see below for more.
class Disc : public BStringItem {
	public: 
		Disc(int newSize, rgb_color newColor);    
		// ~Disc() { printf("Disc of size % i destroyed\n",size);}; // Debug
		virtual void DrawItem(BView *owner, BRect itemRect, bool drawEverything);
			// The draw function is reimplemented in order to draw a disc (not only the string)
		int size; // size of the Disc (values are 1,2,3,4,5). used for drawing, naming and comparing
	private:
		rgb_color color; // The Disc's color
	};


// Definition of DiscTowerView class
//  That one is derived from BListView. We'll use it to display the three towers:
//  One Tower will be one 'DiscTowerView'. If you're unfamiliar with BListView, it displays
//  a list of BListItems. BListItems could be BStringItems (which would display a list of
//  strings) or some other stuff (derived from BListItem). In our case, that will be the Discs
//
class DiscTowerView : public BListView {
	public:
		DiscTowerView(BRect frame, TowerKind kind, char Name, HistoryWindow *History);
		void ResetDiscs(); // Resets the discs according to NumberOfDiscs and myKind
		
		// Data and State access Functions:
		Disc *FirstDiscPtr() { return (Disc*) ItemAt(FirstDisc); };
		bool TowerEmpty() { return (FirstDisc==NumberOfDiscs); };
		bool TowerFull() { return (FirstDisc==0); };
	private:
		TowerKind myKind; // Kind of Tower (START or GOAL)
		int FirstDisc; // Position of the uppermost disc on that tower
		               // if =0, the uppermost of on ther very top (tower is full)
		               // if =NumberOfDiscs, the tower is empty (and FirstDisc points out of range!!!)
		virtual bool InitiateDrag(BPoint point, int32 index, bool wasSelected);
			// Starts drag-and-drop-operation
		void MessageReceived(BMessage *message);
			// This does all the message handling
		HistoryWindow	*theHistoryWindow;
			// A Pointer to the HistoryWindow. Used for 'adding' a move to that window.
		char	myName[2];
			// Name of the Tower. Only myName[0] is used (It's 'A', 'B' or 'C')
			// myName[1] is set to 0 in order to use the name as a string.
};

// Definition of TowerWindow class
//  This is the main window. It contains the three Towers and the menu.
//  Note that it doesn't get the messages from the menu, the app gets them (see above)
class TowerWindow : public BWindow {
	public:
		TowerWindow(BRect frame, const char* name, HistoryWindow* History);
		void Reset(); // Resets all three towers
		DiscTowerView	*TowerA; // This is the View for Tower A,
		DiscTowerView	*TowerB; // B,
		DiscTowerView	*TowerC; // and C.
	private:
		virtual bool QuitRequested(); 
};
 
#endif /* TOWER_WINDOW_H */ 