/******************************************************************************/
/* APAddOns header file.                                                      */
/******************************************************************************/
/* This is part of the APlayer Programming Package (APPP).                    */
/* Copyright (C) 1998-1999 by The APlayer-Team.                               */
/* All rights reserved.                                                       */
/*                                                                            */
/* This source, or parts thereof, may only be used in APlayer related         */
/* software. If you want to use it elsewhere, please contact the author for a */
/* permission.                                                                */
/******************************************************************************/


#ifndef __APAddOns_h
#define __APAddOns_h

// PolyKit headers
#include "PString.h"
#include "PFile.h"

// APlayer headers
#include "APChannel.h"
#include "APlayerClasses_Import.h"


/******************************************************************************/
/* Versions to be written in the APAddOn variables                            */
/******************************************************************************/
#define AP_ADDON_VERSION			4
#define AP_PLAYER_VERSION			8
#define AP_AGENT_VERSION			4



/******************************************************************************/
/* Types used in the different functions.                                     */
/******************************************************************************/
class APGlobalData;

enum ap_result {AP_OK, AP_ERROR, AP_UNKNOWN};
enum ap_infoType {apDescription, apValue};

enum ap_playerSupports
{
	appSamplePlayer				= 0x00000001,	// Set this is it's a sample player your create
	appLoadAddOn				= 0x00000002,	// Set this if your player can't use the same add-on twice
	appUseRingBuffer			= 0x00000004,	// Set this if your player require a ring buffer
	appRewind					= 0x00000010,	// Set this if your player have a rewind function
	appForward					= 0x00000020,	// Set this if your player have a forward function
	appSetPosition				= 0x00000040,	// Set this if your player can change to a certain position
};

enum ap_agentPlugins
{
	apaInfoButton				= 0x00000001,	// Set this flag if your agent want to plug-in at the info button
	apaListEditorButton			= 0x00000002,	// Set this flag if your agent want to plug-in at the module list editor button
	apaSampleButton				= 0x00000004,	// Set this flag if your agent want to plug-in at the sample button
	apaModuleInfoChange			= 0x00000100,	// Your agent will be called when the module info changes
	apaAfterInit				= 0x00010000,	// Your agent will be called after a module have been initialized
	apaAfterFree				= 0x00020000,	// Your agent will be called after a module have been freed
	apaAfterInitSong			= 0x00040000,	// Your agent will be called after the subsong initialization
	apaAfterAddingFiles			= 0x00080000,	// Your agent will be called after one or more files has been added to the module list
	apaAfterRemovingFiles		= 0x00100000,	// Your agent will be called after one or more files has been removed from the module list
	apaAfterLoadingBeforeCheck	= 0x00200000,	// Your agent will be called after a module has been loaded, but before the check begins
	apaVirtualMixer				= 0x10000000,	// Your agent need a virtual mixer
};

enum ap_converterSupports
{
	apcLoader					= 0x00000001,	// The converter has a loader
	apcSaver					= 0x00000002,	// The converter has a saver
};



/******************************************************************************/
/* Sample player information structure.                                       */
/* Fill out this structure in your GetSamplePlayerInfo() function in your     */
/* player.                                                                    */
/******************************************************************************/
typedef struct SamplePlayerInfo
{
	uint16		bitSize;				// Number of bits the sample is (only 8 & 16 supported)
	uint32		frequency;				// The frequency to be played with
} SamplePlayerInfo;



/******************************************************************************/
/* Sample information structure.                                              */
/* Fill out this structure in the GetSampleInfo() function in your player.    */
/******************************************************************************/
enum APSampType {apSample, apAM, apFM, apHybrid, apAdlib};

typedef struct APSampleInfo
{
	PString		name;					// The name of the sample
	uint32		flags;					// Some flags about this sample (see below)
	APSampType	type;					// The type of the sample (see above)
	uint8		bitSize;				// Number of bits the sample is (only 8 & 16 supported)
	uint32		middleC;				// The frequency of the middle C (C-4)
	uint16		volume;					// The volume of the sample (0-256)
	int16		panning;				// The panning value (0-255). -1 means no panning
	const void *address;				// Pointer to the sample data
	uint32		length;					// The length of the sample in samples
	uint32		loopStart;				// The start offset to the loop point in samples
	uint32		loopLength;				// The loop length in samples
} APSampleInfo;



/******************************************************************************/
/* Sample information flags.                                                  */
/* These flags are used in the flag field in the APSampleInfo structure above.*/
/******************************************************************************/
#define APSAMP_LOOP			0x00000001	// The sample is looping
#define APSAMP_PINGPONG		0x00000002	// The sample has ping-pong loop (set this together with APSAMP_LOOP)



/******************************************************************************/
/* Instrument information structure.                                          */
/* Fill out this structure in the GetInstrumentInfo() function in your player.*/
/******************************************************************************/
typedef struct APInstInfo
{
	PString		name;					// The name of the sample
	uint32		flags;					// None at the moment. Write 0 in this field
	int16		notes[10 * 12];			// Sample numbers for each note (-1 means not used)
} APInstInfo;



/******************************************************************************/
/* Converter structure.                                                       */
/******************************************************************************/
typedef struct ConvertInfo
{
	PString		name;					// The name of the sample
	uint32		flags;					// Some flags about this sample (the APSAMP_xxxx)
	uint16		bitSize;				// Number of bits the sample is in
	uint32		frequency;				// The frequency of the sample
	uint16		channels;				// The number of channels the sample is in
	uint16		volume;					// The volume of the sample (0-256)
	int16		panning;				// The panning value (0-255). -1 means no panning
	uint32		loopStart;				// The start offset to the loop point in samples
	uint32		loopLength;				// The loop length in samples
} ConvertInfo;



/******************************************************************************/
/* AfterLoadingBeforeCheck plug-in structure.                                 */
/* You will get a pointer to this structure if you plug-in in the above place.*/
/* If you make a module converter, you have to write the new module into an   */
/* APMemFile.                                                                 */
/******************************************************************************/
typedef struct AfterLoadingBeforeCheckInfo
{
// Read only fields
	PFile *		moduleFile;				// The module to load

// If you convert the module, fill out these fields
	PFile *		newModuleFile;			// A pointer to the file where the new module is stored
	PString		modKind;				// The name of the module kind (player name) you converted from
	PString		modType;				// The module type you converted from (mime string)
} AfterLoadingBeforeCheckInfo;



/******************************************************************************/
/* Configuration information structure.                                       */
/* Fill out this structure in your GetConfigInfo() function in your add-on.   */
/******************************************************************************/
#if __p_os == __p_beos

typedef struct ConfigInfo
{
	BView *		view;					// A pointer to the view which show your config panel
	PString		regKey;					// The name of the key where you store your config
} ConfigInfo;

#else

typedef struct ConfigInfo
{
} ConfigInfo;

#endif



/******************************************************************************/
/* Show information structure.                                                */
/* Fill out this structure in your GetShowInfo() function in your add-on.     */
/******************************************************************************/
#if __p_os == __p_beos

typedef struct ShowInfo
{
} ShowInfo;

#else

typedef struct ShowInfo
{
} ShowInfo;

#endif



/******************************************************************************/
/* APAddOnBase class                                                          */
/******************************************************************************/
#if __p_os == __p_beos && __POWERPC__
#pragma export on
#endif

class _IMPEXP_APLIB APAddOnBase
{
public:
	APAddOnBase(APGlobalData *global);
	virtual ~APAddOnBase(void);

	uint32	addOnVersion;			// Write to these variables in your
	uint32	playerVersion;			// constructor. Write the versions
	uint32	agentVersion;			// defined above.

protected:
	APGlobalData *globalData;
};



/******************************************************************************/
/* APAddOn class                                                              */
/******************************************************************************/
class _IMPEXP_APLIB APAddOn : public APAddOnBase
{
public:
	APAddOn(APGlobalData *global);
	virtual ~APAddOn(void);

	// Override these functions if you want to change them
	virtual float GetVersion(void);

	virtual uint32 GetCount(void);
	virtual PString GetName(uint32 index) = 0;
	virtual PString GetDescription(uint32 index) = 0;

	virtual uint32 GetSupportFlags(void);
	virtual ConfigInfo *GetConfigInfo(void);
	virtual ShowInfo *GetShowInfo(void);		// Not implemented yet

private:
	virtual void Reserved1(void);
	virtual void Reserved2(void);
	virtual void Reserved3(void);
	virtual void Reserved4(void);
	virtual void Reserved5(void);
	virtual void Reserved6(void);
	virtual void Reserved7(void);
	virtual void Reserved8(void);
	virtual void Reserved9(void);
	virtual void Reserved10(void);
};



/******************************************************************************/
/* APAddOnPlayer class                                                        */
/******************************************************************************/
class _IMPEXP_APLIB APAddOnPlayer : public APAddOn
{
public:
	APAddOnPlayer(APGlobalData *global);
	virtual ~APAddOnPlayer(void);

	virtual PString GetModTypeString(uint32 index);

	virtual ap_result ModuleCheck(uint32 index, PFile *file) = 0;
	virtual ap_result LoadModule(uint32 index, PFile *file) = 0;
	virtual ap_result ExtraLoad(PString fileName);

	virtual bool InitPlayer(void);
	virtual void EndPlayer(void);
	virtual void InitSound(uint16 songNum);
	virtual void EndSound(void);
	virtual void Play(void) = 0;

	virtual void GetSamplePlayerInfo(SamplePlayerInfo *sampInfo);
	virtual PString GetModuleName(void);
	virtual PString GetAuthor(void);
	virtual uint16 GetVirtualChannels(void);
	virtual uint16 GetModuleChannels(void);
	virtual uint16 *GetSubSongs(void);

	virtual int16 GetSongLength(void);
	virtual int16 GetSongPosition(void);
	virtual void SetSongPosition(int16 pos);
	virtual bool Rewind(void);					// Not implemented yet
	virtual bool Forward(void);					// Not implemented yet

	virtual uint32 GetInfoCount(void);
	virtual PString GetInfoString(uint32 line, ap_infoType type);

	virtual uint32 GetInstrumentCount(void);
	virtual void GetInstrumentInfo(uint32 num, APInstInfo *info);

	virtual uint32 GetSampleCount(void);
	virtual void GetSampleInfo(uint32 num, APSampleInfo *info);

private:
	virtual void PReserved1(void);
	virtual void PReserved2(void);
	virtual void PReserved3(void);
	virtual void PReserved4(void);
	virtual void PReserved5(void);
	virtual void PReserved6(void);
	virtual void PReserved7(void);
	virtual void PReserved8(void);
	virtual void PReserved9(void);
	virtual void PReserved10(void);
	virtual void PReserved11(void);
	virtual void PReserved12(void);
	virtual void PReserved13(void);
	virtual void PReserved14(void);
	virtual void PReserved15(void);
	virtual void PReserved16(void);

public:
	PFile *OpenExtraFile(PString fileName, PString extension);
	void CloseExtraFile(PFile *file);
	uint32 GetExtraFilesSize(void);

	void ChangeModuleInfo(uint32 line, ap_infoType type, PString newString);
	void ChangePosition(void);

	void SetBPMTempo(uint16 bpm);
	uint16 GetBPMTempo(void) const;

	uint32 GetMixingFrequency(void) const;

	// Public variables
	APChannel **virtChannels;	// A pointer to channel objects, one for each channel
	float playFreq;				// How often your play function has to be called in hertz
	bool endReached;			// Set this to true, when your module has reached the end

protected:
	void OpenFile(PFile *file, PString fileName);

private:
	uint32 totalSize;
};



/******************************************************************************/
/* APAddOnAgent class                                                         */
/******************************************************************************/
class _IMPEXP_APLIB APAddOnAgent : public APAddOn
{
public:
	APAddOnAgent(APGlobalData *global);
	virtual ~APAddOnAgent(void);

	virtual bool InitAgent(void);
	virtual void EndAgent(void);
	virtual void OpenWindows(void);
	virtual void Run(void);

	// Plug-in places
	virtual void Plugin_ModuleInfoChanged(uint32 line, ap_infoType type, PString newString);

	virtual void Plugin_AfterInit(void);
	virtual void Plugin_AfterInitSong(void);
	virtual void Plugin_AfterFree(void);
	virtual void Plugin_AfterAddingFiles(void);
	virtual void Plugin_AfterRemovingFiles(void);
	virtual ap_result Plugin_AfterLoadingBeforeCheck(AfterLoadingBeforeCheckInfo *info);

	virtual uint16 Plugin_InitMixer(void);
	virtual void Plugin_EndMixer(void);
	virtual bool Plugin_DoMixing(APChannel **channels);

private:
	virtual void AReserved1(void);
	virtual void AReserved2(void);
	virtual void AReserved3(void);
	virtual void AReserved4(void);
	virtual void AReserved5(void);
	virtual void AReserved6(void);
	virtual void AReserved7(void);
	virtual void AReserved8(void);
	virtual void AReserved9(void);
	virtual void AReserved10(void);
	virtual void AReserved11(void);
	virtual void AReserved12(void);
	virtual void AReserved13(void);
	virtual void AReserved14(void);
	virtual void AReserved15(void);
	virtual void AReserved16(void);
	virtual void AReserved17(void);

public:
	void FreeModule(void);
	void LoadModule(int32 index);
};



/******************************************************************************/
/* APAddOnConverter class                                                     */
/******************************************************************************/
class _IMPEXP_APLIB APAddOnConverter
{
public:
	// Constructor / Destructor
	APAddOnConverter(APGlobalData *global);
	virtual ~APAddOnConverter(void);

	// Support functions
	virtual float GetVersion(void);
	virtual uint32 GetSupportFlags(void) = 0;

	virtual PString GetName(void) = 0;
	virtual PString GetExtension(void) = 0;

	// Loader initialization functions
	virtual BView *LoaderView(void);
	virtual bool LoaderInit(void);
	virtual void LoaderEnd(void);

	// Loader functions
	virtual ap_result FileCheck(PFile *file);
	virtual ap_result LoadHeader(PFile *file, ConvertInfo *convInfo);
	virtual ap_result LoadData(PFile *file, float *buffer, uint32 length, const ConvertInfo *convInfo);
	virtual ap_result LoadTail(PFile *file, const ConvertInfo *convInfo);

	// Saver initialization functions
	virtual BView *SaverView(void);
	virtual bool SaverInit(const ConvertInfo *convInfo);
	virtual void SaverEnd(const ConvertInfo *convInfo);

	// Saver functions
	virtual ap_result SaveHeader(PFile *file, const ConvertInfo *convInfo);
	virtual ap_result SaveData(PFile *file, const float *buffer, uint32 length, const ConvertInfo *convInfo);
	virtual ap_result SaveTail(PFile *file, const ConvertInfo *convInfo);

private:
	virtual void CReserved1(void);
	virtual void CReserved2(void);
	virtual void CReserved3(void);
	virtual void CReserved4(void);
	virtual void CReserved5(void);
	virtual void CReserved6(void);
	virtual void CReserved7(void);
	virtual void CReserved8(void);
	virtual void CReserved9(void);
	virtual void CReserved10(void);

protected:
	APGlobalData *globalData;
};

#if __p_os == __p_beos && __POWERPC__
#pragma export off
#endif

#endif
