/******************************************************************************/
/* APlayer Module Info window class.                                          */
/******************************************************************************/
/* This is part of the APlayer Programming Package (APPP).                    */
/* Copyright (C) 1998-1999 by The APlayer-Team.                               */
/* All rights reserved.                                                       */
/*                                                                            */
/* This source, or parts thereof, may only be used in APlayer related         */
/* software. If you want to use it elsewhere, please contact the author for a */
/* permission.                                                                */
/******************************************************************************/


// PolyKit headers
#include "PException.h"
#include "PString.h"
#include "PAlert.h"
#include "PDirectory.h"
#include "GlobalFuncs.h"

// APlayer headers
#include "ColumnListView.h"
#include "CLVColumn.h"
#include "CLVListItem.h"
#include "Colors.h"

// Needed headers
#include "MIAgent.h"
#include "MIWindow.h"
#include "MIListItem.h"
#include "ModuleInfo_Lang.h"


/******************************************************************************/
/* Constructor                                                                */
/******************************************************************************/
MIWindow::MIWindow(BRect frame, const PString title, APGlobalData *global, uint32 strID) : BWindow(frame, title, B_TITLED_WINDOW, 0)
{
	BRect rect;
	BPoint winSize;
	PString label;
	font_height fh;
	uint32 colWidth;
	void *regHd;

	// Remember the arguments
	globalData = global;
	strings    = strID;

	// Create window locker
	winLock  = new PMutex();
	onScreen = true;
	shutDown = false;

	try
	{
		// Create background view
		rect    = Bounds();
		topView = new BView(rect, NULL, B_FOLLOW_ALL_SIDES, B_WILL_DRAW);

		// Change the color to light grey
		topView->SetViewColor(BeBackgroundGrey);

		// Add view to the window
		AddChild(topView);

		// Get other needed information
		topView->GetFontHeight(&fh);
		fontHeight = max(12, ceil(fh.ascent + fh.descent + fh.leading));

		// Check to see if the given size is lesser than the minimum size
		winSize = CalcMinSize();
		if ((frame.Width() < winSize.x) || (frame.Height() < winSize.y))
			ResizeTo(winSize.x, winSize.y);

		SetSizeLimits(winSize.x, 65536, winSize.y, 65536);

		// Add the column listview
		rect           = Bounds();
		rect.left     += HSPACE;
		rect.right    -= (HSPACE + B_V_SCROLL_BAR_WIDTH);
		rect.top      += VSPACE;
		rect.bottom   -= (VSPACE + B_H_SCROLL_BAR_HEIGHT);
		columnListView = new MIListView(rect, &containerView, B_FOLLOW_ALL_SIDES, B_WILL_DRAW | B_FRAME_EVENTS | B_NAVIGABLE, B_SINGLE_SELECTION_LIST, false, true, true, true, B_PLAIN_BORDER);

		regHd = globalData->useReg->OpenKey("/Agents/ModuleInfo/");

		try
		{
			// Add the columns
			label.LoadString(strings, IDS_MI_WIN_DESCRIPTION);
			colWidth = globalData->saveReg->GetNumber(regHd, "Col1W");
			columnListView->AddColumn(new CLVColumn(label, colWidth, CLV_NOT_MOVABLE));

			label.LoadString(strings, IDS_MI_WIN_VALUE);
			colWidth = globalData->saveReg->GetNumber(regHd, "Col2W");
			columnListView->AddColumn(new CLVColumn(label, colWidth, CLV_NOT_MOVABLE));
		}
		catch(...)
		{
			globalData->useReg->CloseKey(regHd);
			throw;
		}

		// Close the registry again
		globalData->useReg->CloseKey(regHd);

		// Attach the listview to the window
		topView->AddChild(containerView);

		// Add the items in the listview
		AddItems();
	}
	catch(...)
	{
		try
		{
			PString title, msg;

			title.LoadString(strings, IDS_MI_WIN_TITLE);
			msg.LoadString(strings, IDS_MI_ERR_INIT);

			PAlert alert(title, msg, PAlert::pStop, PAlert::pOk);
			alert.Show();
		}
		catch(...)
		{
			PlayBeep();
		}

		throw;
	}
}



/******************************************************************************/
/* Destructor                                                                 */
/******************************************************************************/
MIWindow::~MIWindow(void)
{
	try
	{
		BRect winPos;
		CLVColumn *column;
		int32 x, y, w, h;
		void *regHd, *saveRegHd;

		// Open the registry
		regHd = globalData->useReg->OpenKey("/Agents/ModuleInfo/");

		try
		{
			saveRegHd = globalData->saveReg->OpenKey("/Agents/ModuleInfo/");

			try
			{
				// Store the window position and size if changed
				winPos = Frame();

				x = (int32)winPos.left;
				y = (int32)winPos.top;
				w = winPos.IntegerWidth();
				h = winPos.IntegerHeight();

				// Check to see if they have changed
				if ((globalData->useReg->GetNumber(regHd, "WinX") != x) ||
					(globalData->useReg->GetNumber(regHd, "WinY") != y) ||
					(globalData->useReg->GetNumber(regHd, "WinW") != w) ||
					(globalData->useReg->GetNumber(regHd, "WinH") != h) ||
					(globalData->useReg->GetNumber(regHd, "OpenWindow") != wasOnScreen))
				{
					globalData->saveReg->CreateNumber(saveRegHd, "WinX", x);
					globalData->saveReg->CreateNumber(saveRegHd, "WinY", y);
					globalData->saveReg->CreateNumber(saveRegHd, "WinW", w);
					globalData->saveReg->CreateNumber(saveRegHd, "WinH", h);
					globalData->saveReg->CreateNumber(saveRegHd, "OpenWindow", wasOnScreen);
				}

				// Store the column sizes
				column = columnListView->ColumnAt(0);
				w      = (int32)column->Width();

				if (globalData->useReg->GetNumber(regHd, "Col1W") != w)
					globalData->saveReg->CreateNumber(saveRegHd, "Col1W", w);

				column = columnListView->ColumnAt(1);
				w      = (int32)column->Width();

				if (globalData->useReg->GetNumber(regHd, "Col2W") != w)
					globalData->saveReg->CreateNumber(saveRegHd, "Col2W", w);
			}
			catch(...)
			{
				globalData->saveReg->CloseKey(saveRegHd);
				throw;
			}

			globalData->saveReg->CloseKey(saveRegHd);
		}
		catch(...)
		{
			globalData->useReg->CloseKey(regHd);
			throw;
		}

		// Close the registry again
		globalData->useReg->CloseKey(regHd);
	}
	catch(...)
	{
		;
	}

	// Delete the locker
	delete winLock;
}



/******************************************************************************/
/* QuitRequested() is called when the user presses the close button.          */
/*                                                                            */
/* Output: Returns true if it's okay to quit, else false.                     */
/******************************************************************************/
bool MIWindow::QuitRequested(void)
{
	bool retVal;

	winLock->Lock();

	wasOnScreen = onScreen;

	if (onScreen)
	{
		Hide();
		onScreen = false;
	}

	retVal = shutDown;
	winLock->Unlock();

	return (retVal);
}



/******************************************************************************/
/* WindowActivated() is called when the window becomes the active one. It     */
/*      will set the focus on the column list view.                           */
/*                                                                            */
/* Input:  "active" is true if the window get active, else false.             */
/******************************************************************************/
void MIWindow::WindowActivated(bool active)
{
	if (active)
		columnListView->MakeFocus();
}



/******************************************************************************/
/* RefreshWindow() will clear the window and add all the items again.         */
/******************************************************************************/
void MIWindow::RefreshWindow(void)
{
	uint32 i, count;
	CLVListItem *item;

	// Lock the window
	Lock();

	// Remove all the items from the list view
	count = columnListView->CountItems();
	for (i = 0; i < count; i++)
	{
		item = (CLVListItem *)columnListView->RemoveItem((int32)0);
		delete item;
	}

	// Add the items
	AddItems();

	// Unlock it again
	Unlock();
}



/******************************************************************************/
/* UpdateWindow() will be called everytime a new value or description has     */
/*      changed.                                                              */
/*                                                                            */
/* Input:  "line" is the line starting from 0.                                */
/*         "type" is the type of string changed.                              */
/*         "newString" is the new string.                                     */
/******************************************************************************/
void MIWindow::UpdateWindow(uint32 line, ap_infoType type, PString newString)
{
	MIListItem *item;
	bool changed;

	// Lock the window
	Lock();

	// Find the item to change
	item = (MIListItem *)columnListView->ItemAt(line + 6);

	if (item)
	{
		// Now change the item
		if (type == apDescription)
			changed = item->ChangeColumn(newString, 0);
		else
			changed = item->ChangeColumn(newString, 1);

		// Invalidate the item so it will be redrawn
		if (changed)
			columnListView->InvalidateItem(line + 6);
	}

	// Unlock the window
	Unlock();
}



/******************************************************************************/
/* CalcMinSize() will calculate the minimum size the window can have.         */
/*                                                                            */
/* Output: Is a BPoint where the x is the minimum width and y is the minimum  */
/*         height.                                                            */
/******************************************************************************/
BPoint MIWindow::CalcMinSize(void)
{
	BPoint size;

	size.x = HSPACE * 2 + B_V_SCROLL_BAR_WIDTH + 50;
	size.y = VSPACE * 2 + B_H_SCROLL_BAR_HEIGHT + 50;

	return (size);
}



/******************************************************************************/
/* AddItems() will add the all items in the column list view.                 */
/******************************************************************************/
void MIWindow::AddItems(void)
{
	PString desc, val;

	// Lock the global data
	globalData->Lock();

	// Check to see if there is any module in memory
	if (globalData->player == NULL)
	{
		// No module in memory
		val.LoadString(strings, IDS_MI_ITEM_NA);

		desc.LoadString(strings, IDS_MI_ITEM_MODULENAME);
		columnListView->AddItem(new MIListItem(fontHeight, desc, val));

		desc.LoadString(strings, IDS_MI_ITEM_AUTHOR);
		columnListView->AddItem(new MIListItem(fontHeight, desc, val));

		desc.LoadString(strings, IDS_MI_ITEM_ACTIVEPLAYER);
		columnListView->AddItem(new MIListItem(fontHeight, desc, val));

		desc.LoadString(strings, IDS_MI_ITEM_MODULESIZE);
		columnListView->AddItem(new MIListItem(fontHeight, desc, val));

		desc.LoadString(strings, IDS_MI_ITEM_CHANNELS);
		columnListView->AddItem(new MIListItem(fontHeight, desc, val));
	}
	else
	{
		// Module in memory, add items
		uint32 i, count;
		APAddOnPlayer *player = globalData->player;

		desc.LoadString(strings, IDS_MI_ITEM_MODULENAME);

		val = player->GetModuleName();
		if (val.IsEmpty())
			val = PDirectory::GetFilePart(globalData->fileName);
		else
		{
			int32 i, count;
			bool ok = false;

			// Check to see if the module name only have spaces
			count = val.GetLength();
			for (i = 0; i < count; i++)
			{
				if (val.GetAt(i) != ' ')
				{
					ok = true;
					break;
				}
			}

			if (!ok)
				val = PDirectory::GetFilePart(globalData->fileName);
		}

		columnListView->AddItem(new MIListItem(fontHeight, desc, val));

		desc.LoadString(strings, IDS_MI_ITEM_AUTHOR);

		if (globalData->author.IsEmpty())
			val.LoadString(strings, IDS_MI_ITEM_UNKNOWN);
		else
			val = globalData->author;

		columnListView->AddItem(new MIListItem(fontHeight, desc, val));

		desc.LoadString(strings, IDS_MI_ITEM_ACTIVEPLAYER);
		columnListView->AddItem(new MIListItem(fontHeight, desc, globalData->playerName));

		desc.LoadString(strings, IDS_MI_ITEM_MODULESIZE);
		val.Format("%u", globalData->moduleLen);
		columnListView->AddItem(new MIListItem(fontHeight, desc, val.MakeNumber()));

		desc.LoadString(strings, IDS_MI_ITEM_CHANNELS);
		val.Format("%u", player->GetModuleChannels());
		columnListView->AddItem(new MIListItem(fontHeight, desc, val));

		// Add player specific items
		count = player->GetInfoCount();

		if (count > 0)
		{
			// Add an empty line
			columnListView->AddItem(new MIListItem(fontHeight, "", ""));
		}

		for (i = 0; i < count; i++)
		{
			desc = player->GetInfoString(i, apDescription);
			val  = player->GetInfoString(i, apValue);

			columnListView->AddItem(new MIListItem(fontHeight, desc, val));
		}
	}

	// Unlock it again
	globalData->Unlock();
}
