//------------------------------------------------------------------------------
// BackOnTrack.cpp
// 
// Copyright © 1998 Tyler Riti & Fly By Night Productions
// Unauthorized use, modification, or distribution of this code in whole or in
// part without prior consent of the original author is strictly prohibited.
// 
// Created: 02/27/99 17:55:40
//------------------------------------------------------------------------------

//-- D E F I N E S -------------------------------------------------------------

#define APP_SIG				"application/x-vnd.fbnp.BackOnTrack"
#define TRACKER_SIG			"application/x-vnd.Be-TRAK"
#define DESKBAR_SIG			"application/x-vnd.Be-TSKB"
#define TERMINAL_SIG		"application/x-vnd.Be-SHEL"
#define RELAUNCH_TRACKER	1
#define RELAUNCH_DESKBAR	2

//-- I N C L U D E S -----------------------------------------------------------

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <Debug.h>
#include <Application.h>
#include <Roster.h>
#include <Message.h>
#include <Alert.h>

//-- C L A S S E S -------------------------------------------------------------

class BOTApp : public BApplication
{
public:
	BOTApp(const char *signature);
	
	virtual void ReadyToRun(void);
	virtual bool QuitRequested(void);
	virtual void ArgvReceived(int32 argc, char **argv);
	virtual void MessageReceived(BMessage *message);
	
private:
	void LaunchApp(const char *signature);
	void AlertUser(int32 job);
	void GiveUp(void);
};

//-- I M P L E M E N T A T I O N -----------------------------------------------

BOTApp::BOTApp(const char *signature) :
	BApplication(signature)
{
	// Goes Nowhere and Does Nothing
}

void BOTApp::ReadyToRun(void)
{
	// Start watching for applications to quit
	if (be_roster->StartWatching(be_app_messenger, B_REQUEST_QUIT) != B_OK)
	{
		fprintf(stderr, "StartWatching() failed!\n");
		exit(EXIT_FAILURE);
	}
}

bool BOTApp::QuitRequested(void)
{
	// Tell the roster that we're done watching
	be_roster->StopWatching(be_app_messenger);
	// Finish quitting
	return BApplication::QuitRequested();
}

void BOTApp::ArgvReceived(int32 argc, char **argv)
{
	for (int32 i = 1; i < argc; ++i)
	{
		if (strcmp(argv[i], "--quit") == 0)
			PostMessage(B_QUIT_REQUESTED);
	}
}

void BOTApp::MessageReceived(BMessage *message)
{
	char	*signature;
	
	// Look for quit events
	if (message->what == B_SOME_APP_QUIT)
	{
		// Fish the signature out of the message
		if (message->FindString("be:signature", &signature) == B_OK)
		{
			printf("%s is quitting\n", signature);
			
			// Compare signatures
			if (strcmp(signature, TRACKER_SIG) == 0)
				// Tell the user our beloved Tracker has quit
				AlertUser(RELAUNCH_TRACKER);
			else if (strcmp(signature, DESKBAR_SIG) == 0)
				// Tell the user that the Deskbar is no more
				AlertUser(RELAUNCH_DESKBAR);
		}
	}
	else
		BApplication::MessageReceived(message);
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

void BOTApp::LaunchApp(const char* signature)
{
	// Relaunch the app or die trying
	if (be_roster->Launch(signature) != B_OK)
		// Something went wrong, give up
		GiveUp();
}

void BOTApp::AlertUser(int32 job)
{
	BAlert	*alert;
	int32	result;
	
	printf("AlertUser() called to fix %s\n", (job == 1 ? "Tracker" : "Deskbar"));
	
	// Alert the user to the situation
	if (job == RELAUNCH_TRACKER)
		alert = new BAlert("Uh oh!", "The Tracker is no longer running. Would you like to relaunch the Tracker or open a Terminal?",
			"Cancel", "Terminal", "Tracker", B_WIDTH_AS_USUAL, B_OFFSET_SPACING, B_WARNING_ALERT);
	else if (job == RELAUNCH_DESKBAR)
		alert = new BAlert("Uh oh!", "The Deskbar is no longer running. Would you like to relaunch the Deskbar or open a Terminal?",
			"Cancel", "Terminal", "Deskbar", B_WIDTH_AS_USUAL, B_OFFSET_SPACING, B_WARNING_ALERT);
	else
		GiveUp();
	
	result = alert->Go();
	
	switch (result)
	{
		case 2: // Tracker or Deskbar
			ASSERT(job == RELAUNCH_TRACKER || job == RELAUNCH_DESKBAR);
			if (job == RELAUNCH_TRACKER)
				LaunchApp(TRACKER_SIG);
			else
				LaunchApp(DESKBAR_SIG);
			break;
		case 1: // Terminal
			LaunchApp(TERMINAL_SIG);
			break;
		case 0: // Cancel
		default:
			break;
	}
}

void BOTApp::GiveUp(void)
{
	BAlert	*alert;
	
	// We tried our best but I guess our best wasn't good enough
	alert = new BAlert("Give Up", "BackOnTrack has failed. You're on your own now.",
		"Sorry", NULL, NULL, B_WIDTH_AS_USUAL, B_STOP_ALERT);
	alert->Go();
	PostMessage(B_QUIT_REQUESTED);
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

int main(void)
{
	BOTApp	*app = new BOTApp(APP_SIG);
	app->Run();
	delete app;
	return EXIT_SUCCESS;
}

//------------------------------------------------------------------------------
