/*
 * September 25, 1991
 * Copyright 1991 Guido van Rossum And Sundry Contributors
 * This source code is freely redistributable and may be used for
 * any purpose.  This copyright notice must be maintained. 
 * Guido van Rossum And Sundry Contributors are not responsible for 
 * the consequences of using this software.
 */

/*
 * Sound Tools SGI/Amiga AIFF format.
 * Used by SGI on 4D/35 and Indigo.
 * This is a subformat of the EA-IFF-85 format.
 * This is related to the IFF format used by the Amiga.
 * But, apparently, not the same.
 *
 * Jan 93: new version from Guido Van Rossum that 
 * correctly skips unwanted sections.
 *
 * Jan 94: add loop & marker support
 */

#include <math.h>
#include "st.h"

/* Private data used by writer */
struct aiffpriv {
	unsigned long nsamples;	/* number of 1-channel samples read or written */
};

double read_ieee_extended(ft_t );
double ConvertFromIeeeExtended(unsigned char *bytes);
ConvertToIeeeExtended(double num, char *bytes);

int aiffstartread(ft_t ft) 
{
	struct aiffpriv *p = (struct aiffpriv *) ft->priv;
	char buf[5];
	unsigned long totalsize;
	unsigned long chunksize;
	int channels;
	unsigned long frames;
	int bits;
	double rate;
	unsigned long offset;
	unsigned long blocksize;
	int littlendian = 0;
	char *endptr;
	int foundcomm = 0, foundmark = 0, foundinstr = 0;
	struct mark {
		int id, position;
		char name[40]; 
	} marks[32];
	int i, nmarks, marker1, marker2, marker3, marker4, seekto, ssndsize;


	/* FORM chunk */
	if (fread(buf, 1, 4, ft->fp) != 4 || strncmp(buf, "FORM", 4) != 0)
		return 0;
//		fail("AIFF header does not begin with magic word 'FORM'");
	totalsize = rblong(ft);
	if (fread(buf, 1, 4, ft->fp) != 4 || strncmp(buf, "AIFF", 4) != 0)
		return 0;
//		fail("AIFF 'FORM' chunk does not specify 'AIFF' as type");

	
	/* Skip everything but the COMM chunk and the SSND chunk */
	/* The SSND chunk must be the last in the file */
	while (1) {
		if (fread(buf, 1, 4, ft->fp) != 4)
			return 0;
//			fail("Missing SSND chunk in AIFF file");

		if (strncmp(buf, "COMM", 4) == 0) {
			/* COMM chunk */
			chunksize = rblong(ft);
			if (chunksize != 18)
				return 0;
//				fail("AIFF COMM chunk has bad size");
			channels = rbshort(ft);
			frames = rblong(ft);
			bits = rbshort(ft);
			rate = read_ieee_extended(ft);
			foundcomm = 1;
		}
		else if (strncmp(buf, "SSND", 4) == 0) {
			/* SSND chunk */
			chunksize = rblong(ft);
			offset = rblong(ft);
			blocksize = rblong(ft);
			chunksize -= 8;
			ssndsize = chunksize;
			if (!ft->seekable)
				break;
			seekto = ftell(ft->fp);
		/*	fseek(ft->fp, chunksize, 1); /* seek to end of sound */
		/*	break; */
		}
		else if (strncmp(buf, "MARK", 4) == 0) {
			/* MARK chunk */
			chunksize = rblong(ft);
			nmarks = rbshort(ft);
			for(i = 0; i < nmarks; i++) {
				int len;

				marks[i].id = rbshort(ft);
				marks[i].position = rblong(ft);
				chunksize -= 6;
				len = getc(ft->fp);
				chunksize -= len + 1;
				for(; len ; len--) 
					marks[i].name[i] = getc(ft->fp);
			}
			while(chunksize--)
				getc(ft->fp);
			foundmark = 1;
		}
		else if (strncmp(buf, "INST", 4) == 0) {
			/* INST chunk */
			chunksize = rblong(ft);
			ft->instr.MIDInote = getc(ft->fp);
			getc(ft->fp);				/* detune */
			ft->instr.MIDIlow = getc(ft->fp);
			ft->instr.MIDIhi = getc(ft->fp);
			getc(ft->fp);			/* low velocity */
			getc(ft->fp);			/* hi  velocity */
			rbshort(ft);				/* gain */
			ft->loops[0].type = rbshort(ft);
			marker1 = rbshort(ft);
			marker2 = rbshort(ft);
			ft->loops[1].type = rbshort(ft);
			marker3 = rbshort(ft);
			marker4 = rbshort(ft);
			foundinstr = 1;
		} else if (strncmp(buf, "ALCH", 4) == 0) {
			/* INST chunk */
			rblong(ft);		/* ENVS - jeez! */
			chunksize = rblong(ft);
			while(chunksize--)
				getc(ft->fp);
		} else {
			buf[4] = 0;
			if (feof(ft->fp))
				break;
//			report("AIFFstartread: ignoring '%s' chunk\n", buf);
			chunksize = rblong(ft);
			if (feof(ft->fp))
				break;
			/* Skip the chunk using getc() so we may read
			   from a pipe */
			while ((long) (--chunksize) >= 0) {
				if (getc(ft->fp) == EOF)
					break;
			}
		}
		if (feof(ft->fp))
			break;
	}

	/* 
	 * if a pipe, we lose all chunks after sound.  
	 * Like, say, instrument loops. 
	 */
	if (ft->seekable)
		fseek(ft->fp, seekto, 0);

	/* SSND chunk just read */
	if (blocksize != 0)
		return 0;
//		fail("AIFF header specifies nonzero blocksize?!?!");
	while ((long) (--offset) >= 0) {
		if (getc(ft->fp) == EOF)
			return 0;
//			fail("unexpected EOF while skipping AIFF offset");
	}

	if (foundcomm) {
		ft->info.channels = channels;
		ft->info.rate = rate;
		ft->info.style = SIGN2;
		switch (bits) {
		case 8:
			ft->info.size = BYTE;
			break;
		case 16:
			ft->info.size = WORD;
			break;
		default:
			return 0;
//			fail("unsupported sample size in AIFF header: %d", bits);
			/*NOTREACHED*/
		}
	} else  {
		if ((ft->info.channels == -1)
			|| (ft->info.rate == -1)
			|| (ft->info.style == -1)
			|| (ft->info.size == -1)) {
				return 0;
//report("You must specify # channels, sample rate, signed/unsigned,\n");
//report("and 8/16 on the command line.");
//			fail("Bogus AIFF file: no COMM section.");
		}

	}

	p->nsamples = ssndsize / ft->info.size;	/* leave out channels */

	/* process instrument and marker notations. */
	if (foundmark && !foundinstr)
		return 0;
//		fail("Bogus AIFF file: MARKers but no INSTrument.");
	if (!foundmark && foundinstr)
		return 0;
//		fail("Bogus AIFF file: INSTrument but no MARKers.");
	if (foundmark && foundinstr) {
		ft->loops[0].start = marks[marker1].position;
		ft->loops[0].length = 
			marks[marker2].position - marks[marker1].position;
		ft->loops[1].start = marks[marker3].position;
		ft->loops[1].length = 
			marks[marker4].position - marks[marker3].position;
		ft->loops[0].count = ft->loops[0].count = 0;
		ft->instr.loopmode = LOOP_SUSTAIN_DECAY;
		ft->instr.nloops = 2;
	}

	endptr = (char *) &littlendian;
	*endptr = 1;
	if (littlendian == 1)
		ft->swap = 1;
	
	return 1;
}

int aiffread(ft_t ft, long *buf, int len)
{
	struct aiffpriv *p = (struct aiffpriv *) ft->priv;

	/* just read what's left of SSND chunk */
	if (len > p->nsamples)
		len = p->nsamples;
	rawread(ft, buf, len);
	p->nsamples -= len;
	return len;
}

aiffstopread(ft_t ft) 
{
	struct aiffpriv *p = (struct aiffpriv *) ft->priv;
	char buf[5];
	unsigned long chunksize;

	if (!ft->seekable)
	    while (! feof(ft->fp)) {
		if (fread(buf, 1, 4, ft->fp) != 4)
			return 0;

		chunksize = rblong(ft);
		if (feof(ft->fp))
			return 0;
		buf[4] = '\0';
//		warn("Ignoring AIFF tail chunk: '%s', %d bytes long\n", 
//			buf, chunksize);
//		if (! strcmp(buf, "MARK") || ! strcmp(buf, "INST"))
//			warn("You're stripping MIDI/loop info!\n");
		while ((long) (--chunksize) >= 0) 
			if (getc(ft->fp) == EOF)
				return 0;
	}
}

double read_ieee_extended(ft_t ft)
{
	char buf[10];
	if (fread(buf, 1, 10, ft->fp) != 10)
		return 0;
//		fail("EOF while reading IEEE extended number");
	return ConvertFromIeeeExtended((unsigned char *)buf);
}

write_ieee_extended(ft_t ft, double x)
{
	char buf[10];
	ConvertToIeeeExtended(x, buf);
	(void) fwrite(buf, 1, 10, ft->fp);
}


/*
 * C O N V E R T   T O   I E E E   E X T E N D E D
 */

/* Copyright (C) 1988-1991 Apple Computer, Inc.
 * All rights reserved.
 *
 * Machine-independent I/O routines for IEEE floating-point numbers.
 *
 * NaN's and infinities are converted to HUGE_VAL or HUGE, which
 * happens to be infinity on IEEE machines.  Unfortunately, it is
 * impossible to preserve NaN's in a machine-independent way.
 * Infinities are, however, preserved on IEEE machines.
 *
 * These routines have been tested on the following machines:
 *    Apple Macintosh, MPW 3.1 C compiler
 *    Apple Macintosh, THINK C compiler
 *    Silicon Graphics IRIS, MIPS compiler
 *    Cray X/MP and Y/MP
 *    Digital Equipment VAX
 *
 *
 * Implemented by Malcolm Slaney and Ken Turkowski.
 *
 * Malcolm Slaney contributions during 1988-1990 include big- and little-
 * endian file I/O, conversion to and from Motorola's extended 80-bit
 * floating-point format, and conversions to and from IEEE single-
 * precision floating-point format.
 *
 * In 1991, Ken Turkowski implemented the conversions to and from
 * IEEE double-precision format, added more precision to the extended
 * conversions, and accommodated conversions involving +/- infinity,
 * NaN's, and denormalized numbers.
 */

#ifndef HUGE_VAL
# define HUGE_VAL HUGE
#endif /*HUGE_VAL*/

# define FloatToUnsigned(f)      ((unsigned long)(((long)(f - 2147483648.0)) + 2147483647L) + 1)

ConvertToIeeeExtended(double num, char *bytes)
{
    int    sign;
    int expon;
    double fMant, fsMant;
    unsigned long hiMant, loMant;

    if (num < 0) {
        sign = 0x8000;
        num *= -1;
    } else {
        sign = 0;
    }

    if (num == 0) {
        expon = 0; hiMant = 0; loMant = 0;
    }
    else {
        fMant = frexp(num, &expon);
        if ((expon > 16384) || !(fMant < 1)) {    /* Infinity or NaN */
            expon = sign|0x7FFF; hiMant = 0; loMant = 0; /* infinity */
        }
        else {    /* Finite */
            expon += 16382;
            if (expon < 0) {    /* denormalized */
                fMant = ldexp(fMant, expon);
                expon = 0;
            }
            expon |= sign;
            fMant = ldexp(fMant, 32);          
#ifdef	ARM
            fsMant = sfloor(fMant); 	/*  as UnixLib has a fit over this */
#else
            fsMant = floor(fMant); 
#endif
            hiMant = FloatToUnsigned(fsMant);
            fMant = ldexp(fMant - fsMant, 32); 
#ifdef	ARM
            fsMant = sfloor(fMant); 	/*  as UnixLib has a fit over this */
#else
            fsMant = floor(fMant); 
#endif
            loMant = FloatToUnsigned(fsMant);
        }
    }
    
    bytes[0] = expon >> 8;
    bytes[1] = expon;
    bytes[2] = hiMant >> 24;
    bytes[3] = hiMant >> 16;
    bytes[4] = hiMant >> 8;
    bytes[5] = hiMant;
    bytes[6] = loMant >> 24;
    bytes[7] = loMant >> 16;
    bytes[8] = loMant >> 8;
    bytes[9] = loMant;
}


/*
 * C O N V E R T   F R O M   I E E E   E X T E N D E D  
 */

/* 
 * Copyright (C) 1988-1991 Apple Computer, Inc.
 * All rights reserved.
 *
 * Machine-independent I/O routines for IEEE floating-point numbers.
 *
 * NaN's and infinities are converted to HUGE_VAL or HUGE, which
 * happens to be infinity on IEEE machines.  Unfortunately, it is
 * impossible to preserve NaN's in a machine-independent way.
 * Infinities are, however, preserved on IEEE machines.
 *
 * These routines have been tested on the following machines:
 *    Apple Macintosh, MPW 3.1 C compiler
 *    Apple Macintosh, THINK C compiler
 *    Silicon Graphics IRIS, MIPS compiler
 *    Cray X/MP and Y/MP
 *    Digital Equipment VAX
 *
 *
 * Implemented by Malcolm Slaney and Ken Turkowski.
 *
 * Malcolm Slaney contributions during 1988-1990 include big- and little-
 * endian file I/O, conversion to and from Motorola's extended 80-bit
 * floating-point format, and conversions to and from IEEE single-
 * precision floating-point format.
 *
 * In 1991, Ken Turkowski implemented the conversions to and from
 * IEEE double-precision format, added more precision to the extended
 * conversions, and accommodated conversions involving +/- infinity,
 * NaN's, and denormalized numbers.
 */

#ifndef HUGE_VAL
# define HUGE_VAL HUGE
#endif /*HUGE_VAL*/

# define UnsignedToFloat(u)         (((double)((long)(u - 2147483647L - 1))) + 2147483648.0)

/****************************************************************
 * Extended precision IEEE floating-point conversion routine.
 ****************************************************************/

double ConvertFromIeeeExtended(unsigned char *bytes)
{
    double    f;
    int    expon;
    unsigned long hiMant, loMant;
    
    expon = ((bytes[0] & 0x7F) << 8) | (bytes[1] & 0xFF);
    hiMant    =    ((unsigned long)(bytes[2] & 0xFF) << 24)
            |    ((unsigned long)(bytes[3] & 0xFF) << 16)
            |    ((unsigned long)(bytes[4] & 0xFF) << 8)
            |    ((unsigned long)(bytes[5] & 0xFF));
    loMant    =    ((unsigned long)(bytes[6] & 0xFF) << 24)
            |    ((unsigned long)(bytes[7] & 0xFF) << 16)
            |    ((unsigned long)(bytes[8] & 0xFF) << 8)
            |    ((unsigned long)(bytes[9] & 0xFF));

    if (expon == 0 && hiMant == 0 && loMant == 0) {
        f = 0;
    }
    else {
        if (expon == 0x7FFF) {    /* Infinity or NaN */
            f = HUGE_VAL;
        }
        else {
            expon -= 16383;
            f  = ldexp(UnsignedToFloat(hiMant), expon-=31);
            f += ldexp(UnsignedToFloat(loMant), expon-=32);
        }
    }

    if (bytes[0] & 0x80)
        return -f;
    else
        return f;
}

