// This is a sample plugin snippet for Eddie
//
// (c) 1998 P.Cisler
//
// Any portion of this code may be used freely to develop Eddie
// plugins 
//
//
// The plugin sample shows you how to:
//
// 		install a plugin with a button
//		scann document text
//		insert and remove text using composite undo
//		intercept button clicks
//		publish and intercept keyboard shortcut primitives
//		run a popup menu during a button press
//
//
// The plugin is a simple C and C++ commenter
//
// This plugin publishes several keyboard primitives. You may want to bind
// some of them to a keyboard shortcut. A good example could be:
//
// SetKey Alt-/ UncommentOrPreferredComment
// SetKey Alt-Win-/ UncommentOrAlternateComment
//
//		- add these two lines to the UserStartup file and you will be able
//		  to comment a selection out an in with a single key press
//
//
//
// Things that would make this plugin even more usefull but are beyond the
// scope of this sample code snippet:
// 	- add more fancy formatting of /* comments
//	- handle nested /* comments by breaking into *//* around them
//	- add support for more document types - shell # comments, Pascal comments, etc.
//	- make the scanner more robust, making it handle quotes, etc. properly
//

#include <PopUpMenu.h>
#include <MenuItem.h>
#include "Plugin.h"

extern const unsigned char kNormalBits[];
extern const unsigned char kPressedBits[];

static int32
CUncomment(PluginInterface *pluginInterface)
{
	// this scanner is simplified, ignores quotes, etc.
	// real C/C++ scanner left as an excercise to the reader

	int32 selStart, selEnd;
	pluginInterface->GetSelection(&selStart, &selEnd);
	int32 lineStart = pluginInterface->StartOfLine(selStart);
	int32 lineEnd = pluginInterface->EndOfLine(selEnd);

	bool removedCComment = false;
	int32 removedCount = 0;
	
	for (int32 index = lineStart; index < lineEnd; ) {

		if (pluginInterface->CharAt(index++) != '/') 
			return removedCount;

		char ch = pluginInterface->CharAt(index++);
		if (ch == '*')
			removedCComment = true;
		else if (ch != '/') 
			return removedCount;

		index -= 2;
		lineEnd -= 2;
		pluginInterface->Select(index, index + 2);
		pluginInterface->CurrentUndoClear();
		removedCount += 2;

		// look for line end
		for (; index < lineEnd; ) {
			if (pluginInterface->CharAt(index) == '\n') 
				break;
			index++;
		}
		if (removedCComment) {
			// found line end; if removing /* style comments
			// remove the closing */ comment from line end
			if (pluginInterface->CharAt(index - 1) == '/'
				&& pluginInterface->CharAt(index - 2) == '*') {
				pluginInterface->Select(index - 2, index);
				pluginInterface->CurrentUndoClear();
				index -= 1;
				lineEnd -= 2;
				removedCount += 2;
				removedCComment = false;
			}
		} else
			index++;
	}
	// return the number of characters we have removed
	return removedCount;
}

static int32 
CPlusCommentSelection(PluginInterface *pluginInterface)
{
	int32 addedCount = 0;
	int32 selStart, selEnd;
	pluginInterface->GetSelection(&selStart, &selEnd);
	int32 lineStart = pluginInterface->StartOfLine(selStart);
	int32 lineEnd = pluginInterface->EndOfLine(selEnd);
	
	for (int32 index = lineStart; index < lineEnd; ) {
		// add C++ comments to beginning of each line
		pluginInterface->Select(index, index);
		pluginInterface->CurrentUndoInsert("//", 2);
		addedCount += 2;
		index += 2;
		lineEnd += 2;

		// scan to next line start
		for (; index < lineEnd; index++) {
			if (index >= lineEnd)
				break;
			if (pluginInterface->CharAt(index) == '\n') {
				index++;
				break;
			}
		}
	}

	// return the number of characters we have added
	return addedCount;
}

static int32 
CCommentSelection(PluginInterface *pluginInterface)
{
	int32 addedCount = 0;
	int32 selStart, selEnd;
	pluginInterface->GetSelection(&selStart, &selEnd);
	int32 lineStart = pluginInterface->StartOfLine(selStart);
	int32 lineEnd = pluginInterface->EndOfLine(selEnd);
	
	for (int32 index = lineStart; index < lineEnd; ) {
		// add C++ comments to beginning of each line
		pluginInterface->Select(index, index);
		pluginInterface->CurrentUndoInsert("/*", 2);
		addedCount += 2;
		index += 2;
		lineEnd += 2;

		// scan to next line start
		for (; index < lineEnd; ) {
			if (pluginInterface->CharAt(index) == '\n')
				break;
			index++;
		}

		pluginInterface->Select(index, index);
		pluginInterface->CurrentUndoInsert("*/", 2);
		addedCount += 2;
		lineEnd += 2;
		index += 3;
	}

	// return the number of characters we have added
	return addedCount;
}

static void
CPlusCommentOrUncommentSelection(PluginInterface *pluginInterface)
{
	int32 selStart, selEnd;
	pluginInterface->GetSelection(&selStart, &selEnd);
	int32 lineStart = pluginInterface->StartOfLine(selStart);
	int32 lineEnd = pluginInterface->EndOfLine(selEnd);
	
	// set up a composite undo record; all the successive inserts and
	// removes will be undone as a single unit
	pluginInterface->StartUndo(lineStart, lineEnd, "Commenter");

	int32 lineEndDelta = - CUncomment(pluginInterface);
	// if already have comments, remove them
	if (!lineEndDelta)
		// else add new onew
		lineEndDelta = CPlusCommentSelection(pluginInterface);

	// select all the text we commented
	pluginInterface->Select(lineStart, lineEnd + lineEndDelta);

	// we are done with this undo
	pluginInterface->SealUndo();
	pluginInterface->ScrollToSelection();
}

static void
CCommentOrUncommentSelection(PluginInterface *pluginInterface)
{
	int32 selStart, selEnd;
	pluginInterface->GetSelection(&selStart, &selEnd);
	int32 lineStart = pluginInterface->StartOfLine(selStart);
	int32 lineEnd = pluginInterface->EndOfLine(selEnd);
	
	// set up a composite undo record; all the successive inserts and
	// removes will be undone as a single unit
	pluginInterface->StartUndo(lineStart, lineEnd, "Commenter");

	int32 lineEndDelta = - CUncomment(pluginInterface);
	// if already have comments, remove them
	if (!lineEndDelta)
		// else add new onew
		lineEndDelta = CCommentSelection(pluginInterface);

	// select all the text we commented
	pluginInterface->Select(lineStart, lineEnd + lineEndDelta);

	// we are done with this undo
	pluginInterface->SealUndo();
	pluginInterface->ScrollToSelection();
}

static bool
CPlusCommentsDefault(PluginInterface *pluginInterface)
{
	DocumentLanguageType type = pluginInterface->LanguageType();
	return type != kCC;
}

static void
RunPopUp(PluginInterface *pluginInterface)
{
	// build a menu
	BPopUpMenu *menu = new BPopUpMenu("commenterPopUp", false, false);
	menu->SetFont(be_plain_font);
		
	BMenuItem *item = new BMenuItem("//", 0);
	menu->AddItem(item);

	item = new BMenuItem("/*", 0);
	menu->AddItem(item);

	// figure out the location of the button and show the menu over it
	BRect rect = pluginInterface->GetButtonRect();
	BMenuItem *result = menu->Go(pluginInterface->Window()->Frame().LeftTop()
		+ BPoint(rect.left, rect.bottom));
	
	if (!result)
		return;
	
	switch (menu->IndexOf(result)) {
		case 0:
			CPlusCommentOrUncommentSelection(pluginInterface);
			break;
		case 1:
			CCommentOrUncommentSelection(pluginInterface);
			break;
	}
}

PluginInterface::PluginResult 
PluginMain(PluginInterface::PluginCallSelector selector, 
	PluginInterface *pluginInterface)
{
	PluginInterface::PluginResult result = PluginInterface::kIgnored;
	
	switch (selector) {

		case PluginInterface::kGetPluginVersion:
			// this is the first selector call a plugin will get
			// Just include the following two lines to let Eddie know the
			// version of plugin API you support and the version you require
			// from Eddie
			//
			// this is the only selector call your plugin is required to support
			//
			pluginInterface->ReturnSupportedPluginVersion();
			pluginInterface->ReturnRequiredPluginVersion();
			result = PluginInterface::kAccepted;
			break;

		case PluginInterface::kInit:
			// a button may allocate any permanent data structures it will
			// need during it's lifetime and initialize it's state
			// 
			// ask for any non-default selector calls you may need
			// these include:
			// kButtonDraw		- if you want to draw the button yourself instead of just
			//					  passing an icon
			// kPulse			- if you want to get called periodically
			// kMessage			- if you want to get a peek at every message sent to the
			//					  text view
			// kMouseDown		- if you want to get called every time the mouse is
			//					  pressed over a text view
			// kKeyDown			- if you want to get called every time a key is hit in a
			//					  text view
			// kDocumentChanged	- if you want to get called every time the document changes
			//
			
			pluginInterface->RequireSelectors(PluginInterface::kRequestDocumentChangedMask
				/* | other possible masks for the other selector calls*/);
		
			// if your plugin supports any keyboard shortcuts, Init is used to publish the
			// editor primitives used by the shortcuts:
			//
			// Note that we publish several different flavors of the same primitive with
			// small differences in behavior.
			// This allows the user to choose the ones that best fit their preference
			// and fit in with Eddie's high level of configurability
			// Having too many primitives does not affect the editor's preformance too much,
			// the only practical limitation is that of the shared namespace
			pluginInterface->RegisterHandledPrimitiveCommand("UncommentOrCPlusComment",
				"If selection commented, remove comments, else comment using C++ style comments");
			pluginInterface->RegisterHandledPrimitiveCommand("UncommentOrCComment",
				"If selection commented, remove comments, else comment using C style comments");
			pluginInterface->RegisterHandledPrimitiveCommand("UncommentOrPreferredComment",
				"If selection commented, remove comments, else use preferred comments, "
				"based on document type");
			pluginInterface->RegisterHandledPrimitiveCommand("UncommentOrAlternateComment",
				"If selection commented, remove comments, else use alternate comments, "
				"based on document type; (C comments for C++ documents and vice versa)");

			result = PluginInterface::kAccepted;
			break;

		case PluginInterface::kWindowOpened:
			// a new window opened
			// the pointer to the window may be obtained from
			// pluginInterface->Window()
			//
			// if your plugin has a button, now is the time to make sure it gets instaled
			pluginInterface->SetupPluginButton(kNormalBits, kPressedBits);			
			// the above is for a simple solid button, if you want a split button, you
			// may use the following:
			// pluginInterface->SetupPluginButton(normalBitmap, pressedBitmap, kVerticallySplitButton);
			// and include the selector for the splitting you need
			//
			result = PluginInterface::kAccepted;
			break;

		case PluginInterface::kButtonClick:
			// the plugin button was clicked
			// if you have a split button, use the call:
			//	pluginInterface->ClickSelector()
			// to determine which part was clicked

			// option key will alternate between using the regular and alternate comments
			bool useAlternate = (modifiers() & B_OPTION_KEY) != 0;

			// use pluginInterface->LanguageType() in CPlusCommentsDefault to see if
			// document looks like C++ (the only thing that doesnt is C only files with .c suffix)
			
			
			if (useAlternate == CPlusCommentsDefault(pluginInterface))
				CCommentOrUncommentSelection(pluginInterface);
			else
				CPlusCommentOrUncommentSelection(pluginInterface);

			result = PluginInterface::kAccepted;
			break;

		case PluginInterface::kButtonPress:
			RunPopUp(pluginInterface);
			result = PluginInterface::kAccepted;
			break;

		case PluginInterface::kPrimitiveInvoked:
			// handle a primitive from a keyboard command
			const char *primitive = pluginInterface->CurrentPrimitive();
			
			// figure out which of our multiple shortcuts was pressed
			if (strcmp(primitive, "UncommentOrCPlusComment") == 0)
				CPlusCommentOrUncommentSelection(pluginInterface);
			else if (strcmp(primitive, "UncommentOrCComment") == 0)
				CCommentOrUncommentSelection(pluginInterface);
			else if (strcmp(primitive, "UncommentOrPreferredComment") == 0) {
				if (CPlusCommentsDefault(pluginInterface))
					CPlusCommentOrUncommentSelection(pluginInterface);
				else
					CCommentOrUncommentSelection(pluginInterface);
			} else if (strcmp(primitive, "UncommentOrAlternateComment") == 0) {
				if (CPlusCommentsDefault(pluginInterface))
					CCommentOrUncommentSelection(pluginInterface);
				else
					CPlusCommentOrUncommentSelection(pluginInterface);
			}
				
			// animate the button as if it was pressed by a mouse to give the user
			// a positive feedback
			pluginInterface->AnimateButtonClick();
			result = PluginInterface::kAccepted;
			break;
			
	}
	return result;
}

const char *
PluginHelp(void)
{
	return "Commenter plugin\n"
		"Click to comment out selection or uncoment if \n"
		"already commented. Uses /* style comments for\n"
		".c files, // style for everything else. Hold\n"
		"down Option to use the alternate commenting style.";
}

// bitmaps for the plugin button
// created by using the icon editor in FileTypes and using the Dump Bitmap
// feature

const unsigned char kNormalBits [] = {
	0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1c,
	0x1f,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1c,0x17,
	0x1f,0x1e,0x1c,0x1c,0x1c,0x1c,0x1c,0x2d,0x14,0x1c,0x1c,0x1c,0x1c,0x1c,0x1a,0x17,
	0x1f,0x1e,0x1c,0x1c,0x1c,0x1c,0x1c,0x2d,0x14,0x1c,0x1c,0x1c,0x1c,0x1c,0x1a,0x17,
	0x1f,0x1e,0x1c,0x1c,0x1c,0x1c,0x2d,0x14,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1a,0x17,
	0x1f,0x1e,0x1c,0x1c,0x1c,0x1c,0x2d,0x14,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1a,0x17,
	0x1f,0x1e,0x1c,0x1c,0x1c,0x2d,0x14,0x1c,0x1c,0x1c,0x2d,0x14,0x1c,0x1c,0x1a,0x17,
	0x1f,0x1e,0x1c,0x1c,0x1c,0x2d,0x14,0x1c,0x2d,0x1c,0x2d,0x14,0x2d,0x14,0x1a,0x17,
	0x1f,0x1e,0x1c,0x1c,0x2d,0x14,0x1c,0x1c,0x1c,0x2d,0x2d,0x2d,0x14,0x14,0x1a,0x17,
	0x1f,0x1e,0x1c,0x1c,0x2d,0x14,0x1c,0x1c,0x2d,0x14,0x2d,0x14,0x2d,0x14,0x1a,0x17,
	0x1f,0x1e,0x1c,0x2d,0x14,0x1c,0x1c,0x1c,0x14,0x1c,0x2d,0x14,0x14,0x14,0x1a,0x17,
	0x1f,0x1e,0x1c,0x2d,0x14,0x1c,0x1c,0x1c,0x1c,0x1c,0x14,0x14,0x1c,0x1c,0x1a,0x17,
	0x1f,0x1e,0x2d,0x14,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1a,0x17,
	0x1f,0x1e,0x2d,0x14,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1a,0x17,
	0x1f,0x1c,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x17,0x17,
	0x1c,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x15
};

const unsigned char kPressedBits [] = {
	0x15,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x1c,
	0x17,0x15,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1a,0x1c,0x1f,
	0x17,0x1a,0x1c,0x1c,0x1c,0x1c,0x1c,0x2d,0x14,0x1c,0x1c,0x1c,0x1c,0x1c,0x1e,0x1f,
	0x17,0x1a,0x1c,0x1c,0x1c,0x1c,0x1c,0x2d,0x14,0x1c,0x1c,0x1c,0x1c,0x1c,0x1e,0x1f,
	0x17,0x1a,0x1c,0x1c,0x1c,0x1c,0x2d,0x14,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1e,0x1f,
	0x17,0x1a,0x1c,0x1c,0x1c,0x1c,0x2d,0x14,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1e,0x1f,
	0x17,0x1a,0x1c,0x1c,0x1c,0x2d,0x14,0x1c,0x1c,0x1c,0x2d,0x14,0x1c,0x1c,0x1e,0x1f,
	0x17,0x1a,0x1c,0x1c,0x1c,0x2d,0x14,0x1c,0x2d,0x1c,0x2d,0x14,0x2d,0x14,0x1e,0x1f,
	0x17,0x1a,0x1c,0x1c,0x2d,0x14,0x1c,0x1c,0x1c,0x2d,0x2d,0x2d,0x14,0x14,0x1e,0x1f,
	0x17,0x1a,0x1c,0x1c,0x2d,0x14,0x1c,0x1c,0x2d,0x14,0x2d,0x14,0x2d,0x14,0x1e,0x1f,
	0x17,0x1a,0x1c,0x2d,0x14,0x1c,0x1c,0x1c,0x14,0x1c,0x2d,0x14,0x14,0x14,0x1e,0x1f,
	0x17,0x1a,0x1c,0x2d,0x14,0x1c,0x1c,0x1c,0x1c,0x1c,0x14,0x14,0x1c,0x1c,0x1e,0x1f,
	0x17,0x1a,0x2d,0x14,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1e,0x1f,
	0x17,0x1a,0x2d,0x14,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1e,0x1f,
	0x17,0x1c,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x3f,0x1f,
	0x1c,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x3f
};


