/*============================================================================
/ attr.c
/ © Michael Pieper 30. Oct. 1999
/============================================================================
/
/	This c-File includes all "Attribute"-Functions for the PSION-FileSystem
/
/============================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <TypeConstants.h>

/*===========================================================================
/	We also need our own Headerfiles
/============================================================================*/

#include "psionfs.h"
#include "debug.h"
#include "attr.h"

/*===========================================================================
/	to walk to our mime-declarations, we need a mime-structure
/============================================================================*/

struct ext_mime {
	char *extension;
	char *mime;
};

/*===========================================================================
/	We have to declare each file with an MIME-Type. 0,0 declares the end of the list
/============================================================================*/

static struct ext_mime mimes[] = {
	{ ".agn", "application/x-vnd.mp.psion-agn" },	/* The AGENDA File		*/

	{ ".app", "application/x-vnd.mp.psion-app" },	/* executable files		*/
	{ ".img", "application/x-vnd.mp.psion-img" },
	{ ".opa", "application/x-vnd.mp.psion-opa" },
	{ ".opl", "application/x-vnd.mp.psion-opl" },
	{ ".opo", "application/x-vnd.mp.psion-opo" },

	{ ".dbf", "application/x-vnd.mp.psion-dbf" },	/* Database file		*/
	{ ".pic", "application/x-vnd.mp.psion-pic" },	/* picture file			*/
	{ ".spr", "application/x-vnd.mp.psion-spr" },	/* Spreadsheet			*/

	{ ".jot", "application/x-vnd.mp.psion-jot" },	/* Jotter Database file	*/
	{ ".wld", "application/x-vnd.mp.psion-wld" },	/* World File			*/

	{ ".wrd", "text/x-vnd.mp.psion-wrd" },			/* Word File			*/
	{ ".txt", "text/x-vnd.mp.psion-txt" },			/* Text File			*/

	{ ".wve", "audio/x-vnd.mp.psion-wve" },			/* Sound File			*/

	{ ".grp", "application/x-vnd.mp.psion-grp" },	/* Groupdefinition		*/
	{ "",     "application/x-vnd.mp.psion"     },	/* every other file!	*/

	{ 0, 0 }
};

/*===========================================================================
/	setup the MIME-Attribute
/============================================================================*/

status_t set_mime_type(nspace *ns, vnode *node)
{
	struct ext_mime *p;
	size_t namelen, ext_len;

	if (node->mode & S_IFDIR) return B_ERROR;	/* if it is a folder, then we will do nothing!	*/
	
	namelen = strlen(node->filename);

	for (p = mimes; p->extension; p++) {
		ext_len = strlen(p->extension);

		if (namelen <= ext_len) continue;
		if (strcasecmp(&node->filename[namelen - ext_len], p->extension) == 0) break;
	}

	if (p->mime != NULL) {
		psionfs_remove_attr(ns, node, BEOS_MIME_NAME);
		namelen = strlen(p->mime)+1;
		psionfs_write_attr(ns, node, BEOS_MIME_NAME, BEOS_MIME_TYPE, p->mime, &namelen, (off_t)0);
		return B_OK;
	} else return B_ERROR;
}

/*===========================================================================
/	open the attribute directory to walk through alle attributes, which
/ 	depends to one file
/============================================================================*/

int	psionfs_open_attrdir	(void *_ns, void *_node, void **cookie) {
	char text[128];
	nspace *ns = (nspace *)_ns;
	vnode *node = (vnode *)_node;

	sprintf(text, "## fs_open_attrdir vnod_id:%Ld,\n", node->id);
	debug_print(text);
	
	if (node->attr == NULL) return EINVAL;	/* no attribute available	*/
	if ((*cookie = malloc(sizeof(int))) == NULL) 
		return ENOMEM;						/* no memory for an int	*/
	
/*	acquire_sem(ns->dev_sem);	*/			/* we don't need this really, because everything is threadsave here!	*/
	*(int *)(*cookie) = 0;					/* set the pointer to the first attribute	*/
/*	release_sem(ns->dev_sem);	*/
	
	return B_OK;
}

/*===========================================================================
/	close the attribute directory. Reset the cookie
/============================================================================*/

int	psionfs_close_attrdir	(void *_ns, void *_node, void *cookie) {
	char text[128];
	nspace *ns = (nspace *)_ns;
	vnode *node = (vnode *)_node;

	sprintf(text, "## fs_close_attrdir vnod_id:%Ld,\n", node->id);
	debug_print(text);

/*	acquire_sem(ns->dev_sem);	*/	/* we don't need this really, because everything is threadsave here!	*/
	*(int *)cookie = 0;					/* set the pointer to the first attribute	*/
/*	release_sem(ns->dev_sem);	*/

	return B_OK;			
}

/*===========================================================================
/	free the memory for the attrdircookie
/============================================================================*/

int psionfs_free_attrdircookie	(void *_ns, void *_node, void *cookie) {
	char text[128];
	vnode *node = (vnode *)_node;
	nspace *ns = (nspace *)_ns;
	int ret = EINVAL;
	
	sprintf(text, "## fs_free_attrdircookie vnod_id:%Ld,\n", node->id);
	debug_print(text);
	
	acquire_sem(ns->dev_sem);		/* we don't need it really.	But get the semaphore! */
	if (cookie != NULL) {
		free(cookie);	/* free the cookie	*/
		ret = B_OK;
	} 
	release_sem(ns->dev_sem);		/* Release the semaphore!	*/
	return ret;
}

/*===========================================================================
/	rewind the attrdir. Simple let the cookie point to the first attr-entry
/============================================================================*/

int	psionfs_rewind_attrdir		(void *_ns, void *_node, void *cookie) {
	char text[128];
	vnode *node = (vnode *)_node;
	nspace *ns = (nspace *)_ns;
	int ret = EINVAL;
	
	sprintf(text, "## fs_rewind_attrdir vnod_id:%Ld,\n", node->id);
	debug_print(text);
	
	acquire_sem(ns->dev_sem);		/* we don't need it really.	But get the semaphore! */
	if (cookie != NULL) {
		*(int *)cookie = 0;
		ret = B_OK;
	}
	release_sem(ns->dev_sem);		/* Release the semaphore!	*/
	return ret;
}

/*===========================================================================
/	read the attrdir. and send back alle the informations in a dirent infoblock
/============================================================================*/

int	psionfs_read_attrdir			(void *_ns, void *_node, void *cookie, long *num, 
								struct dirent *buf, size_t bufsize) {
	char text[128];
	vnode *node = (vnode *)_node;
	nspace *ns = (nspace *)_ns;
	psion_attr	*at;
	int i;

	sprintf(text, "## fs_read_attrdir vnod_id:%Ld bufsize:%ld\n", node->id, bufsize);
	debug_print(text);

	acquire_sem(ns->dev_sem);		/* noone should change the pointers from now! */

	at = node->attr;
	for (i = 0; (i < *(int *)cookie) && (at != NULL); i++, at = at->next) ;

	if (at != NULL) {
		buf->d_ino = node->id;
		buf->d_dev = ns->id;
		buf->d_reclen = at->attri.size;
		strcpy(buf->d_name, at->name);
		*num = 1;
		(*(int *)cookie)++;
	} else {
		debug_print("-- fs_read_attrdir kein Attribut mehr!\n");
		*num = 0;
	}
	release_sem(ns->dev_sem);		/* ok, we've done all things, free the semaphore	*/
	return B_OK;
}

/*===========================================================================
/	Write the attribute. We have to delete the original Attribute before writing!
/============================================================================*/

int	psionfs_write_attr	(void *_ns, void *_node, const char *name, int type, 
							const void *buf, size_t *len, off_t pos) {
	char text[128];
	char *b;
	vnode *node = (vnode *)_node;
	nspace *ns = (nspace *)_ns;
	psion_attr *at = NULL;		/* NULL is the flag for "not found" */

	sprintf(text, "## fs_write_attr Name:%s vnod_id:%Ld\n", name, node->id);
	debug_print(text);

	acquire_sem(ns->dev_sem);		/* noone should change the pointers from now! */
	at = node->attr;
	while ((at != NULL) && (strcmp(name, at->name) != 0)) at = at->next;

	if (at == NULL) {
		debug_print("-- fs_stat_attr Neues Attribut notwendig!\n");
		at = (psion_attr *)malloc(sizeof(psion_attr));	/* we have to allocate the structure itself */
		if (at == NULL) {
			release_sem(ns->dev_sem);		/* ok, we've done all things, free the semaphore	*/
			return ENOMEM;					/* not enough memory!	*/
		}
		
		at->name = (char *)malloc(strlen(name) + 1);	/* we need memory for the attribute name */
		if (at->name == NULL) {							/* if its not enough available -> cleanup */
			free(at);
			release_sem(ns->dev_sem);		/* ok, we've done all things, free the semaphore	*/
			return ENOMEM;
		}		
		strcpy(at->name, name);					/* copy the Attribute Name */
		
		at->data = (char *)malloc(pos + *len);	/* we also have to allocate memory for the data */
		if (at->data == NULL) {					/* if its not enough available -> cleanup */
			free(at->name);
			free(at);
			release_sem(ns->dev_sem);		/* ok, we've done all things, free the semaphore	*/
			return ENOMEM;
		}		
		memcpy(&at->data[pos], buf, *len);	/* copy the data */
		at->attri.type = type;
		at->attri.size = pos + *len;
		at->next = node->attr;
		node->attr = at;					/* connect the new attribute to the vnode */
	} else {
		if (at->attri.size < pos + *len) {	/* we want to write more data then memory is allocated */
			b = at->data;					/* save the old data pos. */
			at->data = (char *)malloc(pos + *len);
			if (at->data == NULL) {
				at->data = b;
				release_sem(ns->dev_sem);		/* ok, we've done all things, free the semaphore	*/
				return ENOMEM;
			}
			memcpy(at->data, b, at->attri.size);	/* copy the old data */
			free(b);								/* free the old data */
		}
		memcpy(&at->data[pos], buf, *len);			/* copy the new data */
		at->attri.size = pos + *len;
	}
	release_sem(ns->dev_sem);		/* ok, we've done all things, free the semaphore	*/
	return B_NO_ERROR;
}

/*===========================================================================
/	Read the Attribute. So fill up the buffer with *len Bytes
/============================================================================*/

int psionfs_read_attr	(void *_ns, void *_node, const char *name, int type, 
							void *buf, size_t *len, off_t pos)
{
	char text[128];
	psion_attr *at;
	int ret = ENOENT;
	
	nspace *ns = (nspace *)_ns;
	vnode *node = (vnode *)_node;

	sprintf(text, "## fs_read_attr Name:%s Type:%c%c%c%c vnod_id:%Ld,\n", name, type>>24, type>>16, type>>8, type, node->id);
	debug_print(text);

	acquire_sem(ns->dev_sem);		/* noone should change the pointers from now! */
	at = node->attr;
	while ((at != NULL) && (strcmp(at->name, name) != 0)) at = at->next;	/* we search for our attribute! */

	if (at != NULL) {
		*len = MIN(*len, at->attri.size - pos);
		memcpy(buf, &at->data[pos], *len);
		ret = B_NO_ERROR;		
	} else {
		*len = 0;
	}
	release_sem(ns->dev_sem);
	return ret;
}

/*===========================================================================
/	remove the attribute.
/============================================================================*/

int	psionfs_remove_attr	(void *_ns, void *_node, const char *name) {
	char text[128];
	nspace *ns = (nspace *)_ns;
	vnode *node = (vnode *)_node;
	psion_attr	*at, *old = NULL;
	int ret = ENOENT;
	
	sprintf(text, "## fs_remove_attr Name:%s vnod_id:%Ld,\n", name, node->id);
	debug_print(text);
	
	acquire_sem(ns->dev_sem);		/* noone should change the pointers from now! */
	at = node->attr;						/* search for the attribute */
	if (at != NULL) {						/* there is no attribute	*/
		if (strcmp(at->name, name) == 0) {	/* if the first attribute is the one	*/
			old = node->attr;				/* remember the old node	*/
			node->attr = at->next;			/* remove the node from the list	*/
		} else {
			while (at != NULL) {			/* maybe the attribut is deeper in the list	*/
				if ((at->next != NULL) && (strcmp(at->next->name, name) == 0)) {
					old = at->next;				/* remember the old node	*/
					at->next = at->next->next;	/* remove the attribute from the list	*/
				}
				at = at->next;				/* next attribute	*/
			}
		}
	}
	if (old != NULL) {
		if (old->name != NULL) free(old->name);	/* free the Attribute name */
		if (old->data != NULL) free(old->data);	/* free the Attribute Data */
		free(old);								/* free the own pointer */
		ret = B_NO_ERROR;
	}
	release_sem(ns->dev_sem);					/* let the other threads continue	*/
	return ret;
}

/*===========================================================================
/	rename the Attribute
/============================================================================*/

int	psionfs_rename_attr	(void *_ns, void *_node, const char *oldname, const char *newname) {
	char text[128], *n;
	nspace *ns = (nspace *)_ns;
	vnode *node = (vnode *)_node;
	psion_attr	*at;
	int ret = ENOENT;
	
	sprintf(text, "## fs_rename_attr old:%s new:%s vnod_id:%Ld,\n", oldname, newname, node->id);
	debug_print(text);

	acquire_sem(ns->dev_sem);		/* noone should change the pointers from now! */
	at = node->attr;
	while ((at != NULL) && (strcmp(at->name, oldname) != 0)) at = at->next;	/* we search for our attribute! */
	if (at != NULL) {				/* we found our old attribute	*/
		n = (char *)malloc(strlen(newname) + 1);	/* we need memory for the new attribute name */
		if (n != NULL) {							/* its enough available  */
			strcpy(n, newname);						/* copy the new name	*/
			free(at->name);							/* free the old memory	*/
			at->name = n;							/* point to the new name	*/
			ret = B_OK;								/* all worked OK	*/
		} else {									/* not enough memory!	*/
			ret = ENOMEM;							/* report this	*/
		}
	}
	release_sem(ns->dev_sem);		/* let the other threads continue	*/
	return ret;
}

/*===========================================================================
/	send back the information of the Attribute. (Size and type)
/============================================================================*/

int psionfs_stat_attr(void *_ns, void *_node, const char *name, struct attr_info *buf) {
	char text[128];
	psion_attr	*at;
	int ret = ENOENT;
	
	nspace *ns = (nspace *)_ns;
	vnode *node = (vnode *)_node;

	sprintf(text, "## fs_stat_attr Name:%s vnod_id:%Ld,\n", name, node->id);
	debug_print(text);
	
	acquire_sem(ns->dev_sem);		/* noone should change the pointers from now! */
	at = node->attr;
	while ((at != NULL) && (strcmp(at->name, name) != 0)) at = at->next;
	
	if (at != NULL) {
		debug_print("-- fs_stat_attr Attribut gefunden!\n");
		buf->type = at->attri.type;
		buf->size = at->attri.size;
		ret = B_NO_ERROR;
	}
	release_sem(ns->dev_sem);					/* let the other threads continue	*/
	return ret;
}

/*===========================================================================
/	This function walks through all Attribute-Structures and frees the memory
/============================================================================*/

void free_all_attr(psion_attr *at) {
	psion_attr *at2;
	
	while (at != NULL) {
		if (at->name != NULL) free(at->name);	/* free the Attribute name */
		if (at->data != NULL) free(at->data);	/* free the Attribute Data */
		at2 = at->next;							/* save the pointer to the next Attribute */
		free(at);								/* free the own pointer */
		at = at2;								/* go on with the next-Attribute */
	}
}
