/* beosserial.cpp: tty line interface code for Pilot serial comms under BeOS
 *
 * Copyright (c) 1999 Marc Schefer
 * Based on code Copyright (c) 1996, 1997, D. Jeff Dionne & Kenneth Albanowski.
 *
 * This is free software, licensed under the GNU Library Public License V2.
 * See the file COPYING.LIB for details.
 */

#include <stdio.h>
#include <SerialPort.h>
#include "pi-source.h"
#include "pi-socket.h"
#include "pi-serial.h"
#include "pi-slp.h"
#include "pi-syspkt.h"
#include "pi-padp.h"

static data_rate calcrate(int baudrate)
{
#ifdef B300
  if(baudrate == 300) return B_300_BPS;
#endif
#ifdef B1200
  if(baudrate == 1200) return B_1200_BPS;
#endif
#ifdef B2400
  if(baudrate == 2400) return B_2400_BPS;
#endif
#ifdef B4800
  if(baudrate == 4800) return B_4800_BPS;
#endif
#ifdef B9600
  if(baudrate == 9600) return B_9600_BPS;
#endif
#ifdef B19200
  else if(baudrate == 19200) return B_19200_BPS;
#endif
#ifdef B38400
  else if(baudrate == 38400) return B_38400_BPS;
#endif
#ifdef B57600
  else if(baudrate == 57600) return B_57600_BPS;
#endif
#ifdef B115200
  else if(baudrate == 115200) return B_115200_BPS;
#endif
#ifdef B230400
  else if(baudrate == 230400) return B_230400_BPS;
#endif
  else {
    printf("Unable to set baud rate %d\n", baudrate);
    abort(); // invalid baud rate
  }
}

static int s_changebaud(struct pi_socket *ps);
static int s_close(struct pi_socket *ps);
static int s_write(struct pi_socket *ps);
static int s_read(struct pi_socket *ps, int timeout);

int pi_serial_open(struct pi_socket *ps, struct pi_sockaddr * addr, int addrlen)
{
  char * tty = addr->pi_device;
  
  // If no serial port name was specified try to get it
  // from the PILOTPORT environnement variable

  if ((!tty) || !strlen(tty))
    tty = getenv("PILOTPORT");
  if (!tty)
    tty = "<Null>";

  ps->pilotPort = new BSerialPort;
  
  if( ps->pilotPort->Open( tty ) <= 0 )
  {
    delete ps->pilotPort;
    return -1;
  }
  
  ps->pilotPort->SetDataRate( calcrate(ps->rate) );
   
#ifndef NO_SERIAL_TRACE
  if(ps->debuglog) {
    ps->debugfd = open(ps->debuglog,O_WRONLY|O_CREAT|O_APPEND,0666);
    // This sequence is magic used by my trace analyzer - kja -/
    write(ps->debugfd, "\0\1\0\0\0\0\0\0\0\0", 10);
  }
#endif

  ps->serial_close = s_close;
  ps->serial_read = s_read;
  ps->serial_write = s_write;
  ps->serial_changebaud = s_changebaud;
  
  return 1; // The other implementations return the file descriptor here
            // I hope this will do
}

static int s_changebaud(struct pi_socket *ps)
{
  ps->pilotPort->SetDataRate( calcrate(ps->rate) );
  return 0;
}

static int s_close(struct pi_socket *ps)
{
  if( ps->pilotPort )
  {
  	ps->pilotPort->Close();
    delete ps->pilotPort;
    ps->pilotPort = NULL;
  }

#ifndef NO_SERIAL_TRACE
  if (ps->debugfd)
    close(ps->debugfd);  
#endif

  return 0;
}

static int s_write(struct pi_socket *ps)
{
  struct pi_skb *skb;
  int nwrote, len;

#ifndef NO_SERIAL_TRACE
  int i;
#endif

  if (ps->txq) {
    ps->busy++;

    skb = ps->txq;
    ps->txq = skb->next;

    len = 0;
    while (len<skb->len) {
      nwrote = 0;
      nwrote = ps->pilotPort->Write( skb->data, skb->len );
      if (nwrote<=0)
        break; // transmission failure
      len += nwrote;
    }

#ifndef NO_SERIAL_TRACE
    if (ps->debuglog)
      for (i=0;i<skb->len;i++) {
        write(ps->debugfd, "2", 1);
        write(ps->debugfd, skb->data+i, 1);
      }
#endif

    ps->tx_bytes += skb->len;
    free(skb);

    ps->busy--;
    
    return 1;
  }
  return 0;
}

// timeout is in tenth of second
static int s_read(struct pi_socket *ps, int timeout)
{
  int r;
  unsigned char *buf;

#ifndef NO_SERIAL_TRACE
  int i;
#endif
  int ready;
  
  ps->pilotPort->SetTimeout( timeout ? timeout * 100000LL : B_INFINITE_TIMEOUT );
  
  pi_serial_flush(ps); // We likely want to be in sync with tx 
  if (!ps->mac->expect) slp_rx(ps);  // let SLP know we want a packet 

  while (ps->mac->expect) {
    buf = ps->mac->buf;

    while (ps->mac->expect) {
  	  ready = ps->pilotPort->WaitForInput();

      // If data is available in time, read it 
      if( ready )
	  { 
        r = ps->pilotPort->Read( buf, ps->mac->expect );
	  }
	  else
      {
        // otherwise throw out any current packet and return 
#ifdef DEBUG
        fprintf(stderr, "Serial RX: timeout\n");
#endif
        ps->mac->state = ps->mac->expect = 1;
        ps->mac->buf = ps->mac->rxb->data;
        ps->rx_errors++;
        return 0;
      }
#ifndef NO_SERIAL_TRACE
      if (ps->debuglog)
        for (i=0;i<r;i++) {
          write(ps->debugfd, "1", 1);
          write(ps->debugfd, buf+i, 1);
        }
#endif

      ps->rx_bytes += r;
      buf += r;
      ps->mac->expect -= r;
    }
    slp_rx(ps);
  }
  return 0;
}
