/* AmToolTarget.h
 * Copyright (c)2000 by Eric Hackborn.
 * All rights reserved.
 *
 * This code is not public domain, nor freely distributable.
 * Please direct any questions or requests to Eric Hackborn,
 * at <hackborn@angryredplanet.com>.
 *
 * ----------------------------------------------------------------------
 *
 * Known Bugs
 * ~~~~~~~~~~
 *	- None!  Ha ha!
 *
 * ----------------------------------------------------------------------
 *
 * History
 * ~~~~~~~
 * 05.18.00		hackborn
 * Created this file
 */
#ifndef AMPUBLIC_AMTOOLTARGET_H
#define AMPUBLIC_AMTOOLTARGET_H

#include <vector.h>
#include <be/interface/View.h>
#include <ArpBuild.h>
#include "AmPublic/AmEvents.h"
#include "AmPublic/AmTrackHandler.h"
class AmPhraseEvent;
class AmSelectionsI;
class AmTool;
class AmTrack;

/***************************************************************************
 * AM-TOOL-TARGET
 * This utility class provides all the behaviour that tools need to
 * function.  This plus the AmTrack is essentially the API that tools have
 * available to them.
 ***************************************************************************/
class AmToolTarget : public AmTrackHandler
{
public:
	/* The default 0 is provided as a convenience in certain situations,
	 * but be aware that until you set a valid view and time converter
	 * this class will not function.
	 */
	AmToolTarget(	BView* view = 0, const AmTimeConverter* mtc = 0,
					float leftFudge = AM_NO_FUDGE, float rightFudge = AM_NO_FUDGE);
	virtual ~AmToolTarget();
	
	/* This method is reluctant admittance that tools should often be
	 * configured differently for different targets.  Implementors can
	 * check the tool's class name and use SetSettings() to set it
	 * up properly.  By default, this method resets the tool to its
	 * original state (i.e. whatever the user last selected).
	 */
	virtual void	PrepareTool(AmTool* tool);
	
	enum {
		NO_CONTAINER		= (1<<0)	// If set, this tells tool clients
										// that this target keeps its events
										// stored directly in the track's
										// phrase list.
	};
	virtual uint32	Flags() const;
	
	/*---------------------------------------------------------
	 * EVENT ACCESSING
	 *---------------------------------------------------------*/
	/* Find whatever event lies at the supplied point.  Answer that
	 * event, if any, as well as the phrase that lies at that
	 * event.  If no event is returned, then the AmPhraseEvent
	 * answer is not valid.
	 */
	AmEvent*		EventAt(const AmTrack* track,
							BPoint where,
							AmPhraseEvent** answer);
	AmEvent*		EventAt(const AmTrack* track,
							AmTime time,
							float y,
							AmPhraseEvent** answer);
	/* By default zero, subclasses can return a fudge time that increases
	 * the bounds of the EventAt() search.  This is useful for single-pixel
	 * events (like control changes) that want to give the user a little
	 * extra selection room.
	 */
	virtual AmTime	EventAtFudge() const;
	/* Answer the phrase event nearest to the supplied time.  There
	 * is a default tolerance within which the phrase must fall --
	 * if none of them do, then answer 0.
	 */
	AmPhraseEvent*	PhraseEventNear( const AmTrack* track, AmTime time );
	/* Subclasses must answer with whatever interesting event they found
	 * at the given location.
	 */
	virtual AmEvent* InterestingEventAt(const AmTrack* track,
										const AmPhrase& phrase,
										AmTime time,
										float y) const = 0;
	/* Answer true if the event intersects the MIDI data rectangle
	 * specified.  The left and right are MIDI time positions, the top
	 * and bottom are YValues for this target.  By default, answer with
	 * true if the event intersects the left and right (i.e. time) coordinates.
	 * Subclasses can augment to determine if it intersects with the top and bottom.
	 */
	virtual bool EventIntersects(	const AmEvent* event,
									AmTime left, int32 top, AmTime right, int32 bottom) const;

	/*---------------------------------------------------------
	 * EVENT MANIPULATION
	 * These methods deal with creating and modifying events.
	 *---------------------------------------------------------*/
	/* The NewEventStrategyType is used to communicate with tools
	 * about what happens when this view creates a new event.
	 */
	enum NewEventStrategyType {
	 	NO_STRATEGY,		// A default -- do nothing.
		ADD1_STRATEGY,		// Add one new event and do nothing more.
	 	ADDN_STRATEGY,		// Add events for as longs as the user is pressing the mouse button.
	 	MOVE_STRATEGY,		// After adding the event, start calling the Move() func.
		TRANSFORM_STRATEGY	// After adding the event, start calling the Transform() func.
	};
	virtual NewEventStrategyType NewEventStrategy() const = 0;
	/* Subclasses must return a new midi event of whatever data they hold.
	 * The mouse point has already been translated, so ulong is the time of
	 * the new event, and float is whatever data goes in the y value for
	 * the particular subclass.
	 */
	virtual AmEvent* NewEvent(AmTime time, float y) = 0;

	/*---------------------------------------------------------
	 * MOVING
	 * The term 'y' value is used to describe what changes in
	 * the event as it is dragged on the y axis.  Not all
	 * subclasses will actually do something with the y value --
	 * many will use the transform y value, which is just
	 * something ELSE that happens to the event on the y axis.
	 *---------------------------------------------------------*/
	/* Subclasses must answer whatever y value resides at the supplied
	 * pixel.
	 */
	virtual int32	MoveYValueFromPixel(float y) const = 0;
	virtual void	GetMoveValues(	const AmEvent* event, AmTime* x, int32* y) const = 0;
	virtual void	GetMoveDelta(	BPoint origin, BPoint where,
									AmTime* xDelta, int32* yDelta) const = 0;
	virtual void	SetMove(AmPhrase& container,
							AmEvent* event,
							AmTime originalX, int32 originalY,
							AmTime deltaX, int32 deltaY,
							uint32 flags) = 0;

	/*---------------------------------------------------------
	 * TRANSFORMATION
	 * These methods define a second type of edit that subclasses
	 * might want to perform.  These edits are utilized by the
	 * transform tool.  Ironically, unlike the primary edits,
	 * most every class should implement transform edits.  If
	 * they don't, then they basically have no data other than
	 * time that changes, and that's just weird.
	 *---------------------------------------------------------*/
	/* Subclasses must implement to get the current secondary
	 * values for the supplied event.
	 */
	virtual void GetTransformValues(AmEvent* event, int32* x, int32* y) const = 0;
	/* Subclasses must answer a transformation delta to the x and y axes, based
	 * on the distance traveled.  Obviously, if subclasses only transform
	 * along one axis, they can ignore the other.
	 */
	virtual void GetTransformDelta(	BPoint origin, BPoint where,
									int32* xDelta, int32* yDelta) const = 0;
	/* Subclasses must set the transform x and y of the given event.  Flags
	 * supplies information about what axis the user began the mouse motion
	 * with.  This is useful because some clients lock editing into the
	 * first axis the user travels along.
	 */
	enum TransformFlags {
		TRANSFORM_X			= 0x00000001,
		TRANSFORM_Y			= 0x00000002,
		TRANSFORM_BOTH		= 0x00000003	// The transform tool defaults to a transform
											// of both axes.  As soon as the user starts
											// dragging the mouse, it sets itself to whatever
											// axis the user is moving along.  This allows
											// programmatic changes, which don't set the axis,
											// to occur along both the x and y.
	};
	/* The container that holds the event is included for any implementors
	 * that need it.
	 */
	virtual void SetTransform(	AmPhrase& container,
								AmEvent* event,
								int32 originalX, int32 originalY,
								int32 deltaX, int32 deltaY,
								uint32 flags) = 0;
	virtual void SetTransform(	AmPhrase& container,
								AmEvent* event,
								BPoint where,
								TransformFlags flags ) = 0;

	/* Subclasses should implement to actually perform the selected events.
	 * Events are performed automatically in certain situations, like
	 * when they are moved.
	 */
	virtual void Perform(const AmTrack* track, const AmSelectionsI* selections); 
	/* Obviously, this only applies if the target can actually perform.  But
	 * if it can, subclasses should take this opportunity to call Stop() on the
	 * performer.
	 */
	virtual void Stop();
	
	/*---------------------------------------------------------
	 * WINDOW INTERFACE
	 * These methods provide an interface for the window containing
	 * this tool target.
	 *---------------------------------------------------------*/
	/* Get and set the target's selections.  Never delete the selections
	 * object, if you want to clear them, send 0 to SetSelections().
	 */
	AmSelectionsI*	Selections() const;
	void			SetSelections(AmSelectionsI* selections);
	/* Get the current saturation value from the window.
	 */
	float			Saturation() const;
	
private:
	typedef AmTrackHandler		inherited;
};

#endif
