/* ArpVelocity.cpp
 */
#include "ArpVelocity.h"

#include <stdio.h>
#include <stdlib.h>
#include <be/interface/MenuField.h>
#include <be/interface/MenuItem.h>
#include <be/interface/RadioButton.h>
#include <be/interface/StringView.h>
#include <experimental/ResourceSet.h>
#include "ArpKernel/ArpDebug.h"
#include "ArpViewsPublic/ArpPrefsI.h"
#include "ArpViews/ArpIntControl.h"
#include "ArpViews/ArpKnobControl.h"
#include "ArpViews/ArpRangeControl.h"
#include "ArpLayout/ArpViewWrapper.h"
#include "AmPublic/AmFilterConfig.h"
#include "AmPublic/AmFilterConfigLayout.h"

static const uint32		ABSOLUTE_MSG		= 'iabs';
static const uint32		SCALE_MSG			= 'iscl';

static const char*		MODE_STR			= "mode";
static const char*		ABSOLUTE_STR		= "absolute";
static const char*		SCALE_STR			= "scale";

ArpMOD();

static BResourceSet gResources;
static int32 gInitResources = 0;
BResourceSet& Resources()
{
	if (atomic_or(&gInitResources, 1) == 0) {
		gResources.AddResources((void*)Resources);
		atomic_or(&gInitResources, 2);
	} else {
		while ((gInitResources&2) == 0) snooze(20000);
	}
	return gResources;
}

static float	MIN_SCALE		= 1;
static float	MAX_SCALE		= 400;

/*************************************************************************
 * _VELOCITY-CONTROL
 * A control that displays a quick-access menu and text control for editing
 * some velocity filter properties.
 *************************************************************************/
/* These are the hardcoded velocity constants.
 */
static const float	FF		= 127;
static const float	FF_F	= 116;
static const float	F		= 106;
static const float	F_MF	= 95;
static const float	MF		= 85;
static const float	MF_MP	= 75;
static const float	MP		= 64;
static const float	MP_P	= 53;
static const float	P		= 43;
static const float	P_PP	= 32;
static const float	PP		= 22;
static const float	PP_PPP	= 11;
static const float	PPP		= 1;

class _VelocityControl : public AmFilterConfig
{
public:
	_VelocityControl(	AmFilterHolderI* target,
						const BMessage& initSettings,
						const char* initialName );

	void			SetAbsolute(uchar velocity);
	void			SetScale(float scale);
	
	void			UpdateAbsolute(const BMessage& from);
	void			UpdateScale(const BMessage& from);
	void			UpdateMode(const BMessage& from);

	virtual void	AttachedToWindow();
	virtual void	MessageReceived(BMessage *msg);

private:
	typedef AmFilterConfig	inherited;
	int32			mShownMode;
	
	void			AddViews(const char* initialName);
	void			AddAbsoluteViews(float left, float top);
	void			AddScaleViews(float left, float top);
};

/*****************************************************************************
 * ARP-VELOCITY-FILTER
 *****************************************************************************/
ArpVelocityFilter::ArpVelocityFilter(	ArpVelocityFilterAddOn* addon,
										AmFilterHolderI* holder,
										const BMessage* config)
		: AmFilterI(addon),
		  mAddOn(addon), mHolder(holder),
		  mMode(ABSOLUTE_MODE), mAbsolute((uchar)F), mScale(100)
{
	if (config) PutConfiguration(config);
}

ArpVelocityFilter::~ArpVelocityFilter()
{
}

AmEvent* ArpVelocityFilter::HandleEvent(AmEvent* event, const am_filter_params* /*params*/)
{
	if (!event) return event;
	ArpVALIDATE(mAddOn != NULL && mHolder != NULL, return event);
	
	if (event->Type() == event->NOTEON_EVENT) {
		AmNoteOn* note = dynamic_cast<AmNoteOn*>(event);
		if (!note) return event;

		uint32		newVelocity = mAbsolute;
		if (mMode == ABSOLUTE_MODE) newVelocity = mAbsolute;
		else if (mMode == SCALE_MODE) newVelocity = (uint32)( note->Velocity() * (mScale / 100) );

		if (newVelocity < 0) newVelocity = 0;
		if (newVelocity > 127) newVelocity = 127;
		note->SetVelocity(newVelocity);
	}	
	return event;
}

BView* ArpVelocityFilter::NewEditView(BPoint requestedSize) const
{
	BMessage config;
	status_t err = GetConfiguration(&config);
	if (err != B_OK) return NULL;
	return new _VelocityControl( mHolder, config, Label() );
}

class ArpVelocityFilterSettings : public AmFilterConfigLayout
{
public:
	ArpVelocityFilterSettings(	AmFilterHolderI* target,
								const BMessage& initSettings)
		: AmFilterConfigLayout(target, initSettings),
		mAbsRadio(NULL), mSclRadio(NULL), mAbsInt(NULL), mSclInt(NULL)
	{
		try {
			AddLayoutChild((new ArpRunningBar("TopVBar"))
				->SetParams(ArpMessage()
					.SetInt32(ArpRunningBar::OrientationP, B_VERTICAL)
					.SetFloat(ArpRunningBar::IntraSpaceP, .5)
				)

				->AddLayoutChild((new ArpTextControl(
										SZ_FILTER_LABEL, "Label:","",
										mImpl.AttachTextControl(SZ_FILTER_LABEL)))
					->SetParams(ArpMessage()
						.SetString(ArpTextControl::MinTextStringP, "8")
						.SetString(ArpTextControl::PrefTextStringP, "8888888888")
					)
					->SetConstraints(ArpMessage()
						.SetFloat(ArpRunningBar::WeightC,3)
						.SetInt32(ArpRunningBar::FillC,ArpEastWest)
					)
				)

				->AddLayoutChild((new ArpRunningBar("ContentHBar"))
					->SetParams(ArpMessage()
						.SetInt32(ArpRunningBar::OrientationP, B_HORIZONTAL)
						.SetFloat(ArpRunningBar::IntraSpaceP, .5)
					)

					->AddLayoutChild((new ArpRunningBar("RadioVBar"))
						->SetParams(ArpMessage()
							.SetInt32(ArpRunningBar::OrientationP, B_VERTICAL)
							.SetFloat(ArpRunningBar::IntraSpaceP, .5)
						)
						->AddLayoutChild((new ArpViewWrapper(mAbsRadio = 
							new BRadioButton(BRect(0,0,10,10), "absolute_radio", "Absolute",
									new BMessage(ABSOLUTE_MSG),
									B_FOLLOW_NONE,
									B_WILL_DRAW|B_FULL_UPDATE_ON_RESIZE|B_NAVIGABLE)))
							->SetConstraints(ArpMessage()
								.SetFloat(ArpRunningBar::WeightC,1)
								.SetInt32(ArpRunningBar::FillC,ArpWest)
							)
						)
						->AddLayoutChild((new ArpViewWrapper(mSclRadio = 
							new BRadioButton(BRect(0,0,10,10), "scale_radio", "Scaled",
									new BMessage(SCALE_MSG),
									B_FOLLOW_NONE,
									B_WILL_DRAW|B_FULL_UPDATE_ON_RESIZE|B_NAVIGABLE)))
							->SetConstraints(ArpMessage()
								.SetFloat(ArpRunningBar::WeightC,1)
								.SetInt32(ArpRunningBar::FillC,ArpWest)
							)
						)
					)
				
					->AddLayoutChild((new ArpRunningBar("TextVBar"))
						->SetParams(ArpMessage()
							.SetInt32(ArpRunningBar::OrientationP, B_VERTICAL)
							.SetFloat(ArpRunningBar::IntraSpaceP, .5)
						)
						->AddLayoutChild((mAbsInt = new ArpIntControl(
												"absolute", "Absolute:",
												mImpl.AttachControl("absolute"),
												0, 127))
							->SetConstraints(ArpMessage()
								.SetFloat(ArpRunningBar::WeightC,3)
								.SetInt32(ArpRunningBar::FillC,ArpEastWest)
							)
						)
						->AddLayoutChild((mSclInt = new ArpIntControl(
												"scale", "Scale:",
												mImpl.AttachControl("scale"),
												MIN_SCALE, MAX_SCALE))
							->SetConstraints(ArpMessage()
								.SetFloat(ArpRunningBar::WeightC,3)
								.SetInt32(ArpRunningBar::FillC,ArpEastWest)
							)
						)
					)
				)
			);
		} catch(...) {
			throw;
		}
		Implementation().RefreshControls(mSettings);
		RefreshControls(initSettings);
	}

	virtual	void MessageReceived(BMessage *msg)
	{
		switch (msg->what) {
			case ABSOLUTE_MSG:
				{
					BMessage		upd;
					if (upd.AddInt32( "mode", ArpVelocityFilter::ABSOLUTE_MODE ) == B_OK)
						Implementation().SendConfiguration(&upd);
					if (mAbsInt) mAbsInt->SetEnabled(true);
					if (mSclInt) mSclInt->SetEnabled(false);
				}
				break;
			case SCALE_MSG:
				{
					BMessage		upd;
					if (upd.AddInt32( "mode", ArpVelocityFilter::SCALE_MODE ) == B_OK)
						Implementation().SendConfiguration(&upd);
					if (mAbsInt) mAbsInt->SetEnabled(false);
					if (mSclInt) mSclInt->SetEnabled(true);
				}
				break;
			case ARP_PUT_CONFIGURATION_MSG:
				{
					BMessage	settings;
					if (msg->FindMessage("settings", &settings) == B_OK)
						RefreshControls(settings);
				}
				// Note: no break on purpose
			default:
				inherited::MessageReceived( msg );
		}
	}

protected:
	typedef AmFilterConfigLayout inherited;

private:
	BRadioButton*		mAbsRadio;
	BRadioButton*		mSclRadio;
	ArpIntControl*		mAbsInt;
	ArpIntControl*		mSclInt;
	
	void RefreshControls(const BMessage& settings)
	{
		int32		mode;
		if (settings.FindInt32( "mode", &mode ) == B_OK) {
			if (mode == ArpVelocityFilter::ABSOLUTE_MODE) {
				if (mAbsRadio) mAbsRadio->SetValue(B_CONTROL_ON);
				if (mAbsInt) mAbsInt->SetEnabled(true);
				if (mSclInt) mSclInt->SetEnabled(false);
			}
			if (mode == ArpVelocityFilter::SCALE_MODE) {
				if (mSclRadio) mSclRadio->SetValue(B_CONTROL_ON);
				if (mAbsInt) mAbsInt->SetEnabled(false);
				if (mSclInt) mSclInt->SetEnabled(true);
			}
		}
	}
};

status_t ArpVelocityFilter::GetConfiguration(BMessage* values) const
{
	status_t err = AmFilterI::GetConfiguration(values);
	if (err != B_OK) return err;
	
	if( values->AddInt32(MODE_STR, mMode) != B_OK ) return B_ERROR;
	if( values->AddInt32(ABSOLUTE_STR, mAbsolute) != B_OK ) return B_ERROR;
	if( values->AddFloat(SCALE_STR, mScale) != B_OK ) return B_ERROR;
	return B_OK;
}

status_t ArpVelocityFilter::PutConfiguration(const BMessage* values)
{
	status_t err = AmFilterI::PutConfiguration(values);
	if (err != B_OK) return err;
	
	int32	i;
	if (values->FindInt32(MODE_STR, &i) == B_OK) mMode = i;
	if (values->FindInt32(ABSOLUTE_STR, &i) == B_OK) mAbsolute = i;
	float	f;
	if (values->FindFloat(SCALE_STR, &f) == B_OK) {
		if (f < MIN_SCALE) mScale = MIN_SCALE;
		else if (f > MAX_SCALE) mScale = MAX_SCALE;
		else mScale = f;
	}
	return B_OK;
}

status_t ArpVelocityFilter::Configure(ArpVectorI<BView*>& panels)
{
	BMessage config;
	status_t err = GetConfiguration(&config);
	if (err != B_OK) return err;
	panels.push_back(new ArpVelocityFilterSettings(mHolder, config));
	return B_OK;
}

/*****************************************************************************
 * ARP-VELOCITY-FILTER-ADDON
 *****************************************************************************/
void ArpVelocityFilterAddOn::GetVersion(int32* major, int32* minor) const
{
	*major = 1;
	*minor = 1;
}

BBitmap* ArpVelocityFilterAddOn::Image(BPoint requestedSize) const
{
	const BBitmap* bm = Resources().FindBitmap("Class Icon");
	if (bm) return new BBitmap(bm);
	return NULL;
}

AmFilterI* ArpVelocityFilterAddOn::NewInstance(	AmFilterHolderI* holder,
												const BMessage* config)
{
	return new ArpVelocityFilter( this, holder, config );
}

extern "C" _EXPORT AmFilterAddOn* make_nth_filter(int32 n, image_id /*you*/,
												  const void* cookie, uint32 /*flags*/, ...)
{
	if (n == 0) return new ArpVelocityFilterAddOn(cookie);
	return NULL;
}

/*************************************************************************
 * _VELOCITY-CONTROL
 *************************************************************************/
static const uint32		KNOB_MSG	= 'knob';

_VelocityControl::_VelocityControl(	AmFilterHolderI* target,
									const BMessage& initSettings,
									const char* initialName )
		: inherited(target, initSettings, be_plain_font,
					B_FOLLOW_LEFT | B_FOLLOW_TOP, B_WILL_DRAW ),
		  mShownMode(-1)
{
	AddViews( initialName );
	UpdateAbsolute(initSettings);
	UpdateScale(initSettings);
	UpdateMode(initSettings);
	float	right = 0, bottom = 0;
	BView*	view;
	for( view = ChildAt(0); view != 0; view = view->NextSibling() ) {
		BRect	f = view->Frame();
		if( f.right > right ) right = f.right;
		if( f.bottom > bottom ) bottom = f.bottom;
	}
	ResizeTo( right, bottom );
	for( view = ChildAt(0); view != 0; view = view->NextSibling() ) {
		BRect	f = view->Frame();
		view->MoveTo(BPoint(floor((right-f.Width())/2), f.top));
	}
}

void _VelocityControl::SetAbsolute(uchar velocity)
{
	BMessage upd;
	if( upd.AddInt32(ABSOLUTE_STR, velocity) == B_OK ) {
		Implementation().SendConfiguration(&upd);
	}
}

void _VelocityControl::SetScale(float scale)
{
	BMessage upd;
	if( upd.AddFloat(SCALE_STR, scale) == B_OK ) {
		Implementation().SendConfiguration(&upd);
	}
}

void _VelocityControl::UpdateAbsolute(const BMessage& from)
{
	int32 vel;
	if (from.FindInt32(ABSOLUTE_STR, &vel) == B_OK) {
		ArpRangeControl* r = dynamic_cast<ArpRangeControl*>(FindView("velocity_range"));
		if (r) r->SetZoomY((float)vel);
	}
}

void _VelocityControl::UpdateScale(const BMessage& from)
{
	float scale;
	if (from.FindFloat(SCALE_STR, &scale) == B_OK) {
		ArpKnobControl* r = dynamic_cast<ArpKnobControl*>(FindView("scale_knob"));
		if (r) r->SetValue(scale);
	}
}
	
void _VelocityControl::UpdateMode(const BMessage& from)
{
	int32 mode;
	if (from.FindInt32(MODE_STR, &mode) == B_OK) {
		if (mode != mShownMode) {
			ArpRangeControl*	range = dynamic_cast<ArpRangeControl*>( FindView( "velocity_range" ) );
			ArpKnobControl*		knob = dynamic_cast<ArpKnobControl*>( FindView( "scale_knob" ) );
			if (mode == ArpVelocityFilter::ABSOLUTE_MODE) {
				if (!knob->IsHidden(knob)) knob->Hide();
				if (range->IsHidden(range)) range->Show();
			} else {
				if (!range->IsHidden(range)) range->Hide();
				if (knob->IsHidden(knob)) knob->Show();
			}
			mShownMode = mode;
		}
	}
}

void _VelocityControl::AttachedToWindow()
{
	inherited::AttachedToWindow();
	if( Parent() ) SetViewColor( Parent()->ViewColor() );

	ArpRangeControl*	range = dynamic_cast<ArpRangeControl*>( FindView( "velocity_range" ) );
	if( range ) range->StartWatching( this, 'fini' );
	ArpKnobControl*		knob = dynamic_cast<ArpKnobControl*>( FindView( "scale_knob" ) );
	if( knob ) knob->SetTarget( this );
}

void _VelocityControl::MessageReceived(BMessage *msg)
{
	switch( msg->what ) {
		case B_OBSERVER_NOTICE_CHANGE:
			float	yValue;
			if( msg->FindFloat( "y_value", &yValue ) == B_OK ) {
				SetAbsolute( (uchar)yValue );
			}
			break;
		case KNOB_MSG:
			{
				int32		value;
				if( msg->FindInt32( "be:value", &value ) == B_OK ) {
					SetScale( value );
				}
			}
			break;
		case ARP_PUT_CONFIGURATION_MSG: {
			BMessage settings;
			msg->FindMessage("settings", &settings);
			UpdateAbsolute(settings);
			UpdateScale(settings);
			UpdateMode(settings);
		} break;
	
		default:
			inherited::MessageReceived( msg );
	}
}

void _VelocityControl::AddViews(const char* initialName)
{
	float				left = 0, top = 0;
	float				labelH = 10;
	if( initialName ) {
		float			labelW = StringWidth( initialName );
		BRect			frame;
		frame.Set( left, top, left + labelW, top + labelH );
		BStringView*	sv = new BStringView( frame, initialName, initialName );
		if( sv ) {
			sv->SetFontSize( labelH );
			AddChild( sv );
			top = top + labelH + 5;
		}
	}
	AddAbsoluteViews(left, top);
	AddScaleViews(left, top);
}

void _VelocityControl::AddAbsoluteViews(float left, float top)
{
	BResourceSet&		r = Resources();
	const BBitmap*		sizeMap = r.FindBitmap("ff");
	BRect				frame(0, 0, 20, 20);
	if( sizeMap ) frame = sizeMap->Bounds();
	frame.OffsetBy( left, top );
	// To account for the borders on the control
	frame.right += 4;
	frame.bottom += 4;

	ArpRangeControl*	ctrl = new ArpRangeControl( frame, "velocity_range", B_FOLLOW_LEFT | B_FOLLOW_TOP, 1, (float)1, ARP_DISPLAY_TEXT );
	if( ctrl ) {
		ctrl->AddVerticalIcon(			FF,		FF,			r.FindBitmap("ff") );
		ctrl->AddVerticalIntermediate(	FF-1,	F+1,		10 );
		ctrl->AddVerticalIcon(			F,		F, 			r.FindBitmap("f") );
		ctrl->AddVerticalIntermediate(	F - 1,	MF + 1, 	10 );
		ctrl->AddVerticalIcon(			MF,		MF,			r.FindBitmap("mf") );
		ctrl->AddVerticalIntermediate(	MF - 1,	MP + 1,		10 );
		ctrl->AddVerticalIcon(			MP,		MP, 		r.FindBitmap("mp" ) );
		ctrl->AddVerticalIntermediate(	MP - 1,	P + 1,		10 );
		ctrl->AddVerticalIcon(			P,		P,			r.FindBitmap("p") );
		ctrl->AddVerticalIntermediate(	P - 1,	PP + 1,		10 );
		ctrl->AddVerticalIcon(			PP,		PP,			r.FindBitmap("pp") );
		ctrl->AddVerticalIntermediate(	PP - 1,	PPP + 1,	10 );
		ctrl->AddVerticalIcon(			PPP,	PPP,		r.FindBitmap("ppp") );
		ctrl->SetFinishedMessage( new BMessage('fini') );
		ctrl->SetTextScale( 1, 1 );
		ctrl->SetTextContext( "", "" );
		ctrl->SetRangeColor( tint_color(ui_color(B_MENU_BACKGROUND_COLOR), B_DARKEN_2_TINT) );
		ctrl->Hide();
		AddChild( ctrl );
	}
}

void _VelocityControl::AddScaleViews(float left, float top)
{
	ArpPreferencesI&	p = Prefs();
	BRect				frame( left, top, left + p.Size(KNOB_RING_TIGHT_X), top + p.Size(KNOB_RING_TIGHT_Y) );
	BMessage*			msg = new BMessage( KNOB_MSG );
	if( !msg ) return;
	ArpKnobControl*		knob = new ArpKnobControl( frame, "scale_knob", msg, 1, 400, ARP_TIGHT_RING_ADORNMENT );
	if( knob ) {
		knob->Hide();
		AddChild( knob );
	}
}
