// RamDisk version 6.00 by Earl Colby Pottinger.

#include <OS.h>
#include <Drivers.h>
#include <KernelExport.h>
#include <Mime.h>
#include <stdlib.h>
#include <Errors.h>
#include <string.h>

status_t vd_open(const char *name, uint32 flags, void **cookie);
status_t vd_free(void *cookie);
status_t vd_close(void *cookie);
status_t vd_control(void *cookie, uint32 msg, void *buf, size_t size);
status_t vd_read(void *cookie, off_t pos, void *buf, size_t *count);
status_t vd_write(void *cookie, off_t pos, const void *buf, size_t *count);

enum { RAM_SECTORS =  512, // Number of blocks per track.
       RAM_TRACKS =    32, // Number of cylinders per platter.
       RAM_HEADS =      8, // Number of surfaces.
       RAM_BLOCKS =   512, // Bytes per sector. Set to 512 to support DOS formatting.
       RAM_SIZE =  ((RAM_SECTORS)*(RAM_TRACKS)*(RAM_HEADS)*(RAM_BLOCKS)), // RAMDrive size in bytes.

       CHUNK =          65536,       // Size of memory chunks to allocate.
       CHUNKY =   (CHUNK + 1),       // Chunk+1.  Used for testing read/writes to allocated chunks.
       LASTBYTE = (CHUNK - 1),       // Used in testing.
       INDEXS = ((RAM_SIZE)/(CHUNK)) // Number of allocation chunks needed to support size of drive.
       } ;

// null-terminated array of device names supported by this driver.
const char *vd_name[] = { "disk/virtual/ram", NULL } ;
const char *vd_path[] = { "boot/home/config/add-ons/kernel/drivers/bin/", NULL };

// Buffers and pointers needed for reading and writing to drive.
static uchar* B_Index[INDEXS+1]; // Index Array of allocated RAMDISK storage Buffers.
#include "icongraphics.h"        // Icon graphics in array.

static device_hooks vd_devices = {
 vd_open,    // -> open entry point
 vd_close,   // -> close entry point
 vd_free,    // -> free cookie
 vd_control, // -> control entry point
 vd_read,    // -> read entry point
 vd_write,   // -> write entry point
 NULL,       // my_device_select,   -> select entry point
 NULL,       // my_device_deselect, -> deselect entry point
 NULL,       // my_device_readv,    -> posix read entry point
 NULL } ;    // my_device_writev    -> posix write entry point

int32 api_version = B_CUR_DRIVER_API_VERSION;
  
status_t init_hardware(void) {
 long i;
 for (i=0; i<INDEXS; i++) { B_Index[i]=NULL; } return B_OK; } // Clear the allocated Buffer indexs.

status_t init_driver(void) { return B_OK; }

void uninit_driver(void) { return; }

const char** publish_devices() { return vd_name; }

device_hooks* find_device(const char* name) { return &vd_devices; }

status_t vd_open(const char *dname, uint32 flags, void **cookie) { return B_NO_ERROR; }

status_t vd_free (void *cookie) { return B_NO_ERROR; }

status_t vd_close(void *cookie) { return B_NO_ERROR; }
   
status_t vd_read(void *cookie, off_t pos, void *buf, size_t *count) {
 long Index;
 off_t i; off_t BufferOffset; size_t len; size_t Hunk;
 uchar* Buffer; // Buffers for reading data.

 Index = pos / CHUNK; pos = pos - (Index * CHUNK); len = (*count); BufferOffset = 0; // Find proper buffer indexs.

 while ( len > 0 ) {
    Buffer = B_Index[Index]; // If there is data to read from RAMDISK Buffers. Get pointer to a pre-allocated buffer.
    // Create buffer if buffer does not exists already. Return error code if no buffer created.
    if ( Buffer == NULL ) {
       Buffer = malloc(CHUNK); if ( Buffer == NULL ) { return B_ERROR; }
       for (i=0; i<CHUNK; i++) { Buffer[i] = (uchar)0; } } // Zero the contents of the buffer.

    // Data all inside this Buffer. Read from buffer. Flag length, End_of_Data.
    if ( (pos+len) < CHUNKY ) { memcpy(buf+BufferOffset, Buffer+pos, len); len = 0; }
    // Data is larger than buffer. How much from buffer? Reduce lenght by amount.
    // Write data to the end of buffer. Increase write pointer. Reset position. Index to next buffer.
    if ( (pos+len) > CHUNK ) { Hunk = CHUNK-pos;  memcpy(buf+BufferOffset, Buffer+pos, Hunk);
       len = len - Hunk; BufferOffset = BufferOffset + Hunk; pos = 0; Index++; }

    // Test if a non-indexed buffer exists. Delete buffer if not in index.
    if ( B_Index[Index] == NULL ) { free(Buffer); }
 }
 return B_NO_ERROR; } // Return error status.

status_t vd_write(void *cookie, off_t pos, const void *buf, size_t *count) { 
 long Index;
 off_t i; off_t BufferOffset; size_t len; size_t Hunk;
 uchar* Buffer; // Buffers for writing data.

 Index = pos / CHUNK; pos = pos - (Index * CHUNK); len = (*count); BufferOffset = 0; // Find proper buffer index.

 while ( len > 0 ) {
    Buffer = B_Index[Index]; // If there is data to write to RAMDISK Buffers. Get pointer to a pre-allocated buffer.
    // Create buffer if buffer does not exists already. Return error code if no buffer created.
    if ( Buffer == NULL ) {
       Buffer = malloc(CHUNK); if ( Buffer == NULL ) { return B_ERROR; }
       // Add buffer to index, zero out the contents of the read buffer before it is read.
       B_Index[Index] = Buffer; for (i=0; i<CHUNK; i++) { Buffer[i] = (uchar)0; } }

    // Data will fit inside this Buffer. Write to buffer. Flag length, End_of_Data.
    if ( (pos+len) < CHUNKY ) { memcpy(Buffer+pos, buf+BufferOffset, len); len = 0; }
    // Data is larger than buffer. How much fits into buffer? Reduce lenght by amount.
    // Write data to the end of buffer. Increase write pointer. Reset position. Index to next buffer.
    if ( (pos+len) > CHUNK ) { Hunk = CHUNK-pos; memcpy(Buffer+pos, buf+BufferOffset, Hunk);
       len = len - Hunk; BufferOffset = BufferOffset + Hunk; pos = 0; Index++; }
 }
 return B_NO_ERROR; } // Return error status.

status_t vd_control(void *cookie, uint32 ioctl, void *arg1, size_t len)
{ device_geometry *dinfo; device_icon *dicon; long i;
  switch (ioctl) {                     // generic mass storage device IO control codes.

  case B_GET_BIOS_GEOMETRY:            // Gets the BIOS settings. Not useful for a RAMDISK I think? Testing.
  case B_GET_GEOMETRY:                 // Get real drive geometry.
     dinfo = (device_geometry *) arg1; // Fills out the specified device_geometry structure to describe the device.
     dinfo->bytes_per_sector = RAM_BLOCKS; 
     dinfo->sectors_per_track = RAM_SECTORS; 
     dinfo->cylinder_count = RAM_TRACKS; 
     dinfo->head_count = RAM_HEADS; 
     dinfo->device_type = B_DISK; 
     dinfo->removable = FALSE; 
     dinfo->read_only = FALSE; 
     dinfo->write_once = FALSE;
     return B_OK; // Function supported. Returns device geometry.

  case B_FORMAT_DEVICE:                                // Low-Level formats the drive if boolean data is true.
     if (!*((char *) arg1)) return B_NO_ERROR;         // Flag = false, do not format. <<unclear what should be done >>
        for (i=0; i<INDEXS; i++) {                     // Flag = true, do low-level format.
           if (B_Index[i] != NULL) {                   // Test if a Buffer is allocated.
              free(B_Index[i]); B_Index[i] = NULL; } } // Free allocated Buffer. Clear index pointer.
     return B_OK;                                      // Function supported.

  case B_GET_ICON:                 // Returns a ICON_info structure for the device.
     dicon = (device_icon *) arg1; // Pointer to ICON_info structure.
     switch (dicon->icon_size) {

     case B_LARGE_ICON: // Function supported. Returns large icon image.
        memcpy(dicon->icon_data, icon_disk, B_LARGE_ICON * B_LARGE_ICON);    return B_OK;
     case B_MINI_ICON:  // Function supported. Returns small icon image.
        memcpy(dicon->icon_data, icon_disk_mini, B_MINI_ICON * B_MINI_ICON); return B_OK;
     default:           // Unknown icon size, returns small icon image.
        memcpy(dicon->icon_data, icon_disk_mini, B_MINI_ICON * B_MINI_ICON); return B_BAD_TYPE; }

  case B_GET_DEVICE_SIZE:                        // A size_t indicating the device size in bytes is returned.
     *(size_t*)arg1 = RAM_SIZE; return B_OK ;    // Function supported. Signals valid drive size.

  case B_GET_MEDIA_STATUS:                       // Gets the status of the media in the device by returning
     *(status_t*)arg1 = B_OK; return B_NO_ERROR; // a status_t at the location pointed to. Function is under test.

  case B_FLUSH_DRIVE_CACHE:                      // Flushes the drive's cache. Not useful for a RAMDISK. */
     *(status_t*)arg1 = B_OK; return B_NO_ERROR; // Returns B_OK status_t to function under test.

  case B_GET_DRIVER_FOR_DEVICE:                  // Returns the path of the driver executable handling the device.
     return B_ERROR;                             // Function is not presently supported.

  case B_GET_NEXT_OPEN_DEVICE: return B_ERROR;   // ??? Function not supported.

  case 10199:
     *(size_t*)arg1 = B_OK; return B_NO_ERROR;   // Totally unknown function, found while debugging.                         

  default: 
     return B_ERROR; }} // Returns B_OK status_t to unknown/unsupportted function.