// YColorControl.cpp
// by John Yanarella
// Copyright (c) 1999 John Yanarella, All Rights Reserved

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <TranslationUtils.h>
#include <Bitmap.h>
#include <Autolock.h>
#include <String.h>
#include "YColorControl.h"

#define TEXT_FIELD_MSG 'txtf'

const int shaded_triangle[] = { 0,0,0,0,0,0,1,0,0,0,0,0,0,
                                0,0,0,0,0,1,2,1,0,0,0,0,0,
                                0,0,0,0,1,2,4,3,1,0,0,0,0,
                                0,0,0,1,2,4,4,4,3,1,0,0,0,
                                0,0,1,2,4,4,4,4,4,3,1,0,0,								
                                0,1,2,4,4,4,4,4,4,4,3,1,0,
                                1,2,3,3,3,3,3,3,3,3,3,3,1,
                                1,1,1,1,1,1,1,1,1,1,1,1,1 };

YColorControl::YColorControl(BPoint leftTop, const char *name, BMessage *fgMessage, 
			     bool twoSwatches, BMessage *bgMessage)
   : BControl(BRect(leftTop.x, leftTop.y, leftTop.x + 260, leftTop.y + 70), name, "", fgMessage, B_FOLLOW_NONE, B_WILL_DRAW | B_FRAME_EVENTS)
{
   SetViewColor(216, 216, 216);
	
   // create offscreen bitmap
   BRect rect(0,0,212,70);
   m_image = new BBitmap(rect, B_RGB32, true);
   m_offscreenView = new BView(rect, "Offscreen View", B_FOLLOW_NONE, 0);
   m_image->AddChild(m_offscreenView);
	
   // set foreground and background color values
   m_foreground.red = 255;
   m_foreground.green = 255;
   m_foreground.blue = 255;
   m_background.red = 0;
   m_background.green = 0;
   m_background.blue = 0;
	
   // select top swatch
   m_selected = true;

   // two swatches or just one?
   m_twoSwatches = twoSwatches;
	
   // set background message
   m_bgMessage = bgMessage;
	
   // construct text views
   BRect tRect(0.0,0.0,38.0,14.0);
   rBTC = new BTextControl(tRect.OffsetToSelf(213, 7), "Red", "R", "", NULL);
   gBTC = new BTextControl(tRect.OffsetBySelf(0.0, 18.0), "Green", "G", "", NULL);
   bBTC = new BTextControl(tRect.OffsetBySelf(0.0, 18.0), "Blue", "B", "", NULL);
	
   // set message for changes in text 
   rBTC->SetModificationMessage(new BMessage(TEXT_FIELD_MSG));
   gBTC->SetModificationMessage(new BMessage(TEXT_FIELD_MSG));
   bBTC->SetModificationMessage(new BMessage(TEXT_FIELD_MSG));
	
   // set divider
   rBTC->SetDivider(10.0);
   gBTC->SetDivider(10.0);
   bBTC->SetDivider(10.0);
	
   // set maximum length for the text field to 3 charactors
   rBTC->TextView()->SetMaxBytes(3);
   gBTC->TextView()->SetMaxBytes(3);
   bBTC->TextView()->SetMaxBytes(3);
	
   // only allow 0-9 as input	
   int c;
   for (c = 23; c < 48; c++) {		
      rBTC->TextView()->DisallowChar(c);
      gBTC->TextView()->DisallowChar(c);
      bBTC->TextView()->DisallowChar(c);
   }
   for (c = 58; c < 127; c++) {
      rBTC->TextView()->DisallowChar(c);
      gBTC->TextView()->DisallowChar(c);
      bBTC->TextView()->DisallowChar(c);
   }

   // add text views
   AddChild(rBTC);
   AddChild(gBTC);
   AddChild(bBTC);
}

YColorControl::~YColorControl() {	
   // clean up - free up offscreen view and bitmap
   if (m_offscreenView) {
      m_image->RemoveChild(m_offscreenView);
      delete m_offscreenView;
   }
   if (m_image)
      delete m_image;	
	
	// clean up second message if needed
   if (m_bgMessage)
      delete m_bgMessage;
}

void YColorControl::AttachedToWindow() {
   // make this view the target for text control modification messages
   rBTC->SetTarget(NULL, this->Looper());
   gBTC->SetTarget(NULL, this->Looper());
   bBTC->SetTarget(NULL, this->Looper());
   UpdateTextFields();
   UpdateOffscreen();
}

void YColorControl::MouseDown(BPoint point) {
   // respect Enabled state
   if (IsEnabled() == false)
      return;

	// switch f/b selection
   if (m_twoSwatches && (BRect(12, 15, 52, 55).Contains(point))) {
      m_selected = !m_selected;
      UpdateTextFields();
      UpdateOffscreen();
      Draw(Bounds());
      return;
   }

   // determine color and component in question
   rgb_color* color;
   uint8* component = NULL;
	
   if (m_selected)
      color = &m_foreground;
   else
      color = &m_background;	
	
	// define input regions
   BRect rect1(71.0, 10.0, 212.0, 26.0);
   BRect rect2(71.0, 28.0, 212.0, 44.0);
   BRect rect3(71.0, 46.0, 212.0, 62.0);

   if (rect1.Contains(point))
      component = &color->red;
   if (rect2.Contains(point))
      component = &color->green;
   if (rect3.Contains(point))
      component = &color->blue;
	
   if (component != NULL) {
      int x = 0;
      BPoint prevPt(0.0,0.0);
      uint32 buttons = 1;

      // turn off modification messages
      rBTC->SetModificationMessage(NULL);
      gBTC->SetModificationMessage(NULL);
      bBTC->SetModificationMessage(NULL);

      while (buttons) {
	 if (point != prevPt) {
	    if (component != NULL) {
	       x = ((int)point.x - 77) * 2;
	       if (x <= 0)
		  x = 0;
	       if (x >= 254)
		  x = 255;					
	       if (*component != (uint8)x) {
		  *component = (uint8)x;		
		  UpdateTextFields();
		  UpdateOffscreen();
		  Draw(Bounds());
	       }
	    }							
	 }
	 prevPt = point; 
	 snooze(40000);
	 GetMouse(&point, &buttons, false);			
      }
		
      // turn 'em back on!
      rBTC->SetModificationMessage(new BMessage(TEXT_FIELD_MSG));
      gBTC->SetModificationMessage(new BMessage(TEXT_FIELD_MSG));
      bBTC->SetModificationMessage(new BMessage(TEXT_FIELD_MSG));

      // send a message
      if (Target() != NULL) {
	 if (m_selected && (Message() != NULL))
	    Invoke();
	 if (!m_selected && (m_bgMessage != NULL))
	    Invoke(m_bgMessage);
      }	
   }
			
   BControl::MouseDown(point);
}

void YColorControl::Draw(BRect updateRect) {
   if (m_image) {
      SetDrawingMode(B_OP_COPY);
      DrawBitmap(m_image);
   }
}

void YColorControl::UpdateOffscreen() {
   //BAutolock(m_offscreenView->Looper());
   m_offscreenView->LockLooper();

	// fill the screen with background color
   m_offscreenView->SetLowColor(216, 216, 216);
   m_offscreenView->SetHighColor(216, 216, 216); 
   m_offscreenView->FillRect(Bounds()); // would memset() be faster?

   BRect rect(0, 0, 26, 26);
   if (m_twoSwatches) {
      // draw foreground and background swatch
      if (m_selected) {
	 DrawFilledBox(rect.OffsetToSelf(26, 29), m_background, !m_selected);
	 DrawFilledBox(rect.OffsetToSelf(12, 15), m_foreground, m_selected);
      } else {
	 DrawFilledBox(rect.OffsetToSelf(12, 15), m_foreground, m_selected);
	 DrawFilledBox(rect.OffsetToSelf(26, 29), m_background, !m_selected);	
      }
   } else {
      rect.Set(16, 19, 48, 51);
      DrawFilledBox(rect, m_foreground, true);
   }
		
   // draw separator
   m_offscreenView->SetHighColor(128,128,128); 
   m_offscreenView->StrokeLine(BPoint(65, 5), BPoint(65, 65));
   m_offscreenView->SetHighColor(255,255,255);
   m_offscreenView->StrokeLine(BPoint(66, 5), BPoint(66, 65));
	
	// draw color ramps
   rect.Set(0,0,131,11);
   DrawBox(rect.OffsetToSelf(76.0, 10.0));
   DrawBox(rect.OffsetBySelf(0.0, 18.0));
   DrawBox(rect.OffsetBySelf(0.0, 18.0));			
   m_offscreenView->Sync();
		
   int i, x, y;
   uint32 c;
   rgb_color color, color2;
	
   if (m_selected)
      color2 = m_foreground;
   else
      color2 = m_background;	
		
   int32 xmax = 256;
   int32 ymax = 3;
   int32 bpr = m_image->BytesPerRow();
   uint32 *bits = (uint32 *)m_image->Bits(); 
   color.alpha = 255;
   for (x = 0; x < (xmax/2); x++) {
      for (y = 0; y < ymax; y++) {
	 int t = x*2;
	 if (t == 254)
	    t = 255;			
	 switch (y) {
	 case 0:
	    color.red = t;
	    color.green = color2.green;
	    color.blue = color2.blue;
	    break;
	 case 1:
	    color.red = color2.red;
	    color.green = t;
	    color.blue = color2.blue;
	    break;
	 case 2:
	    color.red = color2.red;
	    color.green = color2.green;
	    color.blue = t;
	    break;
	 }
	
#if B_HOST_IS_BENDIAN
	 c = ((uint32)color.blue<<24)|((uint32)color.green<<16)
	    |((uint32)color.red<<8)|(uint32)color.alpha;
#else  // B_HOST_IS_BENDIAN 
	 c = ((uint32)color.blue)|((uint32)color.green<<8)
	    |((uint32)color.red<<16)|(uint32)color.alpha<<24;
#endif // B_HOST_IS_BENDIAN 
	 int j = y*18 + 12;
	 for (i = j; i < (j+8); i++) {
	    bits[i*bpr/4+((x)+78)] = c;
	 }	
      }		
   }
		
   // draw triangles
   BPoint p;
   p.y = 18;
   p.x = color2.red/2 + 78;
   DrawShadedTriangle(p);
   p.y += 18;
   p.x = color2.green/2 + 78;
   DrawShadedTriangle(p);
   p.y += 18;
   p.x = color2.blue/2 + 78;
   DrawShadedTriangle(p);
   m_offscreenView->UnlockLooper();
}

void YColorControl::DrawBox(BRect rect, bool selected) {
   BAutolock(m_offscreenView->Looper());
   if (selected == true) {
      m_offscreenView->SetHighColor(0,0,0); 
      m_offscreenView->StrokeLine(BPoint(66, 5), BPoint(66, 65));
      m_offscreenView->StrokeRect(rect);
      m_offscreenView->StrokeRect(rect.InsetByCopy(2.0, 2.0));
		
      m_offscreenView->SetHighColor(255,255,255);
      m_offscreenView->StrokeRect(rect.InsetByCopy(1.0, 1.0));
			
      m_offscreenView->SetHighColor(m_foreground);
      m_offscreenView->FillRect(rect.InsetByCopy(3.0, 3.0));		
   } else {
      m_offscreenView->SetHighColor(128,128,128);
      m_offscreenView->StrokeRect(rect);
		
      rect.left += 1;
      rect.top += 1;
      m_offscreenView->SetHighColor(255,255,255);
      m_offscreenView->StrokeRect(rect);
		
      rect.bottom -= 1;
      rect.right -= 1;
      m_offscreenView->SetHighColor(0,0,0);
      m_offscreenView->StrokeRect(rect);
		
      rect.top += 1;
      rect.left += 1;
      m_offscreenView->SetHighColor(216,216,216);
      m_offscreenView->StrokeRect(rect);
   }
}

void YColorControl::DrawFilledBox(BRect rect, rgb_color &color, bool selected) {
   BAutolock(m_offscreenView->Looper());
   DrawBox(rect, selected);		
   m_offscreenView->SetHighColor(color);
	
   if (selected)
      rect.InsetBy(3.0, 3.0);
   else
      rect.InsetBy(2.0, 2.0);
	
   m_offscreenView->FillRect(rect);	
}

void YColorControl::DrawShadedTriangle(BPoint point) {
   BAutolock(m_offscreenView->Looper());
   rgb_color color;
   int x, y;
   uint32 c;
   int32 bpr = m_image->BytesPerRow();
   uint32 *bits = (uint32 *)m_image->Bits(); 

   point.x -= 6.0;

   int mW = (int)Bounds().right;
   int mH = (int)Bounds().bottom;

   for (y = 0; y <= 7; y++) {
      for (x = 0; x <= 12; x++) {
	 if ((shaded_triangle[x + (y*13)] != 0) && 
	     (y + (int)point.y < mH) && (x + (int)point.x < mW))
	 {
	    switch(shaded_triangle[x + (y*13)]) {
	    case 1:
	       color.red = 0;
	       color.green = 0;
	       color.blue = 0;
	       break;
	    case 2:
	       color.red = 255;
	       color.green = 255;
	       color.blue = 255;
	       break;		
	    case 3:
	       color.red = 128;
	       color.green = 128;
	       color.blue = 128;
	       break;
	    case 4:
	       color.red = 216;
	       color.green = 216;
	       color.blue = 216;
	       break;
	    default:					
	       break;			
	    }			

#if B_HOST_IS_BENDIAN
	    c = ((uint32)color.blue<<24)|((uint32)color.green<<16)
	       |((uint32)color.red<<8)|(uint32)color.alpha;
#else  // B_HOST_IS_BENDIAN 
	    c = ((uint32)color.blue)|((uint32)color.green<<8)
	       |((uint32)color.red<<16)|(uint32)color.alpha<<24;
#endif // B_HOST_IS_BENDIAN 
	    bits[(y+(int)point.y)*bpr/4+(x+(int)point.x)] = c;
	 }				
      }
   }
}

void YColorControl::UpdateTextFields() {
   rgb_color* color;

   if (m_selected)
      color = &m_foreground;
   else
      color = &m_background;	
	
   BString text = "";
   text << (int32)color->red;
   rBTC->SetText(text.String());

   text = "";
   text << (int32)color->green;
   gBTC->SetText(text.String());
	
   text = "";
   text << (int32)color->blue;
   bBTC->SetText(text.String());
}

void YColorControl::UpdateValues() {
   rgb_color* color;
	
   if (m_selected)
      color = &m_foreground;
   else
      color = &m_background;	
	
   int32 r = atoi(rBTC->Text());
   int32 g = atoi(gBTC->Text());
   int32 b = atoi(bBTC->Text());
	
   color->red = r;
   color->green = g;
   color->blue = b;

   // fix for if user inputs a number n such that n < 0  or n > 255
   if ((r < 0) || (r > 255)) {
      BString text = "";
      text << (int32)color->red;
      rBTC->SetText(text.String());
   }
	
   if ((g < 0) || (g > 255)) {
      BString text = "";
      text = "";
      text << (int32)color->green;
      gBTC->SetText(text.String());
   }
	
   if ((b < 0) || (b > 255)) {
      BString text = "";
      text = "";
      text << (int32)color->blue;
      bBTC->SetText(text.String());
   }
	
   UpdateOffscreen();
   Invalidate();
}

void YColorControl::MessageReceived(BMessage* message) {
   switch (message->what) {
   case TEXT_FIELD_MSG:
      // send a message
      if (Target() != NULL) {
	 if (m_selected && (Message() != NULL))
	    Invoke();
	 if (!m_selected && (m_bgMessage != NULL))
	    Invoke(m_bgMessage);
      }
			
      UpdateValues();
   default:
      BControl::MessageReceived(message);	
   }
}

void YColorControl::SetValue(rgb_color color, bool foreground) {
   rgb_color* current;
	
   if (foreground)
      current = &m_foreground;
   else
      current = &m_background;
	
   *current = color;
   UpdateOffscreen();
   UpdateTextFields();
   Invalidate();
}

rgb_color YColorControl::ValueAsColor(bool foreground) {
   rgb_color* color;
	
   if (foreground)
      color = &m_foreground;
   else
      color = &m_background;	
	
   return *color;
}

void YColorControl::SetSelection(bool foreground) {
   if (m_twoSwatches && (foreground != m_selected)) {
      m_selected = !m_selected;
      UpdateTextFields();
      UpdateOffscreen();
      Invalidate();
      return;
   }
}

void YColorControl::SetEnabled(bool enabled) {
   rBTC->SetEnabled(enabled);
   gBTC->SetEnabled(enabled);
   bBTC->SetEnabled(enabled);
   BControl::SetEnabled(enabled);	
}
