// | BeOS port of g3dkit
//
// | The g3dkit engine (and this source code) is (c)1998 by Peter Walser
// | Feel free to use this 3d engine for non-commercial purpose,
// | and please set a link to my homepage.
// | If you have any questions concerning the source code the 
// | methods and algorithms: send me e-mail.
// |
// | Peter Walser  
// | proxima@active.ch
// | http://www2.active.ch/~proxima
// | -------------------------------------------------------------

#ifndef _IDX3D_H_
#define _IDX3D_H_

#include <InterfaceKit.h>

extern float sinus[360];		//=new float[360];
extern float cosinus[360];		//=new float[360];	 


// DATA STRUCTURES

class idx3d_matrix
{
	public:
		float matrix[4][4];

		idx3d_matrix();

		void crossproduct(idx3d_matrix *a, idx3d_matrix *b);

		void shiftMatrix(float dx, float dy, float dz);

		void scaleMatrix(float dx, float dy, float dz);

		void rotateMatrix(float dx, float dy, float dz);

		void init(idx3d_matrix *m);

		void afficher();
};


class idx3d_vector
{
	public:
		float x;
		float y;
		float z;

		idx3d_vector();
		idx3d_vector(float xval, float yval, float zval);
		~idx3d_vector();
		
		void setValues(float xval, float yval, float zval);
		
		void vectordist(idx3d_vector *v1, idx3d_vector *v2);

		float vectorlength();

		void normalize(idx3d_vector *v);

		void normalize();

		void getNormal(idx3d_vector *a, idx3d_vector *b, idx3d_vector *c);

		void matrixvectorproduct(idx3d_matrix *b, idx3d_vector *a);
};


class idx3d_node
{
	public:
		idx3d_vector *v;	// new idx3d_vector();
		idx3d_vector *n;	// normal
		idx3d_vector *n2;	// normal (temp)
		long xx;				//Coordinates of the
		long yy;				// screen projection
		long zz;				//distance for z-buffer
		float tx;			// Horizontal texture position (0..1+)
		float ty;			// Vertical texture position   (0..1+)

		idx3d_node(float x,float y,float z);
		idx3d_node();
		~idx3d_node();
};

class idx3d_triangle
{
	public:
		long p1;
		long p2;
		long p3;
		idx3d_vector *n;
	
		idx3d_triangle(long q1,long q2,long q3);
		~idx3d_triangle();
};

class idx3d_object
{
	public:
		long mode;
		long color;

		idx3d_triangle **triangle;
		idx3d_node		**node;

		long triangles;
		long maxtriangles;
		long nodes;
		long maxnodes;
		long texture;
		idx3d_matrix *matrix;		// =new idx3d_matrix();
		idx3d_matrix *matrix2; 		// =new idx3d_matrix();

		idx3d_object(long rm, long col);
		~idx3d_object();
};

class idx3d_light
{
	public:
		long x;
		long y;
		long z;
		long mode;			//=0;              //0=parallel light, 1=polong light
		long intensity; 		//=255;

		idx3d_light(idx3d_vector *lvect, long lmode, long lintensity);
};

class idx3d_texture
{
	public:
		long w;
		long h;
		long *pixel;
};

const float pi = 3.14159265;
const float	deg2rad = pi/180;

class idx3d
{
	// Copyright information
	public:
/*		char *info="g3dkit BeOS 3d ENGINE";
		char *copyright="c1998 by Peter Walser (BeOS port by dragfly)";
		char *version="Version 2.0 BETA";
		char *build="15.01.1999";*/
		char *sysinfo;

	//Scene attributes
	private:
		long w;
		long h;
		float centerx;
		float centery;

	//Objects and Lights
	public:
		idx3d_object	**object;
		idx3d_light	**light;
		idx3d_texture	**texture;
		long objects;
		long lights;	
		long textures;
		long maxobjects;
		long maxlights;
		long maxtextures;
	private:
		idx3d_matrix worldmatrix;		//=new idx3d_matrix();
	
	//Color management
	public:
		long bgcolor;
	private:
		enum { alpha = 0xff000000,
				rbit = 0x00ff0000,
				gbit = 0x0000ff00,
				bbit = 0x000000ff};

	//Trigonometry
	private:
//		float sinus[360];		//=new float[360];
//		float cosinus[360];		//=new float[360];	 
//		enum { pi = 3.14159265,
//				deg2rad = pi/180};

	//Scene parameters
	public:
		float	perspective;		//=(float)4;
		float	zoomcorrection;	//=(float)0.8;
		int		ambient;			//=0;  //0..255
		int		phong;				//=80; //0..255
		int		reflectivity;		//=255; //0..255
		bool	staticLight;	//=false;

	//Render structures
	private:
		long LightMap[256*256];
		long ColorMap[256*256];
		long EnvMap[256*256];
		long StaticLightMap[256*256];
		long StaticEnvMap[256*256];
		long *zBuffer;
		long *LEdgeBuffer;
		long *REdgeBuffer;
		long *ZEdgeBufferL;
		long *ZEdgeBufferR;
		long *IEdgeBufferL;
		long *IEdgeBufferR;
		long *NXEdgeBufferL;
		long *NYEdgeBufferL;
		long *NXEdgeBufferR;
		long *NYEdgeBufferR;
		long *TXEdgeBufferL;
		long *TYEdgeBufferL;
		long *TXEdgeBufferR;
		long *TYEdgeBufferR;
		enum { zInfinite = 1000<<16,
				zNear = -2<<16};
	
	//Output
	private:
	public:
		long *TargetPixel;	//[];
		long *Background;	//[];
		
	// GENERAL METHODS
	
		public: idx3d(long width, long height);

		public: void resize(long width, long height);

		private: char *getSysInfo();

		private: void init_LightMap();

		public: void setStatic();

		private: void init_ColorMap();

		private: void init_TrigTables();

		public: void setPhong(long p);

		public: float crop(float a, float b, float c);

		public: long crop(long a, long b, long c);

		public: bool inrange(long a, long b, long c);

		public: bool inrange(float a, float b, float c);

		public: long rndColor();

	// GEOMETRY FUNCTIONS



	// DATA MANAGEMENT
		
		public: void addObject(long rendermode, long color);

		public: void addLight(idx3d_vector *v, long mode, long intensity);

		public: void addTexture(BBitmap *img);

		public: void addTriangle(long objectNr, long p1, long p2, long p3);

		public: void addNode(long objectNr, idx3d_node *t);

		public: void addNode(long objectNr,float x, float y, float z);

	// OBJECT MANIPULATION

		public: idx3d_matrix *shiftMatrix(float dx, float dy, float dz);

		public: idx3d_matrix *scaleMatrix(float dx, float dy, float dz);

		public: idx3d_matrix *rotateMatrix(float dx, float dy, float dz);

		public: void shiftObject(long o, float dx, float dy, float dz);

		public: void scaleObject(long o, float dx, float dy, float dz);

		public: void rotateObject(long o, float dx, float dy, float dz);

		public: void rotateObjectWorld(long o, float dx, float dy, float dz);

		public: void rotateObject(long obj, float px, float py, float pz, float dx, float dy, float dz);

		public: void shiftWorld(float dx, float dy, float dz);

		public: void scaleWorld(float dx, float dy, float dz);

		public: void rotateWorld(float dx, float dy, float dz);

		public: void scaleObject(long obj, float d);

		public: void scaleWorld(float d);


	// RENDERING

		public: void renderScene();

		private: void clearDoubleBuffer();

		private: void clearBackground();

		private: void paintObjects();

		private: void projectNode(long obj, idx3d_node *q);

		private: long getIntensity(long nx, long ny);

		private: long getIntensity(idx3d_vector *v);

		private: long getEnvironment(long nx, long ny);

		public: inline long getIntColor(long r, long g, long b);

		private: inline long getColor(long c, long intensity);

		private: long getGray(long c);

		public: void setBackground(BBitmap *TempImage);

		public: void setEnvironment(BBitmap *TempImage);

	// RENDERMODES

		private: void drawLine(idx3d_node *q1,idx3d_node *q2,long color);

		private: void drawWireframe(long obj, idx3d_triangle *t);

		private: void drawFlatshaded(long obj, idx3d_triangle *t);

		private: void drawGouraud(long obj, idx3d_triangle *t);

		private: void drawPhong(long obj, idx3d_triangle *t);

		private: void drawEnvmapped(long obj, idx3d_triangle *t);

		private: void drawGouraudTexture(long obj, idx3d_triangle *t);

		private: void drawPhongTexture(long obj, idx3d_triangle *t);

		private: void drawEnvmappedTexture(long obj, idx3d_triangle *t);

		private: void drawFlatshadedLine(long y, long color);

		private: void drawGouraudLine(long y, long color);

		private: void drawPhongLine(long y, long color);

		private: void drawEnvmappedLine(long y, long color);

		private: void drawGouraudTextureLine(long y, long t);

		private: void drawPhongTextureLine(long y, long t);

		private: void drawEnvmappedTextureLine(long y, long t);

	// SPECIAL FUNCTIONS TO CREATE OBJECTS

/*		public: void generateField(float data[][],long xmax, long ymax, long rm, long c);

		public: void generateField(BBitmap *img, float zscale, long rm, long c);

*/		public: void generateRotationObject(long obj, long steps);

		public: void generateNodeNormals(long obj);

	// BEMCHMARK
/*
		public: char *getFPS(long frames);
*/
};

#endif // _IDX3D_H_
