// Basic implementations for blanket module base classes

#include "BlanketModule.h"

// Saver class implementation

BlanketModule::Saver::Saver(BlanketModule *, BView* view, bigtime_t snoozetime)
: m_sem(-1),
  m_view(view),
  m_thread(-1),
  m_snoozetime(snoozetime)
{
}

BlanketModule::Saver::~Saver()
{
}

int32 BlanketModule::Saver::thread_entry_stub(void* arg)
{
	return ((Saver *)arg)->ThreadEntry();
}

int32 BlanketModule::Saver::ThreadEntry()
{
	bool	ready = false;

	if(acquire_sem(m_sem) == B_NO_ERROR)
	{
		Initialize();
		release_sem(m_sem);

		while(! ready && acquire_sem(m_sem) == B_NO_ERROR)
		{
			if(m_view->Window()->Lock())
			{
				if(! m_view->Window()->IsHidden())
					ready = true;
				m_view->Window()->Unlock();
			}
			release_sem(m_sem);
			snooze(m_snoozetime);
		}

		while(acquire_sem(m_sem) == B_NO_ERROR)
		{
			if(m_view->Window()->Lock())
			{
				Tick();
				m_view->Window()->Unlock();
			}
			release_sem(m_sem);
			snooze(m_snoozetime);
		}
	}

	return 0;
}

void BlanketModule::Saver::Start()
{
	m_sem = create_sem(1, "SaverSem");
	m_thread = spawn_thread(thread_entry_stub, "SaverTick", B_LOW_PRIORITY, this);
	resume_thread(m_thread);
}

void BlanketModule::Saver::Stop()
{
	int32 thread_return;

	// We must not kill the thread outside a lock or we
	// may leave the window locked...
	acquire_sem(m_sem);
	delete_sem(m_sem);

	wait_for_thread(m_thread, &thread_return);
}

void BlanketModule::Saver::Initialize()
{
	if(m_view->Window())
		m_view->Window()->Show();
}

void BlanketModule::Saver::Tick()
{
	// Empty Tick implementation lessens chance of call into
	// pure virtual!
}

// Config class implementation

BlanketModule::Config::Config(BlanketModule *owner, BView* view, bool wantpreview)
 : c_owner(owner), c_view(view)
{
	if(wantpreview)
	{
		c_preview = new BView(BRect(0, 0, 0, 0), B_EMPTY_STRING, B_FOLLOW_ALL, B_WILL_DRAW);
		c_preview->SetViewColor(0, 0, 0);
		c_preview->SetLowColor(0, 0, 0);
		c_preview->SetHighColor(255, 255, 255);
		c_saver = owner->InstantiateSaver(c_preview, true);	// it's a preview!
		c_saver->Start();
	}
	else
	{
		c_preview = 0;
		c_saver = 0;
	}
}

BlanketModule::Config::~Config()
{
	if(c_saver)
	{
		c_saver->Stop();
		delete c_saver;
	}
}

BView *BlanketModule::Config::GetPreview(BRect frame, uint32 resizeMask, uint32 flags)
{
	if(c_preview)
	{
		c_preview->MoveTo(frame.LeftTop());
		c_preview->ResizeTo(frame.Width(), frame.Height());
		c_preview->SetResizingMode(resizeMask);
		c_preview->SetFlags(flags);
	}

	return c_preview;
}

// special implementation of Config, for default case
class DefaultConfig : public BlanketModule::Config
{
public:
	DefaultConfig(BlanketModule *owner, BView* view)
	 : BlanketModule::Config(owner, view, true)
	{
		view->AddChild(new BStringView(BRect(10, 10, 200, 35), "Module name", class_name(owner)));

		BRect previewFrame = view->Bounds();
		previewFrame.left += 10.0;
		previewFrame.right -= 10.0;
		previewFrame.bottom -= 10.0;
		previewFrame.top = previewFrame.bottom - (previewFrame.Width() / 4 * 3);

		view->AddChild(GetPreview(previewFrame, B_FOLLOW_ALL, B_WILL_DRAW));
	}
};

// BlanketModule class implementation

BlanketModule::BlanketModule(BMessage* archive)
 : BArchivable(archive)
{
}

BlanketModule::~BlanketModule()
{
}

BlanketModule::Saver *BlanketModule::InstantiateSaver(BView* view, bool preview)
{
#pragma unused(preview)
	return new BlanketModule::Saver(this, view);
}

BlanketModule::Config* BlanketModule::InstantiateConfig(BView* view)
{
	return new DefaultConfig(this, view);
}
