// Stub routines to dispatch from old-style C interface

#include "BlanketModule.h"

// We need to export the following six functions
// This could be an interim measure if Blanket chooses
// to support the C++ way directly!
#pragma export on

extern "C" {

// Uncomment the following line and the corresponding line in
// "SaverModule.cp" to prevent 'Random' from calling your module
// (Many thanks to Jon Watte, author of 'Random')

//extern char inhibit_random;

extern void module_initialize(void *inSettings, long inSettingsSize);
extern void module_cleanup(void **outSettings, long *outSettingsSize);
extern void	module_start_saving(BView *inView);
extern void	module_stop_saving();
extern void module_start_config(BView *inView);
extern void module_stop_config();

}

#pragma export reset

static BlanketModule* module = 0;
static BlanketModule::Saver* saver = 0;
static BlanketModule::Config* config = 0;


// Uncomment the following line to prevent 'Random' from calling
// your module (thanks Jon!)

//char inhibit_random = 1;


// ------------------------------------------------------------
// 		• module_initialize  [optional]
// ------------------------------------------------------------
// This function gets called once, immediately after saver has
// loaded your module
// It is not known at this point whether you are being loaded
// to save the screen or to configure yourself (or both)
// inSettings is a pointer to data inSettingsSize long that saver 
// has stored for you, it is your responsibility to free it before 
// your module is unloaded
// saver may pass you NULL and 0 if it cannot find any data that
// belongs to you

void module_initialize(void* inSettings, long inSettingsSize)
{
	BMessage settings;

	if(inSettings)
	{
		BMessage temp;

		if(temp.Unflatten((const char *)inSettings) == B_OK)
			settings = temp;

		free(inSettings);
	}

	BArchivable *a = BlanketModule::Instantiate(&settings);
	module = dynamic_cast<BlanketModule *>(a);
	if(! module)
		delete a;	// problem: bail out
}


// ------------------------------------------------------------
// 		• module_cleanup  [optional]
// ------------------------------------------------------------
// This function gets called once, immediately before saver will
// unload your module
// This is the last chance to clean up after yourself
// Set outSettings to a pointer that is outSettingsSize long, 
// and saver will store it for you until the next time your module 
// is loaded
// (pass back NULL and 0 if you do not need to store anything)
// Do not attempt to free what you pass back to saver

void module_cleanup(void** outSettings, long* outSettingsSize)
{
	*outSettings = NULL;
	*outSettingsSize = 0;

	// Be safe if module does not exist		
	if (module)
	{
		// We are going to fill this with the settings
		BMessage settings;
		if (module->Archive(&settings, true) == B_OK)
		{
			size_t message_size = settings.FlattenedSize();
			char* message_buffer = (char*)(malloc(message_size));
			
			if (settings.Flatten(message_buffer, message_size) == B_OK)
			{
				// Everything went OK, so pass back reference to data
				*outSettings = message_buffer;
				*outSettingsSize = message_size;
			}
			else
			{
				// Failed to flatten correctly so free up
				// the garbage
				free(message_buffer);
			}
		}
		
		// Whatever, make sure we tidy up the module
		delete module;
		module = 0; 
	}
}


// ------------------------------------------------------------
// 		• module_start_saving  [required]
// ------------------------------------------------------------
// This function gets called immediately after a BWindow
// covering the entire screen has been made visible
// The BWindow contains one BView (passed to you as inView)
// Spawn your thread(s) here and return from this function ASAP
// Do all your drawing in inView, or in its children (if you 
// decide to add any)
// inView's View and Low color are set to black, High color is white
// saver never locks inView's window for you, so be sure to
// inView->Window()->Lock() before you try to do anything with
// inView, otherwise you win a free trip to the debugger

void module_start_saving(BView* view)
{
	if(module)
	{
		saver = module->InstantiateSaver(view, false);	// not a preview
		saver->Start();
	}		
}


// ------------------------------------------------------------
// 		• module_stop_saving  [required]
// ------------------------------------------------------------
// This function gets called right before the BWindow that
// you have been drawing into gets closed
// Kill all your thread(s), free any memory that you no longer need, 
// unlock anything that you locked, be good and clean up after
// yourself

void module_stop_saving()
{
	if(saver)
	{
		saver->Stop();
		delete saver;
		saver = 0;
	}
}


// ------------------------------------------------------------
// 		• module_start_config  [required]
// ------------------------------------------------------------
// This function gets called when the user has selected your module
// in the settings panel
// You are free to add any children to inView or modify it
// in any way (but don't even think about deleting it)
// inView is 241 units wide and 244 units high

void module_start_config(BView* view)
{
	if(module)
		config = module->InstantiateConfig(view);
}


// ------------------------------------------------------------
// 		• module_stop_config  [required]
// ------------------------------------------------------------
// This function gets called right before the BView that was passed
// to you in module_start_config gets deleted
// Poll the values of your controls, free any memory that you may
// have allocated, you know the game...
// Since your view will be deleted (mercilessly), it is not necessary
// to delete the children that you attached to it

void module_stop_config()
{
	delete config;
	config = 0;
}
