/*
 *  mon_m68k.cpp - 68k disassembler
 *
 *  (C) 1997-1998 Marc Hellwig
 */

/*
 * TODO
 * - add missing f-lines
 * - MULx.L/DIVx.L
 * - use tab instead of spaces
 */
 
/*
 * LOG
 * 
 * 5.11.97	added TODO and LOG section.
 */

#include <SupportDefs.h>
#include <stdio.h>
#include <string.h>

#include "mon_68k.h"
#include "mon_atraps.h"

enum t_size {BYTE, WORD, LONG, NOSIZE, BYTE_SILENT, WORD_SILENT, LONG_SILENT};	// "silent" means "don't append size qualifier to opcode"
enum t_errors {ER_UNKNOWN_ADDRESS_MODE, ER_UNKNOWN_OPCODE, ER_ILLEGAL_OPCODE};
enum t_address {AD_C=1000, AD_BRANCH, AD_MULT, AD_DATA, AD_DATA_IMM, AD_CACHE};

static void error(t_errors code);
static int8 read_int8(void);
static uint16 read_uint16(void);
static int16 read_int16(void);
static int32 read_int32(void);
static void parse ();
static void add_int32_str (char *old,int32 value);
static void add_int16_str (char *old,int16 value);
static void add_int8_str (char *old,int8 value);
static void add_dec_int_str (char *old,int32 value);
static void add_uint16_str (char *old,uint16 value);
static void print_range (int from,int to);
static int decode_address_mode (uint16 eee,uint16 aaa,t_size size);
static void diss (char *ndest,uint16 *nmem,int &npos);

enum t_control {C_UNKNOWN,C_SR,C_CCR,C_SFC,C_DFC,C_CACR,C_TC,C_ITT0,C_ITT1,C_DTT0,C_DTT1,C_BUSCR,C_USP,C_VBR,C_CAAR,C_MSP,C_ISP,C_MMUSR,C_URP,C_SRP,C_PCR};
static char *con_str[]={"???","sr","ccr","sfc","dfc","cacr","tc","itt0","itt1","dtt0","dtt1","buscr","usp","vbr","caar","msp","isp","mmusr","urp","srp","pcr"};
static char *cc_str[]= {"t","f","hi","ls","cc","cs","ne","eq","vc","vs","pl",
  "mi","ge","lt","gt","le","ra","sr"};
static uint16 opcode;
static uint16 source_eee;
static uint16 source_aaa;
static uint16 dest_eee;
static uint16 dest_aaa;
static int32 data_val;
static uint16 mask;				// for movem !!! aaarrrggglll ...
static t_size size;
static int count_param;
static uint16 *mem;
static int mem_pos;
static char *dest;
static char op_name[50];
static unsigned int adr_off;

static void error (t_errors code) {
	switch (code) {
	case ER_UNKNOWN_ADDRESS_MODE:
		printf ("unknown address mode\n");
		break;
	case ER_UNKNOWN_OPCODE:
		printf ("unknown opcode\n");
		break;
	case ER_ILLEGAL_OPCODE:
		printf ("illegal opcode\n");
		break;
	default:
		printf ("general error\n");
	}
}

static int8 read_int8(void) {
	return mem[mem_pos++];
}

static uint16 read_uint16(void) {
	return mem[mem_pos++];
}

static int16 read_int16(void) {
	return (int16)(mem[mem_pos++]);
}

static int32 read_int32(void) {
	int32 val = (mem[mem_pos] << 16) | mem[mem_pos+1];
	mem_pos+=2;
	return val;
}

// opcode table
#define def_param(n,c,s,se,sa,de,da) \
	strcpy(op_name,n);\
	count_param=c;\
	size=s;\
	source_eee=se;\
	source_aaa=sa;\
	dest_eee=de;\
	dest_aaa=da; 
#define no_param(n) \
	def_param(n,0,NOSIZE,0,0,0,0)
#define bits(pos,count) ((opcode >> pos)&((1<<count)-1))
#define mshort(s) \
	if (bits(6,3)==1) \
		{def_param("movea",2,s,bits(3,3),bits(0,3),1,bits(9,3))}\
	else\
		{def_param("move",2,s,bits(3,3),bits(0,3),bits(6,3),bits(9,3))}
#define imshort(n) \
	if (bits(6,2)<2) data_val = read_int16(); else data_val = read_int32();\
	def_param(n,2,(t_size)bits(6,2),AD_DATA_IMM,0,bits(3,3),bits(0,3))
#define imshortcs(n) \
	switch (bits(0,8)) { \
	case 0x3c: \
		data_val = read_int16();\
		def_param(n,2,BYTE_SILENT,AD_DATA_IMM,0,AD_C,C_CCR) break; \
	case 0x7c:	\
		data_val = read_int16();\
		def_param(n,2,WORD_SILENT,AD_DATA_IMM,0,AD_C,C_SR) break; \
		break; \
	default: \
		imshort(n) \
		break;\
	}
#define addshort(n) \
	if (bits(8,1)==0)\
		{def_param(n,2,(t_size)bits(6,2),bits(3,3),bits(0,3),0,bits(9,3))}\
	else\
		{def_param(n,2,(t_size)bits(6,2),0,bits(9,3),bits(3,3),bits(0,3))}
#define addxshort(n) \
	if (bits(3,1)==0) {\
		def_param(n,2,(t_size)bits(6,2),0,bits(0,3),0,bits(9,3))\
	} else {\
		def_param(n,2,(t_size)bits(6,2),4,bits(0,3),4,bits(9,3))\
	}
#define addashort(n) \
	def_param(n,2,NOSIZE,bits(3,3),bits(0,3),1,bits(9,3))\
	if (bits(8,1)==0) size=WORD; else size=LONG;
#define deashort(n) \
	def_param(n,2,(t_size)bits(6,2),0,bits(9,3),bits(3,3),bits(0,3))
#define bitshort(n) \
	def_param(n,2,BYTE_SILENT,AD_DATA_IMM,0,bits(3,3),bits(0,3))\
	data_val=read_int8();
#define bitdynshort(n) \
	def_param(n,2,BYTE_SILENT,0,bits(9,3),bits(3,3),bits(0,3))
#define prwshort(n) \
	if (((op2 >> 9) & 1)==1) \
		{def_param(n,2,LONG,AD_C,C_UNKNOWN,bits(3,3),bits(0,3))}\
	else\
		{def_param(n,2,LONG,bits(3,3),bits(0,3),AD_C,C_UNKNOWN)}
#define mshiftshort(n) \
	def_param(n,1,WORD,bits(3,3),bits(0,3),0,0)\
	if (bits(8,1)==0) {\
		strcat(op_name,"r");\
	} else {\
		strcat(op_name,"l");\
	}
#define rshiftshort(n) \
	if (bits(5,1)==0) {\
		data_val=bits(9,3);\
		if (data_val==0) data_val=8;\
		def_param(n,2,(t_size)bits(6,2),AD_DATA_IMM,0,0,bits(0,3))\
	} else {\
		def_param(n,2,(t_size)bits(6,2),0,bits(9,3) % 64,0,bits(0,3))\
	}\
	if (bits(8,1)==0) {\
		strcat(op_name,"r");\
	} else {\
		strcat(op_name,"l");\
	}
#define abcdshort(n) \
	if (bits(3,1)==0) {\
		def_param(n,2,BYTE,0,bits(0,3),0,bits(9,3))\
	} else {\
		def_param(n,2,BYTE,4,bits(0,3),4,bits(9,3))\
	}
#define cinvpush(n) \
	def_param(n,bits(3,2) == 3 ? 1 : 2,NOSIZE,AD_CACHE,bits(6,2),2,bits(0,3))\
	switch (bits(3,2)) {\
		case 1: strcat(op_name, "l"); break;\
		case 2: strcat(op_name, "p"); break;\
		case 3: strcat(op_name, "a"); count_param = 1;\
	}

static void parse () {
switch (bits(12,4)) {
case 0:										// Bit Manipulation/MOVEP/Immediate
	if (bits(8,1)==0) {						// Immediate + Bit Manipulation static
		if (bits(6,2)==3) {					//	RTM, CALLM, CMP2, CHK2, BSET, CAS2, CAS	
			switch (bits(9,3)) {
			case 3:							// RTM, CALLM
				break;
			case 0: case 1: case 2:			// CMP2, CHK2
				break;
			case 4:							// BSET !
				bitshort("bset") break;
			default:						// CAS2, CAS
				break;
			}
		} else {
			switch (bits(9,3)) {
			case 0:							// ORI
				imshortcs("ori") break;
			case 1:							// ANDI
				imshortcs("andi") break;
			case 2:							// SUBI
				imshort("subi") break;
			case 3:							// ADDI
				imshort("addi") break;
			case 4:							// BTST, BCHG, BCLR static
				switch (bits(6,2)) {
				case 0: 					// BTST
					bitshort("btst") break;
				case 1:						// BCHG
					bitshort("bchg") break;
				case 2:						// BCLR
					bitshort ("bclr") break;
				}
				break;
			case 5:							// EORI
				imshortcs("eori"); break;
			case 6:							// CMPI
				imshort("cmpi"); break;
			case 7:							// MOVES
				break;
			}
		}
	} else {								// Bit Manipulation dynamic, MOVEP
		if (bits(3,3)==1) {					// MOVEP
			if (bits(7,1)==0) {				// MOVEP d(Ay),Dx
				def_param("movep",2,NOSIZE,5,bits(0,3),0,bits(9,3))
			} else {						// MOVEP Dx,d(Ay)
				def_param("movep",2,NOSIZE,0,bits(9,3),5,bits(0,3))
			}
			if (bits(6,1)==0) size=WORD; else size=LONG;
		} else {							// Bit Manipulation
			switch (bits(6,2)) {
			case 0:								// BTST
				bitdynshort("btst") break;
			case 1:								// BCHG
				bitdynshort("bchg") break;
			case 2:								// BCLR
				bitdynshort ("bclr") break;
			case 3:								// BSET
				bitdynshort ("bset") break;
			}
		}
	}
	break;
case 1:										// Move Byte
	mshort(BYTE)
	break;
case 2:										// Move Long
	mshort(LONG)
	break;
case 3:										// Move Word
	mshort(WORD)
	break;
case 4:										// Miscellaneous
	switch (bits(0,12)) {					// First all fixed opcodes ...
	case 0xcfa:								// BGND
		no_param("bgnd")
		break;
	case 0xafc:								// ILLEGAL
		no_param("illegal")
		break;
	case 0xe70:								// RESET
		no_param("reset")
		break;
	case 0xe71:								// NOP
		no_param("nop")
		break;
	case 0xe72:								// STOP
		data_val=read_int16();
		def_param("stop",1,WORD_SILENT,AD_DATA_IMM,0,0,0) break;
	case 0xe73:								// RTE
		no_param("rte")
		break;
	case 0xe74:								// RTD
		data_val=read_int16();
		def_param("rtd",1,WORD_SILENT,AD_DATA,0,0,0) break;
	case 0xe75:								// RTS
		no_param("rts")
		break;
	case 0xe76:								// TRAPV
		no_param("trapv")
		break;
	case 0xe77:								// RTR
		no_param("rtr")
		break;
	case 0xe7a: case 0xe7b:					// MOVEC
		uint16 op2 = read_uint16();
		uint16 val;
		switch (op2 & 0xfff) {
		case 0:	val=C_SFC; break;
		case 1: val=C_DFC; break;
		case 2: val=C_CACR; break;
		case 3: val=C_TC; break;
		case 4: val=C_ITT0; break;
		case 5: val=C_ITT1; break;
		case 6: val=C_DTT0; break;
		case 7:	val=C_DTT1; break;
		case 8:	val=C_BUSCR; break;
		case 0x800:	val=C_USP; break;
		case 0x801:	val=C_VBR; break;
		case 0x802:	val=C_CAAR; break;
		case 0x803:	val=C_MSP; break;
		case 0x804:	val=C_ISP; break;
		case 0x805:	val=C_MMUSR; break;
		case 0x806:	val=C_URP; break;
		case 0x807:	val=C_SRP; break;
		case 0x808:	val=C_PCR; break;
		default: val=C_UNKNOWN; break;						// ERROR unknown control_reg
		}
		if (bits(0,1)==0) {					// MOVEC Ctrl,AD
			def_param("movec",2,NOSIZE,AD_C,val,((op2 >> 15) & 1),((op2 >> 12) & 7))
		} else {							// MOVEC AD,Ctrl
			def_param("movec",2,NOSIZE,((op2 >> 15) & 1),((op2 >> 12) & 7),AD_C,val)
		}
		break;
	default:
		switch (bits(3,9)) {
		case 0x110: case 0x118:				// EXT maybe more ??? nicht ganz sauber
			def_param("ext",1,((bits(6,2)==2) ? WORD : LONG),0,bits(0,3),0,0) break;
		case 0x101:							// LINK
			data_val=read_int32();
			def_param("link",2,LONG,1,bits(0,3),AD_DATA_IMM,0) break;
		case 0x108:							// SWAP
			def_param("swap",1,NOSIZE,0,bits(0,3),0,0) break;
		case 0x109:							// BKPT
			data_val=bits(0,3);
			def_param("bkpt",1,BYTE,AD_DATA,0,0,0) break;
		case 0x1c8: case 0x1c9:				// TRAP
			data_val=bits(0,4);
			def_param("trap",1,BYTE,AD_DATA,0,0,0) break;
		case 0x1ca:							// LINK
			data_val=read_int16();
			def_param("link",2,WORD_SILENT,1,bits(0,3),AD_DATA_IMM,0) break;
		case 0x1cb:							// UNLK
			def_param("unlk",1,NOSIZE,1,bits(0,3),0,0) break;
		case 0x1cc: case 0x1cd:				// MOVE USP
			if (bits(3,1)==0) {				// MOVE An,USP
				def_param("move",2,LONG,1,bits(0,3),AD_C,C_USP)
			} else {						// MOVE USP,An
				def_param("move",2,LONG,AD_C,C_USP,1,bits(0,3))
			}
			break;
		default:
			switch (bits(6,6)) {
			case 3:							// MOVE from SR
				def_param("move",2,WORD_SILENT,AD_C,C_SR,bits(3,3),bits(0,3)) break;
			case 0xb:						// MOVE from CCR
				def_param("move",2,WORD_SILENT,AD_C,C_CCR,bits(3,3),bits(0,3)) break;
			case 0: case 1: case 2:			// NEGX
				def_param("negx",1,(t_size)bits(6,2),bits(3,3),bits(0,3),0,0) break;
			case 8: case 9: case 0xa:		// CLR
				def_param("clr",1,(t_size)bits(6,2),bits(3,3),bits(0,3),0,0) break;
			case 0x13:						// MOVE to CCR
				def_param("move",2,WORD_SILENT,bits(3,3),bits(0,3),AD_C,C_CCR) break;
			case 0x10: case 0x11: case 0x12:// NEG
				def_param("neg",1,(t_size)bits(6,2),bits(3,3),bits(0,3),0,0) break;
			case 0x18: case 0x19: case 0x1a:// NOT
				def_param("not",1,(t_size)bits(6,2),bits(3,3),bits(0,3),0,0) break;
			case 0x1b:						// MOVE to SR
				def_param("move",2,WORD_SILENT,bits(3,3),bits(0,3),AD_C,C_SR) break;
			case 0x20:						// NBCD
				def_param("nbcd",1,BYTE,bits(3,3),bits(0,3),0,0) break;
			case 0x21:						// PEA
				def_param("pea",1,LONG,bits(3,3),bits(0,3),0,0) break;
			case 0x2b:						// TAS
				def_param("tas",1,BYTE,bits(3,3),bits(0,3),0,0) break;
			case 0x28: case 0x29: case 0x2a:// TST
				def_param("tst",1,(t_size)bits(6,2),bits(3,3),bits(0,3),0,0) break;
			case 0x30:						// MULU, MULS, DIVU, DIVUL
				//!!
				break;
			case 0x31:						// DIVS, DIVSL
				//!!
				break;
			case 0x3a:						// JSR
				def_param("jsr   ",1,NOSIZE,bits(3,3),bits(0,3),0,0) break;
			case 0x3b:						// JMP
				def_param("jmp   ",1,NOSIZE,bits(3,3),bits(0,3),0,0) break;
			case 0x22: case 0x23: case 0x32: case 0x33:// MOVEM
				mask=read_uint16();
				if (bits(10,1)!=0)
					{def_param("movem",2,NOSIZE,bits(3,3),bits(0,3),AD_MULT,0)}
				else
					{def_param("movem",2,NOSIZE,AD_MULT,((bits(3,3)==4) ? 1 : 0),bits(3,3),bits(0,3))}
				if (bits(6,1)==0) size=WORD; else size=LONG;
				break;
			default:
				switch (bits(6,3)) {
				case 7:						// LEA
					def_param("lea",2,NOSIZE,bits(3,3),bits(0,3),1,bits(9,3)) break;
				case 0: case 2: case 4:	 case 6:// CHK -> possible error ! check !
					def_param("chk",2,WORD,bits(3,3),bits(0,3),0,bits(9,3)) break;
				default:					// ERROR unknown opcode
					break;										
				}
			}
		}
	}
	break;
case 5:										// ADDQ/SUBQ/Scc/DBcc/TRAPcc
	if (bits(6,2)==3) {						// DBcc/TRAPcc/Scc
		switch (bits(3,3)) {
		case 1:								// DBcc
			def_param("db",2,WORD_SILENT,0,bits(0,3),AD_BRANCH,bits(8,4))
			break;
		case 7:								// TRAPcc
			// not yet implemented
			break;
		default:							// Scc
			def_param("s",1,BYTE,bits(3,3),bits(0,3),AD_BRANCH,bits(8,4))
		}
	} else {								// ADDQ/SUBQ
		if (bits(8,1)==0) {					// ADDQ
			data_val=bits(9,3);
			def_param("addq",2,(t_size)bits(6,2),AD_DATA_IMM,0,bits(3,3),bits(0,3))
		} else {							// SUBQ
			data_val=bits(9,3);
			def_param("subq",2,(t_size)bits(6,2),AD_DATA_IMM,0,bits(3,3),bits(0,3))			
		}
	}
	break;
case 6:										// Bcc/BSR/BRA
	def_param("b",1,NOSIZE,AD_BRANCH,bits(0,8),AD_BRANCH,bits(8,4))
	if (dest_aaa < 2) dest_aaa += 16;		//Bcc -> BSR,BRA
	switch (bits(0,8)) {
	case 0: size = WORD; break;
	case 0xff: size = LONG; break;
	default: size = BYTE;
	}
	break;
case 7:										// MOVEQ
	data_val=bits(0,8);
	def_param("moveq",2,BYTE_SILENT,AD_DATA_IMM,0,0,bits(9,3))
	break;
case 8:										// OR/DIV/SBCD
	switch (bits(4,5)) {
	case 0x10:								// SBCD
		abcdshort("sbcd") break;
	case 0x14:								// PACK
		break;
	case 0x18:								// UNPK
		break;
	default:
		switch (bits(6,3)) {
		case 3:								// DIVU/DIVUL
			def_param("divu",2,WORD,bits(3,3),bits(0,3),0,bits(9,3)) break;
		case 7:								// DIVS/DIVSL
			def_param("divs",2,WORD,bits(3,3),bits(0,3),0,bits(9,3)) break;
		default:							// OR
			addshort("or")
			break;
		}
	}
	break;
case 9:										// SUB/SUBX
	if (bits(6,2)==3) {						// SUBA
		addashort("suba")
	} else {
		if ((bits(4,2)==0)&&(bits(8,1)==1)) {// SUBX
			addxshort("subx")
		} else {							// SUB
			addshort("sub")
		}
	}
	break;
case 0xa:									// (Unassigned, Reserved)
	int position = 0;
	no_param("a-line")
	while (atraps[position].word!=0) {
		if (atraps[position].word == opcode) {
			no_param(atraps[position].name)
			break;
		}
		position++;
	}
	break;
case 0xb:									// CMP/EOR
	if (bits(6,2)==3) {						// CMPA
		def_param("cmpa",2,((bits(8,1)==0) ? WORD : LONG),bits(3,3),bits(0,3),1,bits(9,3))
	} else {
		if (bits(8,1)==1) {					// CMPM/EOR
			if (bits(3,3)==1) {				// CMPM
				def_param("cmpm",2,(t_size)bits(6,2),3,bits(0,3),3,bits(9,3))
			} else {						// EOR
				deashort("eor")
			}
		} else {							// CMP
			deashort("cmp")
		}
	}
	break;
case 0xc:									// AND/MUL/ABCD/EXG
	switch (bits(6,3)) {
	case 3:									// MULU
		def_param("mulu",2,WORD,bits(3,3),bits(0,3),0,bits(9,3)) break;
	case 7:									// MULS
		def_param("muls",2,WORD,bits(3,3),bits(0,3),0,bits(9,3)) break;
	default:								// ABCD/EXG/AND
        if (bits(4,5)==0x10) {				// ABCD          		
        	abcdshort("abcd")
		} else {							// EXG/AND
			if ((bits(3,6)==0x28)||(bits(3,6)==0x29)||(bits(3,6)==0x31)) {//EXG
				switch (bits(3,5)) {
				case 8: def_param("exg",2,LONG,0,bits(0,3),0,bits(9,3)) break;
				case 9: def_param("exg",2,LONG,1,bits(0,3),1,bits(9,3)) break;
				case 0x11: def_param("exg",2,LONG,1,bits(0,3),0,bits(9,3)) break;
				}
			} else {						// AND
				addshort("and")
			}
		}
	}
	break;
case 0xd:									// ADD/ADDX
	if (bits(6,2)==3) {						// ADDA
		addashort("adda")
	} else {								// ADDX/ADD
		if ((bits(4,2)==0)&&(bits(8,1)==1)) {// ADDX
			addxshort("addx")
		} else {							// ADD
			addshort("add")
		}
	}
	break;
case 0xe:									// Shift/Rotate/Bit Field
	if (bits(6,2)==3) {						// Mem Shift/Rotate, Bit Field
		if (bits(11,1)==0) {				// Mem Shift/Rotate
			switch (bits(9,2)) {
			case 0:							// ASL/ASR
				mshiftshort("as") break;
			case 1:							// LSL/LSR
				mshiftshort("ls") break;
			case 2:							// ROXL/ROXR
				mshiftshort("rox") break;
			case 3:							// ROL/ROR
				mshiftshort ("ro") break;			
			}
		} else {							// Bit Field
			switch (bits(8,3)) {
			case 0:							// BFTST
				break;
			case 1:							// BFEXTU
				break;
			case 2:							// BFCHG
				break;
			case 3:							// BFEXTS
				break;
			case 4:							// BFCLR
				break;
			case 5:							// BFFFO
				break;
			case 6:							// BFSET
				break;
			case 7:							// BFINS
				break;			
			}
		}
	} else {								// Register Shift/Rotate
		switch (bits(3,2)) {
		case 0:								// ASL/ASR
			rshiftshort("as") break;
		case 1:								// LSL/LSR
			rshiftshort("ls") break;
		case 2:								// ROXL/ROXR
			rshiftshort("rox") break;
		case 3:								// ROL/ROR
			rshiftshort("ro") break;
		}
	}
	break;
case 0xf:									// Coprocessor Interface/MC68040 
											// and CPU32 Extensions
	switch (bits(6,6)) {
	case 0:									// P... 
		uint16 op2 = read_uint16();
		switch ((op2 >> 13) & 7) {
		case 0:								// PMOVE
			prwshort("pmove") break;
		case 1:								// PLOAD/PVALID/PFLUSH
			no_param("pload/pvalid/pflush") break;
		case 2:								// PMOVE
			prwshort ("pmove") break;
		case 3:								// PMOVE
			prwshort ("pmove") break;
		case 4:								// PTST
			prwshort ("ptest") break;
		case 5:								// PFLUSHR
			def_param ("pflushhr",1,LONG,bits(3,3),bits(0,3),0,0) break;
		} 
		break;
	case 0x10: case 0x11: case 0x12: case 0x13:// Cache
		if (bits(5,1)==0) {					// CINV
			cinvpush("cinv");
		} else {							// CPUSH
			cinvpush("cpush");
		}
		break;
	}
	break;
default: 									// should never happen
	break;
}}
//

static void add_int32_str (char *old,int32 value) {
	sprintf (&(old[strlen(old)]),"$%08x",value);
}

static void add_int16_str (char *old,int16 value) {
	sprintf (&(old[strlen(old)]),"$%04x",value);
}

static void add_int8_str (char *old,int8 value) {
	sprintf (&(old[strlen(old)]),"$%02x",value);
}

static void add_dec_int_str (char *old,int32 value) {
	sprintf (&(old[strlen(old)]),"%d",value);
}

static void add_uint16_str (char *old,uint16 value) {
	sprintf (&(old[strlen(old)]),"$%04x",value);
}

static void print_range (int from,int to) {
	if (from < 8) dest = strcat (dest,"d"); else dest = strcat(dest,"a");
	add_dec_int_str(dest,from & 7);
	if (to > from) {
		dest = strcat(dest,"-");
		if (to < 8) dest = strcat (dest,"d"); else dest = strcat(dest,"a");
		add_dec_int_str(dest,to & 7);
	}	
}

static void decode_extended (bool a_reg,uint16 aaa,int32 old_mem) {
	uint16 value = read_uint16();
	uint16 scale = 1 << ((value >> 9) & 3);
	uint16 d_a = (value >> 15) & 1;
	uint16 reg = (value >> 12) & 7;
	uint16 w_l = (value >> 11) & 1;
	strcat (dest,"(");
	if (((value >> 8) & 1)==0) {				// standard format
		if (a_reg) {
			add_int8_str(dest,value & 0xff);
			strcat (dest,",a");
			add_dec_int_str(dest,aaa);
		} else {
			add_int32_str(dest,old_mem*2+adr_off+(int32)(int8)(value));
			strcat (dest,",pc");
		}
		if (d_a==0) strcat (dest,",d"); else strcat (dest,",a");
		add_dec_int_str(dest,reg);
		if (w_l==0) strcat (dest,".w"); else strcat (dest,".l");
		if (scale > 1) {
			strcat (dest,"*");
			add_dec_int_str(dest,scale);
		}
	} else {									// extended format
		uint16 i_is = value & 3;
		uint16 bd_size = (value >> 4) & 3;
		uint16 bs = (value >> 7) & 1;
		uint16 is = (value >> 6) & 1;
		if (i_is!=0) strcat(dest,"[");
		if (bd_size!=1) {						// base displacement
			switch (bd_size) {
				case 2:	if (a_reg)
							add_int32_str(dest,read_int16());
						else
							add_int32_str(dest,old_mem*2+adr_off+(int32)read_int16());
						break;
				case 3:	if (a_reg)
							add_int32_str(dest,read_int32());
						else
							add_int32_str(dest,old_mem*2+adr_off+read_int32());
						break;
				default:
						strcat (dest,"illegal base displacement ");
			}
		}
		if (bs==0) {
			if (bd_size!=1) strcat(dest,",");
			if (a_reg) {
				strcat (dest,"a");
				add_dec_int_str(dest,aaa);
			} else {
				strcat (dest,"pc");
			}
		}	
		if ((is==0)&&(i_is>4)&&(i_is<8)) strcat(dest,"]");
		if (is==0) {
			if (bs==0||bd_size!=1) strcat(dest,",");
			if (d_a==0) strcat (dest,",d"); else strcat (dest,",a");
			add_dec_int_str(dest,reg);
			if (w_l==0) strcat (dest,".w"); else strcat (dest,".l");
			if (scale > 1) {
				strcat (dest,"*");
				add_dec_int_str(dest,scale);
			}
		} 
		if ((i_is>0)&&(i_is<4)) strcat(dest,"]");
		if (((is==0)&&(i_is==4))||((is==1)&&(i_is>4))) strcat(dest,"reserved"); else
		switch (i_is & 3) {
		case 2:	strcat(dest,",");
				add_int32_str(dest,read_int16());
				break;
		case 3:	strcat(dest,",");
				add_int32_str(dest,read_int32());
				break;
		}
	}				
	strcat (dest,")");
}

static int decode_address_mode (uint16 eee,uint16 aaa,t_size size) {
	int32 old_mem = mem_pos;	
	switch (eee) {
	case 0:	strcat (dest,"d");
			add_dec_int_str(dest,aaa);
			break;
	case 1:	strcat (dest,"a");
			add_dec_int_str(dest,aaa);
			break;
	case 2:	strcat (dest,"(a");
		 	add_dec_int_str(dest,aaa);
		 	strcat (dest,")");
			break;
	case 3:	strcat (dest,"(a");
			add_dec_int_str(dest,aaa);
			strcat (dest,")+");
			break;
	case 4:	strcat (dest,"-(a");
			add_dec_int_str(dest,aaa);
			strcat (dest,")");
			break;
	case 5:	strcat (dest,"(");
			add_int16_str(dest,read_int16());
			strcat (dest,",a");
			add_dec_int_str(dest,aaa);
			strcat (dest,")");
			break;
	case 6:	decode_extended(true,aaa,0);	 
			break;
	case 7:	switch (aaa) {
			case 0:	add_int16_str(dest,read_int16());
					strcat (dest,".w");
					break; 
			case 1:	add_int32_str(dest,read_int32());
					strcat (dest,".l");
					break;
			case 2:	strcat (dest,"(");
					add_int32_str(dest,old_mem*2+adr_off+(int32)read_int16());
					strcat (dest,",pc)");
					break;
			case 3:	decode_extended(false,0,old_mem);	 
					break;
			case 4: strcat (dest,"#");
					switch (size) {
					case BYTE_SILENT:
					case BYTE:	add_int8_str(dest,read_int8() & 0xff); 
								break;
					case WORD_SILENT:
					case WORD:	add_int16_str(dest,read_int16());
								break;
					case LONG_SILENT:
					case LONG:	add_int32_str(dest,read_int32());
								break;
					case NOSIZE:	add_int8_str(dest,opcode & 0xff);
								break;
					default: ;
					}
					break;
			default: 	error (ER_UNKNOWN_ADDRESS_MODE);
						return -1;
			}
			break;
	case AD_C:	strcat (dest,con_str[aaa]);
			break;
	case AD_BRANCH:
			switch (size){
			case BYTE_SILENT:
			case BYTE:	add_int32_str(dest, (int32)(int8)aaa+old_mem*2+adr_off); 
						break;
			case WORD_SILENT:
			case WORD:	add_int32_str(dest, (int32)read_int16()+old_mem*2+adr_off);
						break;
			case LONG_SILENT:
			case LONG:	add_int32_str(dest, read_int32()+old_mem*2+adr_off);
						break;
			}
			break;
	case AD_MULT:
			int dir = 1;
			int off = 0;
			if (aaa	!= 0) {
				dir = -1;
				off = 15;
			} 
			int lend = -1;
			bool first = true;
			for (int i=0;i < 16;i++) {
				if (((mask >> (off + dir * i))& 1) == 0) {
					// print reg from lend+1 to i-1
					if ((lend+1) <= (i-1)) {
						if (!first) dest = strcat (dest,"/"); else first = false;
						if (((lend + 1) < 8) && ((i-1) > 7)) {
							print_range (lend+1,7);
							dest = strcat (dest,"/");
							print_range (8,i-1);	
						} else {
							print_range (lend+1,i-1);
						}
					} 
					lend = i;
				}
			}
			if (((mask >> (off + dir * 15)) & 1) == 1) {		// print last regs
				if (!first) dest = strcat (dest,"/");
				print_range (lend+1,15);
			}
			break;
	case AD_DATA_IMM:
		strcat (dest,"#");
		switch (size) {
		case BYTE: case BYTE_SILENT: add_int8_str(dest,data_val); break;
		case WORD: case WORD_SILENT: add_int16_str(dest,data_val); break;
		case LONG: case LONG_SILENT: add_int32_str(dest,data_val); break;
		default: ;
		}
		break;
	case AD_DATA:
		switch (size) {
		case BYTE: case BYTE_SILENT: add_int8_str(dest,data_val); break;
		case WORD: case WORD_SILENT: add_int16_str(dest,data_val); break;
		case LONG: case LONG_SILENT: add_int32_str(dest,data_val); break;
		default: ;
		}
		break;
	case AD_CACHE:
		switch (aaa) {
			case 1: strcat(dest, "dc"); break;
			case 2: strcat(dest, "ic"); break;
			case 3: strcat(dest, "dc/ic"); break;
			default: strcat(dest, "?"); break;
		}
		break;
	default:
		error (ER_UNKNOWN_ADDRESS_MODE);
		return -1;
	}
	return 0;
}

static void diss (char *ndest,uint16 *nmem,int &npos,unsigned int adr) {
	dest = ndest; mem = nmem; mem_pos = npos; adr_off=adr;
	dest[0]=0;
	opcode = read_uint16 ();
	strcat(dest,"\t");
	
	no_param("?")
	parse();

	dest = strcat(dest,op_name);
	if ((dest_eee == AD_BRANCH) && (size != NOSIZE)) dest = strcat (dest,cc_str[dest_aaa]);
	switch (size) {
	case BYTE:	strcat (dest,source_eee == AD_BRANCH ? ".s\t" : ".b\t"); break;
	case WORD:	strcat (dest,source_eee == AD_BRANCH ? "\t" : ".w\t"); break;
	case LONG:	strcat (dest,".l\t"); break;
	default:	dest = strcat (dest,"\t"); break;
	}
	if (count_param > 0) {
		if (decode_address_mode(source_eee,source_aaa,size) == -1) {
			npos = mem_pos;
			return;
		}
		if (count_param > 1) {
			strcat (dest, ",");
			if (decode_address_mode(dest_eee,dest_aaa,size) == -1) {
				 npos = mem_pos;
				return;
			}
		}
	}
	npos = mem_pos;
}

int disass_68k(FILE *f, unsigned int adr, uint16 *m)
{
	char output[80];
	int mem_pos = 0;
	diss(output, m, mem_pos,adr);
	for (int i=0;i<4;i++) if (i<mem_pos) fprintf(f,"%04x ", m[i]); else fprintf(f,"     ");
	fprintf(f, "%s\n", output);
	return mem_pos << 1;
}
