/*  
	BeOS Front-end du PDF file reader xpdf.
    Copyright (C) 1996 Derek B. Noonburg
	Copyright (C) 2000 Michael Pfeiffer

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#ifndef BE_OUTPUT_FONT_H
#define BE_OUTPUT_FONT_H

#ifdef __GNUC__
#pragma interface
#endif

#include <stddef.h>
#if HAVE_T1LIB_H
#include <t1lib.h>
#endif
#include "config.h"
#include "OutputDev.h"
#include <interface/Font.h>
#include <interface/View.h>

class GString;
class GfxColor;
class GfxFont;
class GfxSubpath;
class TextPage;
struct RGBColor;
class FontEncoding;
class BeOutputFontCache;

//------------------------------------------------------------------------
// Constants
//------------------------------------------------------------------------

#define maxRGBCube 8				// max size of RGB color cube

#define numTmpPoints 256		// number of XPoints in temporary array
#define numTmpSubpaths 16		// number of elements in temporary arrays
								//   for fill/clip

//------------------------------------------------------------------------
// Misc types
//------------------------------------------------------------------------

struct BoundingRect {
  short xMin, xMax;				// min/max x values
  short yMin, yMax;				// min/max y values
};

struct RGBColor {
  double r, g, b;
};

//------------------------------------------------------------------------
// Parameters
//------------------------------------------------------------------------

#if HAVE_T1LIB_H
// Type of t1lib font rendering to use:
//     "none"   -- don't use t1lib
//     "plain"  -- t1lib, without anti-aliasing
//     "low"    -- t1lib, with low-level anti-aliasing
//     "high"   -- t1lib, with high-level anti-aliasing
extern GString *t1libControl;
#endif

// If any of these are set, xpdf will use t1lib to render those font(s)
// instead of using the X server font(s).
extern GString *t1Courier;
extern GString *t1CourierBold;
extern GString *t1CourierBoldOblique;
extern GString *t1CourierOblique;
extern GString *t1Helvetica;
extern GString *t1HelveticaBold;
extern GString *t1HelveticaBoldOblique;
extern GString *t1HelveticaOblique;
extern GString *t1Symbol;
extern GString *t1TimesBold;
extern GString *t1TimesBoldItalic;
extern GString *t1TimesItalic;
extern GString *t1TimesRoman;
extern GString *t1ZapfDingbats;

// Use the EUC-JP encoding.
extern GBool useEUCJP;

#if JAPANESE_SUPPORT
// X font name pattern to use for Japanese text.
extern GString *japan12Font;
#endif

//------------------------------------------------------------------------
// BeOutputFont
//------------------------------------------------------------------------

class BeOutputFont {
public:

  BeOutputFont(GfxFont *gfxFont, double m11, double m12,
		      double m21, double m22, /*Display *display,*/
		      BeOutputFontCache *cache);

  virtual ~BeOutputFont();

  // Does this font match the ID and transform?
  GBool matches(Ref id1, double m11, double m12, double m21, double m22)
    { return id.num == id1.num && id.gen == id1.gen &&
		     m11 == tm11 && m12 == tm12 && m21 == tm21 && m22 == tm22; }

  // Was font created successfully?
  virtual GBool isOk() = 0;

  // Update <gc> with this font.
  // virtual void updateGC(/*GC gc*/) = 0;
  // virtual void update(rgb_color fill, rgb_color stroke);
  // Draw character <c> at <x>,<y>.
  virtual void drawChar(GfxState *state, BBitmap *bitmap, BView *view,
  						/*Pixmap pixmap,*/ /*GC gc,*/
						double x, double y, int c) = 0;
  // Draw character <c> at <x>,<y>.
  virtual void drawChar16(GfxState *state, BBitmap *bitmap, BView *view,
  						/*Pixmap pixmap,*/ /*GC gc,*/
						double x, double y, int c) = 0;

  // Does this font use hex char codes?
  GBool isHex() { return hex; }

protected:
  // rgb_color mFill, mStroke;     // stroke and fill color
  Ref id;						// font ID
  double tm11, tm12,				// original transform matrix
         tm21, tm22;
  //Display *display;				// X display
  GBool hex;						// subsetted font with hex char codes
								//   (this flag is used for text output)
};

#if HAVE_T1LIB_H
//------------------------------------------------------------------------
// BeOutputT1Font
//------------------------------------------------------------------------

class BeOutputT1Font: public BeOutputFont {
public:

  BeOutputT1Font(GfxFont *gfxFont, GString *pdfBaseFont,
				double m11, double m12, double m21, double m22,
				double size, double ntm11, double ntm12,
				double ntm21, double ntm22,
				/*Display *display,*/ BeOutputFontCache *cache);

  virtual ~BeOutputT1Font();

  // Was font created successfully?
  virtual GBool isOk();

  // Update <gc> with this font.
  //virtual void update(rgb_color fill, rgb_color stroke);

  // Draw character <c> at <x>,<y>.
  virtual void drawChar(GfxState *state, BBitmap *bitmap, BView *view,
  						/*Pixmap pixmap,*/ /*GC gc,*/
						double x, double y, int c);
  virtual void drawChar16(GfxState *state, BBitmap *bitmap, BView *view,
  						/*Pixmap pixmap,*/ /*GC gc,*/
						double x, double y, int c) {};

private:

  float size;						// font size
  int t1ID;						// t1lib font ID
  GBool t1libAA, t1libAAHigh;   // use Anti Aliasing
};
#endif

//------------------------------------------------------------------------
// BeOutputServerFont
//------------------------------------------------------------------------

class BeOutputServerFont: public BeOutputFont {
public:

  BeOutputServerFont(GfxFont *gfxFont, char *fontFamily, char *fontStyle,
				    BeFontEncoding *encoding,
				    double m11, double m12, double m21, double m22,
				    double size, double ntm11, double ntm12,
				    double ntm21, double ntm22,
				    /*Display *display,*/ BeOutputFontCache *cache);

  virtual ~BeOutputServerFont();

  // Was font created successfully?
  virtual GBool isOk();

  // Update <gc> with this font.
  //virtual void update(rgb_color fill, rgb_color stroke);

  // Draw character <c> at <x>,<y>.
  virtual void drawChar(GfxState *state, BBitmap *bitmap, BView *view,
  						/*Pixmap pixmap,*/ /*GC gc,*/
						double x, double y, int c);
  virtual void drawChar16(GfxState *state, BBitmap *bitmap, BView *view,
  						/*Pixmap pixmap,*/ /*GC gc,*/
						double x, double y, int c);

private:

  BFont *beFont;				// the X font
  Gushort map[256];				// forward map (PDF code -> font code)
  Guchar revMap[256];				// reverese map (font code -> PDF code)
  BeFontEncoding *encoding;
};

//------------------------------------------------------------------------
// BeOutputFontCache
//------------------------------------------------------------------------

#if HAVE_T1LIB_H
struct BeOutputT1BaseFont {
  int num, gen;
  int t1ID;
  char **enc;
  char *encStr;
};
#endif

class BeOutputFontCache {
public:

  // Constructor.
  BeOutputFontCache(/*Display *display*/);

  // Destructor.
  ~BeOutputFontCache();

  // Initialize (or re-initialize) the font cache for a new document.
  void startDoc(/*int screenNum,*/ Guint depth
				/*, Colormap colormap*/);

  // Get a font.  This creates a new font if necessary.
  BeOutputFont *getFont(GfxFont *gfxFont, double m11, double m12,
				       double m21, double m22);

#if HAVE_T1LIB_H
  // Get a t1lib font.
  int getT1Font(GfxFont *gfxFont, GString *pdfBaseFont);

  // Use anti-aliased Type 1 fonts?
  GBool getT1libAA() { return t1libAA; }
  GBool getT1libAAHigh() { return t1libAAHigh; }
#endif

private:

  void delFonts();
  void clear();

  /*Display *display;*/				// X display pointer

#if HAVE_T1LIB_H
  GBool useT1lib;				// if false, t1lib is not used at all
  GBool t1libAA;				// true for anti-aliased fonts
  GBool t1libAAHigh;				// low or high-level anti-aliasing
  GBool t1Init;						// set when t1lib has been initialized
  BeOutputT1Font *				// Type 1 fonts in reverse-LRU order
    t1Fonts[t1FontCacheSize];
  int nT1Fonts;						// number of valid entries in t1Fonts[]
  BeOutputT1BaseFont *				// list of t1lib base fonts
    t1BaseFonts;
  int t1BaseFontsSize;				// size of t1BaseFonts array
#endif

  BeOutputServerFont *				// X server fonts in reverse-LRU order
    serverFonts[serverFontCacheSize];
  int nServerFonts;				// number of valid entries in serverFonts[]
};

#endif
