//-----------------------------------------------------------------------------
// File and Version Information
//      $Id: parser.cc,v 1.1 1999/01/23 02:30:33 wglp09 Exp $
//
// Description:
//      parser.cc.
//      Source file
//      A simple parser class.
// 
// Environment:
//	Software developed for the BaBar Detector at the SLAC B-Factory.
//
// Author List:
//      W.G.J. Langeveld, SLAC
//
// Modification History:
//      1998 - Langeveld : created
//-----------------------------------------------------------------------------
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/stat.h>

#include "parser.hh"

static unsigned long numerictype(char *, long);
static char *EscapeString(char *);

ParserToken::ParserToken(void)
{
   _next    = _previous = 0;
   _token    = _escaped = 0;
   length   = 0;
   type     = TK_NULL;
   line_pos = line_num = file_pos = 0;
   return;
}

ParserToken::~ParserToken()
{
   if (_token)   free(_token);
   if (_escaped) free(_escaped);
   return;
}

/**
*
*   Return pointer to the string in the current token
*
**/
char *ParserToken::get(void)
{
   return(_token);
}

/**
*
*   Get the current token and translate escapes.
*
**/
char *ParserToken::tostring(void)
{
   char *ptr = get();
   if (ptr && (_escaped == 0)) _escaped = EscapeString(ptr);
   return(_escaped);
}

char *ParserToken::typestring(void)
{
   switch (type) {
      case TK_NULL    : return("null token");
      case TK_ALPHA   : return("alphabetical token");
      case TK_INT     : return("integer");
      case TK_FLOAT   : return("floating point");
      case TK_NUM     : return("numeric token");
      case TK_ASCII   : return("ascii token");
      case TK_SUB     : return("sub-expression");
      case TK_STRING  : return("string");
      case TK_OP      : return("operator");
      case TK_LB      : return("left brace");
      case TK_RB      : return("right brace");
      case TK_SD      : return("string delimiter");
      case TK_COMMENT : return("start of comment");
      case TK_EOL     : return("end-of-line character");
      case TK_SEP     : return("separator");
      case TK_CONT    : return("continuation character");
   }
   return("unknown");
}

void ParserToken::dump(FILE *fp)
{
   char buff[132];

   dump(buff);

   if (fp == NULL) fp = stdout;
   fprintf(fp, "%s\n", buff);

   return;
}

void ParserToken::dump(char *buff)
{
   char s[32];
   long i, l;

   if (buff == NULL) return;

   strncpy(s, _token, 32);

   l = length;
   if (l > 16) {
      s[16] = s[17] = s[18] = '.';
      s[19] = '|';
   }
   else {
      s[l] = '|';
      for (i = l + 1; i < 20; i++) {
         s[i] = ' ';
      }
   }
   s[20] = '\0';

   for (i = 0; i < 20; i++) {
      if (s[i] < 32) s[i] = '?';
   }

   sprintf(buff, "|%s - %s", s, typestring());

   return;
}

/**
*
*   Match a string to the token
*   Case insensitive, abbreviations indicated by '*' in s.
*
**/
long ParserToken::match(char *s)
{
   char *t = _token;
   long flag;

   if (s == NULL) return(0);

   flag = 0;
   while (*s) {
      if (toupper(*s) - toupper(*t)) return(0);
      s++;
      t++;
      if (*s == '*') {
         flag = 1;
         s++;
      }
      if (flag && (*t == '\0')) return(1);
   }

   return(*t ? 0 : 1);
}

/**
*
*   Remove a token (but don't delete!)
*
**/
void ParserToken::remove(void)
{
   if (_previous) {
      _previous->_next = _next;
      _previous = 0;
   }
   if (_next) {
      _next->_previous = _previous;
      _next = 0;
   }
   return;
}

/**
*
*   Append a token to this one
*
**/
ParserToken *ParserToken::insert(char *s, long len,
                                 long t,  long fp,
                                 long ln, long lp)
{
   if (s == 0) return(0);

   ParserToken *pt = new ParserToken;
   if (pt == 0) return(0);

   pt->_token = (char *) malloc(len + 1);
   if (pt->_token == 0) {
      delete pt;
      return(0);
   }
   else {
      strncpy(pt->_token, s, len);
      pt->_token[len] = '\0';
   }

   pt->length   = len;
   pt->type     = t;
   pt->file_pos = fp;
   pt->line_num = ln;
   pt->line_pos = lp;

   if (_next) _next->_previous = pt;
   pt->_next     = _next;
   pt->_previous = this;
   _next         = pt;

   return(pt);
}

/**
*
*   ParsedStream constructors, destructor
*
**/
ParsedStream::ParsedStream(FILE *fp)
{
   init();
   if (fp == NULL) return;

   struct stat st;
   if (fstat(fileno(fp), &st) == 0) {
      _cache = (char *) malloc(st.st_size + 1);
      if (_cache) {
         long l = fread(_cache, 1, st.st_size + 1, fp);
         _cache[l] = '\0';
      }
   }
   _cur_cache_ptr = _last_cache_ptr = _cache;
   return;
}

ParsedStream::ParsedStream(char *s)
{
   init();
   if (s) _cache = strdup(s);
   return;
}

ParsedStream::ParsedStream(ParserToken *t)
{
   init();
   if (t) _cache = strdup(t->get());
   _cur_cache_ptr = _last_cache_ptr = _cache;
   return;
}   

#define BM_INIT_SIZE 128

BookMark::BookMark(void)
{
   _head = _current = _tail = new ParserToken();
   _head->_token      = strdup("head");
   _head->length      = 4;
   return;
}

BookMark::BookMark(const BookMark &bm)
{
   _head    = bm._head;
   _current = bm._current;
   _tail    = bm._tail;
   return;
}
      
BookMark &BookMark::operator=(const BookMark &bm)
{
   if (this == &bm) return(*this);

   _head    = bm._head;
   _current = bm._current;
   _tail    = bm._tail;

   return(*this);
}
      
BookMark::~BookMark(void)
{
   return;
}
      

void ParsedStream::init(void)
{
   _bookmark          = new BookMark *[BM_INIT_SIZE];
   _bookmarkalloc     = BM_INIT_SIZE;
   _nbookmarks        = 0;

   _cur_line_num      = 0;
   _cur_line_pos      = 0;
   _cur_file_pos      = 0;
   _cur_tok_type      = TK_NULL;
   _cur_tok_len       = 0;

   _cache             = 0;

   _whitespace        = strdup(" \t\r");
   _eolchars          = strdup("\n");
   _continuationchars = strdup("\\");
   _operators         = strdup("");
   _leftbrackets      = strdup("<[{(");
   _rightbrackets     = strdup(">]})");
   _stringdelimiters  = strdup("\"\'");
   _commentchars      = strdup("#!");
   _separators        = strdup(";");

   return;
}

ParsedStream::~ParsedStream(void)
{
   ParserToken *tk;

   if (_cache)    free(_cache);
   if (_bookmark) {
      for (long i = 0; i < _nbookmarks; i++) delete _bookmark[i];
      delete [] _bookmark;
   }

   tk = _head;
   while (tk) {
      _head = tk->_next;
      tk->remove();
      delete tk;
      tk = _head;
   }

   _head = NULL;

   free(_whitespace);
   free(_eolchars);
   free(_continuationchars);
   free(_operators);
   free(_leftbrackets);
   free(_rightbrackets);
   free(_stringdelimiters);
   free(_commentchars);
   free(_separators);

   return;
}

/**
*
*   Insertion function
*
**/
ParserToken *ParsedStream::insert(char *s, long len,
                                  long t,  long fp,
                                  long ln, long lp)
{
   if ((s == 0) || (_head == 0)) return(0);
   ParserToken *tk = _current->insert(s, len, t, fp, ln, lp);
   if (tk && (tk->_next == NULL)) _tail = tk;
   return(tk);
}

ParserToken *ParsedStream::addhead(char *s, long len,
                                   long t,  long fp,
                                   long ln, long lp)
{
   if ((s == 0) || (_head == 0)) return(0);
   ParserToken *tk = _head->insert(s, len, t, fp, ln, lp);
   if (tk && (tk->_next == NULL)) _tail = tk;
   return(tk);
}

ParserToken *ParsedStream::addtail(char *s, long len,
                                   long t,  long fp,
                                   long ln, long lp)
{
   if ((s == 0) || (_head == 0)) return(0);
   ParserToken *tk = _tail->insert(s, len, t, fp, ln, lp);
   if (tk) _tail = tk;
   return(tk);
}

/**
*
*   Set parser preferences
*
**/
long ParsedStream::setwhitespace(char *s)
{
   if (_whitespace) free(_whitespace);
   _whitespace = strdup(s);
   return(_whitespace ? 1 : 0);
}

long ParsedStream::seteolchars(char *s)
{
   if (_eolchars) free(_eolchars);
   _eolchars = strdup(s);
   return(_eolchars ? 1 : 0);
}

long ParsedStream::setcontinuationchars(char *s)
{
   if (_continuationchars) free(_continuationchars);
   _continuationchars = strdup(s);
   return(_continuationchars ? 1 : 0);
}

long ParsedStream::setoperators(char *s)
{
   if (_operators) free(_operators);
   _operators = strdup(s);
   return(_operators ? 1 : 0);
}

long ParsedStream::setleftbrackets(char *s)
{
   if (_leftbrackets) free(_leftbrackets);
   _leftbrackets = strdup(s);
   return(_leftbrackets ? 1 : 0);
}

long ParsedStream::setrightbrackets(char *s)
{
   if (_rightbrackets) free(_rightbrackets);
   _rightbrackets = strdup(s);
   return(_rightbrackets ? 1 : 0);
}

long ParsedStream::setcommentchars(char *s)
{
   if (_commentchars) free(_commentchars);
   _commentchars = strdup(s);
   return(_commentchars ? 1 : 0);
}

long ParsedStream::setseparators(char *s)
{
   if (_separators) free(_separators);
   _separators = strdup(s);
   return(_separators ? 1 : 0);
}

long ParsedStream::setstringdelimiters(char *s)
{
   if (_stringdelimiters) free(_stringdelimiters);
   _stringdelimiters = strdup(s);
   return(_stringdelimiters ? 1 : 0);
}

/**
*
*   Manipulate bookmarks
*
**/
BookMark *ParsedStream::bookmark(void)
{
   if (_bookmark) _bookmark[_nbookmarks] = new BookMark(*this);

   _nbookmarks++;

   if (_nbookmarks == _bookmarkalloc) {
      BookMark **bm = new BookMark *[2 * _bookmarkalloc];
      if (bm == NULL) {
         _nbookmarks--;
         return(0);
      }

      _bookmarkalloc *= 2;

      for (int i = 0; i < _nbookmarks; i++) bm[i] = _bookmark[i];

      delete [] _bookmark;

      _bookmark = bm;
   }
   return(_bookmark[_nbookmarks - 1]);
}

ParserToken *ParsedStream::tobookmark(const BookMark *b)
{
   if (b == NULL) return(0);
   if ((_head != b->_head) || (_tail != b->_tail)) return(0);
   return(_current = b->_current);
}

void ParsedStream::deletebookmark(BookMark *b)
{
   long shift, i;
   for (shift = 0, i = 0; i < _nbookmarks - 1; i++) {
      if (_bookmark[i] == b) {
         delete b;
         shift = 1;
      }
      if (shift) _bookmark[i] = _bookmark[i + 1];
   }
   _nbookmarks--;
   _bookmark[_nbookmarks] = 0;
   return;
}

/**
*
*   Traverse the parsed stream one command at a time.
*
**/
ParserToken *BookMark::firstline(void)
{
   ParserToken *tk = _current = _head;
   while (tk) {
      tk = next(TK_ANY);
      if (tk == NULL) break;
      if ((tk->type & (TK_EOL | TK_SEP)) == 0) break;
   }
   if (tk == NULL) _current = _head;
   return(tk);
}

ParserToken *BookMark::nextline(void)
{
   ParserToken *tk = next(TK_EOL | TK_SEP);
   while (tk) {
      tk = next(TK_ANY);
      if (tk == NULL) break;
      if ((tk->type & (TK_EOL | TK_SEP)) == 0) break;
   }
   if (tk == NULL) _current = _tail;
   return(tk);
}

ParserToken *BookMark::previousline(void)
{
   ParserToken *tk = previous(TK_EOL | TK_SEP);
   while (tk) {
      tk = next(TK_ANY);
      if (tk == NULL) break;
      if ((tk->type & (TK_EOL | TK_SEP)) == 0) break;
   }
   if (tk == NULL) _current = _head;
   return(tk);
}

/**
*
*   Get the next token. This code returns 0 upon EOL,
*   except if it are requested as indicated in the include
*   mask.
*
**/
ParserToken *BookMark::first(unsigned long include)
{
   ParserToken *pt = _current;

   while (pt) {
      if (pt->type & include) {
         _current = pt;
         break;
      }
      else if (pt->type & (TK_EOL | TK_SEP)) { 
         pt = 0;
         break;
      }
      pt = pt->_next;
   }
   return(pt);
}

ParserToken *BookMark::current(void)
{
   return(_current);
}

ParserToken *BookMark::next(unsigned long include)
{
   ParserToken *pt = _current->_next;

   while (pt) {
      if (pt->type & include) {
         _current = pt;
         break;
      }
      else if (pt->type & (TK_EOL | TK_SEP)) { 
         pt = 0;
         break;
      }
      pt = pt->_next;
   }
   return(pt);
}

ParserToken *BookMark::previous(unsigned long include)
{
   ParserToken *pt = _current->_previous;

   while (pt) {
      if (pt->type & include) {
         _current = pt;
         break;
      }
      else if (pt->type & (TK_EOL | TK_SEP)) { 
         pt = 0;
         break;
      }
      pt = pt->_previous;
   }
   return(pt);
}

/**
*
*   Parse the cache with the current settings
*
**/
long ParsedStream::parse(void)
{
   if (_cache == 0) return(0);

   _cur_line_num = 1;
   _cur_line_pos = 1;
   _cur_file_pos = 1;

   _cur_cache_ptr = _last_cache_ptr = _cache;

   while (readtoken()) {
      switch (_cur_tok_type) {
         case TK_EOL :
            addtoken();
            _cur_line_num++;
            break;
         case TK_LB :
         case TK_SD :
            addtoken();
            findmatch();
            break;
         default:
            addtoken();
            break;
      }
   }
/*
*   Kludge to deal with files that do not have a linefeed at the end
*/
   if (_tail->type != TK_EOL) {
      addtail("\n", 1, TK_EOL, _tail->file_pos + 1, _tail->line_num,
              _tail->line_pos + 1);
   }

   return(1);
}

void ParsedStream::addtoken(void)
{
   addtail(_last_cache_ptr, _cur_tok_len, _cur_tok_type, _cur_file_pos, _cur_line_num,
           _cur_line_pos);
   return;
}

long ParsedStream::readtoken(void)
{
   long t;
   while (1) {
/*
*   Skip white space
*/
      t = 0;
      while (*_cur_cache_ptr && strchr(_whitespace, *_cur_cache_ptr)) {
         _cur_cache_ptr++;
         t++;
      }

      _cur_file_pos   += t;
      _cur_line_pos   += t;
      _last_cache_ptr = _cur_cache_ptr;
/*
*   If we're out of characters, there are no more tokens
*/
      if (*_cur_cache_ptr == '\0') return(0);
/*
*   Find the type of the current token
*/
      _cur_tok_type = TK_NULL;

      if      (strchr(_continuationchars, *_cur_cache_ptr)) {
         _cur_tok_type = TK_CONT;
      }
      else if (strchr(_commentchars,      *_cur_cache_ptr)) {
         _cur_tok_type = TK_COMMENT;
      }
/*
*   Check if we're at a continuation or comment character.
*   If not, quit this loop and handle the token.
*/
      if (_cur_tok_type == TK_NULL) break;
/*
*   Skip to the end of the line.
*/
      t = 0;
      while (*_cur_cache_ptr && (strchr(_eolchars, *_cur_cache_ptr) == 0)) {
         _cur_cache_ptr++;
         t++;
      }

      _cur_file_pos   += t;
      _cur_line_pos   += t;
      _last_cache_ptr = _cur_cache_ptr;
/*
*   If we're out of characters, there are no more tokens
*/
      if (*_cur_cache_ptr == '\0') return(0);
/*
*   Found an eol. If we had a continuation character before, ignore
*   the eol and loop to read more white space.
*/
      _cur_line_num++;

      if (_cur_tok_type == TK_CONT) {
         _cur_cache_ptr++;
         _last_cache_ptr = _cur_cache_ptr;

         _cur_line_pos = 1;
         _cur_file_pos++;
      }
      else {
         _cur_tok_type = TK_EOL;
         break;
      }
   }
/*
*   We must have a real token, either not yet determined, or EOL.
*/
   if (_cur_tok_type == TK_NULL) {
      if      (strchr(_operators,         *_cur_cache_ptr)) {
         _cur_tok_type = TK_OP;
      }
      else if (strchr(_leftbrackets,      *_cur_cache_ptr)) {
         _cur_tok_type = TK_LB;
      }
      else if (strchr(_rightbrackets,     *_cur_cache_ptr)) {
         _cur_tok_type = TK_RB;
      }
      else if (strchr(_stringdelimiters,  *_cur_cache_ptr)) {
         _cur_tok_type = TK_SD;
      }
      else if (strchr(_separators,        *_cur_cache_ptr)) {
         _cur_tok_type = TK_SEP;
      }
      else if (strchr(_eolchars,          *_cur_cache_ptr)) {
         _cur_tok_type = TK_EOL;
      }
   }

   if (_cur_tok_type != TK_NULL) {
/*
*   We must have a one-character token of some sort
*   Set current cache pointer to next character
*/
      _cur_cache_ptr++;
      _cur_line_pos++;
      _cur_file_pos++;
      _cur_tok_len = 1;
      return(1);
   }
/*
*   We have a possibly multi-character token. Find the type.
*   Note that sub-expressions and strings are handled in
*   findmatch().
*/
   if (isalpha(*_cur_cache_ptr)) {
      _cur_tok_type = TK_ALPHA;
   }
   else {
      _cur_tok_type = TK_ASCII;
   }
/*
*   Loop until we find something else.
*/
   t = 0;
   while (1) {
      if ((*_cur_cache_ptr == '\0')                 ||
          strchr(_whitespace,       *_cur_cache_ptr) || 
          strchr(_operators,        *_cur_cache_ptr) ||
          strchr(_leftbrackets,     *_cur_cache_ptr) ||
          strchr(_rightbrackets,    *_cur_cache_ptr) ||
          strchr(_stringdelimiters, *_cur_cache_ptr) ||
          strchr(_commentchars,     *_cur_cache_ptr) ||
          strchr(_separators,       *_cur_cache_ptr) ||
          strchr(_eolchars,         *_cur_cache_ptr))  break;
      _cur_cache_ptr++;
      t++;
   }

   _cur_tok_len   = t;
   _cur_line_pos += t;
   _cur_file_pos += t;
/*
*   Check if the token (ASCII or ALPHA) might be a number
*/
   if (_cur_tok_type & (TK_ASCII | TK_ALPHA)) {
      unsigned long tt = numerictype(_last_cache_ptr, _cur_tok_len);
      if (tt & TK_NUM) _cur_tok_type = tt;
   }

   return(t);
}

static unsigned long numerictype(char *s, long l)
{
   long i = 0, digitsbeforedot = 0;
/*
*   Numbers can start with unary + or - 
*/
   if ((s[i] == '-') || (s[i] == '+')) i++;
/*
*   Single + or - is not a number
*/
   if (i == l) return(TK_ASCII);
/*
*   Next may be digits
*/
   for ( ; i < l; i++) {
      if (!isdigit(s[i])) break;
      digitsbeforedot++;
   }
/*
*   If we encounter nothing else, quit
*/
   if (i == l) return(TK_INT);
/*
*   We did encounter something. This means the number must be a
*   float.
*   Handle numbers of type (XXX).XXX...(EXXX)
*/
   if (s[i] == '.') {
      i++;
/*
*   +. or -. is not a number
*/
      if (i == l) {
         if (digitsbeforedot == 0) return(TK_ASCII);
         else                      return(TK_FLOAT);
      }
/*
*   Next must be a number, unless we already had digits
*/
      if (digitsbeforedot == 0) {
         if (!isdigit(s[i])) return(TK_ASCII);
         i++;
      }
/*
*   Next come digits or E.
*/
      for ( ; i < l; i++) {
         if (toupper(s[i]) == 'E') goto checkexponent;
         if (!isdigit(s[i])) return(TK_ASCII);
      }
      return(TK_FLOAT);
   }
   else if (toupper(s[i]) != 'E') {
      return(TK_ASCII);
   }
/*
*   Numbers of type (XXX)(.XXX)EXXX or (XXX)(.XXX)eXXX
*/
checkexponent:
   i++;
/*
*   Exponents can have unary +/- ...
*/
   if ((s[i] == '+') || (s[i] == '-')) i++;
/*
*   ...and then only digits, but at least one.
*/
   if (!isdigit(s[i])) return(TK_ASCII);
   i++;

   for ( ; i < l; i++) {
      if (!isdigit(s[i])) return(TK_ASCII);
   }
   return(TK_FLOAT);
}

void ParsedStream::findmatch(void)
{
   long level = 0;
   long t = 0;
   long c1, c2, sub;
   char *b;

   sub = _cur_tok_type == TK_LB;

   c1 = *_last_cache_ptr;

   if (sub) b = strchr(_leftbrackets, c1);
   else     b = strchr(_stringdelimiters, c1);
/*
*   Start at sub expression
*/
   _last_cache_ptr = _cur_cache_ptr;

   if ((b == 0) || (*b == '\0')) {
/*
*   Can't happen.
*/
      if (*_cur_cache_ptr) _cur_cache_ptr++;
      return;
   }
/*
*   Matching character from the collection of right brackets:
*/
   if (sub) {
      c2 = *(char *) ((long) _rightbrackets + (long) b - (long) _leftbrackets);
   }
   else {
      c2 = c1;
   }
/*
*   Loop until we find it.
*/
   while (*_cur_cache_ptr && ((*_cur_cache_ptr != c2) || level)) {
      if (*_cur_cache_ptr == c1) level++;
/*
*   Note: no "else" in the next statement! If c1 == c2, we want no level.
*/
      if (*_cur_cache_ptr == c2) level--;

      if (strchr(_eolchars, *_cur_cache_ptr)) _cur_line_num++;

      _cur_cache_ptr++;
      t++;
   }

   if (sub) _cur_tok_type = TK_SUB;
   else     _cur_tok_type = TK_STRING;
   _cur_tok_len   = t;
   _cur_line_pos += t;
   _cur_file_pos += t;
/*
*   Add the subexpression (if not NULL)
*/
   if (t) addtoken();
/*
*   Add the matching bracket
*/
   _last_cache_ptr = _cur_cache_ptr;

   if (*_cur_cache_ptr) {
      _cur_cache_ptr++;

      if (sub) _cur_tok_type = TK_RB;
      else     _cur_tok_type = TK_SD;

      _cur_tok_len    = 1;
      _cur_line_pos++;
      _cur_file_pos++;

      addtoken();
   }
   return;
}

/**
*
*   Replaces ARP EscapeString
*
**/
static char *EscapeString(char *s)
{
   static char *hexchrs = "0123456789ABCDEF";
   char *t;
   long c, c2;

   if (s == 0) return(0);

   t = strdup(s);

   while (c = *s++) {
      if (c == '\\') {
         c = *s++;
         switch (toupper(c)) {
            case 0   :
               goto cleanup;
               break;
            case 'N' :
               c = '\n';
               break;
            case 'T' :
               c = '\t';
               break;
            case 'V' :
               c = '\v';
               break;
            case 'B' :
               c = '\b';
               break;
            case 'R' :
               c = '\r';
               break;
            case 'F' :
               c = '\f';
               break;
            case 'E' :
               c = 27;
               break;
            case 'X' :
               c = *s++;
               if (c == 0) goto cleanup;
               c = toupper(c);
               c = ((((long) strchr(hexchrs, c) -
                      (long) hexchrs             ) & 0xF) << 4) & 0xF0;

               c2 = *s++;
               if (c2 == 0) goto cleanup;
               c2 = toupper(c2);
               c |= ((long) strchr(hexchrs, c2) - (long) hexchrs) & 0xF;
               break;
            default :
               break;
         }
      }
      *t++ = (char) c;
   }

cleanup:
   *t = 0;
   return(t);
}

/**
*
*   Convert a character string to a long, allowing 0x... constructs.
*
**/
long axtol(char *s)
{
   if ((s[0] == '0') && (toupper(s[1]) == 'X')) {
      long r = 0, c, o;

      s += 2;
      while (c = toupper(*s)) {
         r *= 16;
         if (c < '0') return(r);
         if (c > '9') {
            if (c < 'A') return(r);
            if (c > 'F') return(r);
            o = 'A' - 10;
         }
         else {
            o = '0';
         }
         r += c - o;
         s++;
      }
      return(r);
   }
   else {
      return(atol(s));
   }
}

