/* indent:4  tabsize:8  font:fixed-width */


/*--------------------------------------------------------------------------
| --------------------------------------------------------------------------
|
|    CardPileView :NSView :NSResponder :NSObject
|
|    Instances of this class provide a visual representation of a pile of 
|    cards.  CardPileView uses CardPile as the underlying representation, 
|    but adds a visual aspect and user interaction.  Users can click on 
|    CardPileViews and drag cards from one view to another.  The decisions 
|    on when this is allowed and what happens as a result are left to the
|    CardPileView's delegate.
|
| ---------------------------------------------------------------------------
\------------------------------------------------------------------------- */

#import <AppKit/AppKit.h>
#import "card-defs.h"
#import "CardPile.h"


/*---------------------------------------------------------------------------
|
|    Card dimensions
|
\---------------------------------------------------------------------------*/

#define CS_CARDWIDTH		107
#define CS_CARDHEIGHT		168

#define CS_SMALLCARDWIDTH	80
#define CS_SMALLCARDHEIGHT	126

// The default for our maxVisibleCards method.
#if WIN32
#define CS_SHOWALL	2147483647
#else
// This used to be MAXINT but since Rhapsody doesn't define it
// this should suffice.
#define CS_SHOWALL	2147483647
#endif

/*---------------------------------------------------------------------------
|
|    Delegate method flags
|
\---------------------------------------------------------------------------*/

#define CS_CLICKED		(1 << 0)
#define CS_DRAGGED		(1 << 1)
#define CS_CANACCEPT	(1 << 2)
#define CS_ACCEPT		(1 << 3)
#define CS_REMOVED		(1 << 4)
#define CS_GETOFFSET	(1 << 5)
#define CS_DOUBLECLICKED (1 << 6)


@interface CardPileView : NSView
{
    float xOffset, yOffset;	/*" The current drawing offset between cards."*/
    CardPile* cardPile;		/*" The CardPile holding all the cards. "*/
    CardPile* dragCardPile;	/*" Used internally for dragging cards. "*/
    BOOL useDragCardPile;	/*" Used internally for dragging cards. "*/
    id delegate;			/*" The object's delegate. "*/
    int	delegateFlags;		/*" Used internally. "*/
    int	tag;				/*" The pile's tag. "*/
    NSColor* backgroundColor;	/*" Background color. "*/
    float backgroundGray;		/*" I'll be removing this. "*/
    BOOL drawOutline;			/*" Do we draw an outline around our view? "*/
    int	maxVisibleCards;		/*" The number of overlapping cards you'll see. "*/
	
    BOOL coversOthers;			/*" YES if this pile overlaps other piles. "*/
    CardPileView* coverPile1;	/*" A CardPileView overlapped by this pile. "*/
    CardPileView* coverPile2;	/*" A CardPileView overlapped by this pile. "*/
    CardPileView* coverPile3;	/*" A CardPileView overlapped by this pile. "*/
    CardPileView* coverPile4;	/*" A CardPileView overlapped by this pile. "*/
    NSBitmapImageRep* beneath;	/*" Used internally. "*/

    // Because of AppKit related bugs on Windows this needed to be added.
    // The problem is that -(void)draggedImage:(NSImage *)image endedAt:(NSPoint)screenPoint	
	// deposited:(BOOL)flag; always seems to have deposited == YES, even
    //  if we return a NO from prepareForDraggingOperation 
   CardPileView* currentCardView;
   

}	




/*" Class initialization "*/
+ (void) initialize;

/*" Creating/releasing "*/
- (id) initWithFrame:(NSRect)frameRect;
- (void) dealloc;

/*" Our visible cards "*/
- (void) setMaxVisibleCards:(int)cardsVisible;
- (int) maxVisibleCards;

/*" Accessors "*/
- (void) setDelegate:(id)anObject;
- (id) delegate;
- (CardPile*) cardPile;
//Kluge method below added because of problem with prepareForDragOperation:sender on NT
- (CardPileView *) currentCardView;
//Kluge method below added because of problem with prepareForDragOperation:sender on NT
- (void) setCurrentCardView: (CardPileView *) theView;

/*" Display attributes "*/
- (NSColor *) backgroundColor;
- (void) setBackgroundColor:(NSColor *)aColor;
- (void) setCardSize:(CardSize)aSize;
- (BOOL) willDrawOutline;
- (void) setDrawOutline:(BOOL)aFlag;
- (void) drawRect:(NSRect)theRect;


/*" Overlapping piles "*/
- (BOOL) pileCovered:sender;
- (BOOL) pileCoveredBy:(CardPileView*)aCardPileView;
- (void) setCoversOthers:(BOOL)doesCover;
- (void) setCoverPile:(int)offset to:(CardPileView*)aPile;
- (void) resetBacking:sender;

/*" Responder methods "*/
- (void) mouseDown:(NSEvent *)theEvent;

/*" Dragging methods "*/
- (unsigned int) draggingSourceOperationMaskForLocal:(BOOL)flag;
- (void) draggedImage:(NSImage *)image beganAt:(NSPoint)screenPoint;
- (void) draggedImage:(NSImage *)image endedAt:(NSPoint)screenPoint 
	deposited:(BOOL)flag;
- (unsigned int) draggingEntered:sender;
- (BOOL) prepareForDragOperation:sender;
- (BOOL) performDragOperation:sender;

/*" Utility "*/
- (Card*) findCardAtPoint:(NSPoint)thePoint;
- (NSRect) getRectForCard:(Card*)aCard;
- (void) drawDragCard:sender;
- (void) setTag:(int)theTag;
- (int) tag;

@end


/*---------------------------------------------------------------------------
| ---------------------------------------------------------------------------
|
|    CardPileViewDelegate
|
|    These methods may be implemented by a delegate of CardPileView
|
| ---------------------------------------------------------------------------
\---------------------------------------------------------------------------*/

@interface NSObject (CardPileViewDelegate)

- (void) clickedCard:(Card*)aCard in:(CardPileView*)aCardPileView;
- (void) doubleClickedCard:(Card*)aCard in:(CardPileView*)aCardPileView;
- (BOOL) draggedPile:(CardPile*)aCardPile from:(CardPileView*)aCardPileView;
- (BOOL) canAcceptPile:(CardPile*)aCardPile from:sender in:(CardPileView*)aCardPileView;
- (void) acceptPile:(CardPile*)aCardPile in:(CardPileView*)aCardPileView;
- (void) removedPile:(CardPile*)aCardPile from:(CardPileView*)aCardPileView;
- (void) getOffset:(float*)xOffset :(float*)yOffset forSize:(CardSize)aSize;

@end

