/* pref.c
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 * Copyright (c) 1992, 1993, 1994, 1995, 1996, 1997 by Mike Gleason, NCEMRSoft.
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, without modification,
 * are permitted.
 * 
 */

#include "syshdrs.h"

#include "progress.h"
#include "pref.h"
#include "util.h"

/* How many times they've run this program. */
int gNumProgramRuns = 0;

/* Firewall/proxy configuration parameters. */
int gFirewallType;
char gFirewallHost[64];
char gFirewallUser[32];
char gFirewallPass[32];
char gFirewallExceptionList[256];
unsigned int gFirewallPort;

/* Their $PAGER. */
char gPager[128];

/* These correspond to the various timeouts from LibNcFTP. */
int gConnTimeout, gXferTimeout, gCtrlTimeout;

/* Active or passive FTP?  (PORT or PASV?)  Or both? */
int gDataPortMode;

/* "Save a bookmark to this site before closing?" */
int gConfirmClose;

/* "Save your password with the bookmark?" */
int gSavePasswords;

/* Which meter to use. */
FTPProgressMeterProc gProgressMeter;

/* Do we need to save the prefs, or can we skip it? */
int gPrefsDirty = 0;

/* Fool gcc into thinking some parameters are used. */
static int gUnused = 0;

extern FTPLibraryInfo gLib;
extern FTPConnectionInfo gConn;
extern char gOurDirectoryPath[], gUser[];

PrefOpt gPrefOpts[] = {
	{ "anonopen",				PREFOBSELETE },
	{ "anonpass", 				SetAnonPass, 0 },
	{ "anon-password",			SetAnonPass, 1 },
	{ "blank-lines",			PREFOBSELETE },
	{ "confirm-close",			SetConfirmClose, 1 },
	{ "connect-timeout",			SetConnTimeout, 1 },
	{ "control-timeout",			SetCtrlTimeout, 1 },
	{ "logsize",				PREFOBSELETE },
	{ "maxbookmarks",			PREFOBSELETE },
	{ "pager",				SetPager, 1 },
	{ "passive",				SetPassive, 1 },
	{ "progress-meter",			SetProgressMeter, 1 },
	{ "remote-msgs", 			PREFOBSELETE },
	{ "restore-lcwd", 			PREFOBSELETE },
	{ "save-passwords",			SetSavePasswords, 1 },
	{ "show-trailing-space",		PREFOBSELETE },
	{ "startup-lcwd", 			PREFOBSELETE },
	{ "startup-msgs", 			PREFOBSELETE },
	{ "timeout",				PREFOBSELETE },
	{ "total-runs", 			PREFOBSELETE },
	{ "total-xfer-hundredths-of-seconds", 	PREFOBSELETE },
	{ "total-xfer-kbytes", 			PREFOBSELETE },
	{ "trace",				PREFOBSELETE },
	{ "utime",				PREFOBSELETE },
	{ "visual",				PREFOBSELETE },
	{ "xfer-timeout",			SetXferTimeout, 1 },
	{ NULL,					(PrefProc) 0, kPrefOptInvisible, },
};

int gNumPrefOpts = ((int)(sizeof(gPrefOpts) / sizeof(PrefOpt)) - 1);



void
SetAnonPass(int t, const char *const val, FILE *fp)
{
	gUnused = t;
	if (fp != NULL) {
		(void) fprintf(fp, "%s", gLib.defaultAnonPassword);
	} else {
		(void) STRNCPY(gLib.defaultAnonPassword, val);
	}
}	/* SetAnonPass */



void
SetConfirmClose(int t, const char *const val, FILE *fp)
{
	gUnused = t;
	if (fp != NULL) {
		(void) fprintf(fp, "%s", YESNO(gConfirmClose));
	} else {
		gConfirmClose = StrToBool(val);
	}
}	/* SetConfirmClose */



void
SetConnTimeout(int t, const char *const val, FILE *fp)
{
	gUnused = t;
	if (fp != NULL) {
		(void) fprintf(fp, "%d", gConnTimeout);
	} else {
		gConn.connTimeout = gConnTimeout = atoi(val);
	}
}	/* SetConnTimeout */



void
SetCtrlTimeout(int t, const char *const val, FILE *fp)
{
	gUnused = t;
	if (fp != NULL) {
		(void) fprintf(fp, "%d", gCtrlTimeout);
	} else {
		gConn.ctrlTimeout = gCtrlTimeout = atoi(val);
	}
}	/* SetCtrlTimeout */



void
SetPager(int t, const char *const val, FILE *fp)
{
	gUnused = t;
	if (fp != NULL) {
		(void) fprintf(fp, "%s", gPager);
	} else {
		(void) STRNCPY(gPager, val);
	}
}	/* SetPager */



void
SetPassive(int t, const char *const val, FILE *fp)
{
	gUnused = t;
	if (fp != NULL) {
		if (gDataPortMode == kSendPortMode) {
			(void) fprintf(fp, "%s", "off");
		} else if (gDataPortMode == kPassiveMode) {
			(void) fprintf(fp, "%s", "on");
		} else {
			(void) fprintf(fp, "%s", "optional");
		}
	} else {
		if (ISTREQ(val, "optional"))
			gDataPortMode = kFallBackToSendPortMode;
		else if (ISTREQ(val, "on"))
			gDataPortMode = kPassiveMode;
		else
			gDataPortMode = kSendPortMode;
		gConn.dataPortMode = gDataPortMode;
	}
}	/* SetPassive */



void
SetProgressMeter(int t, const char *const val, FILE *fp)
{
	gUnused = t;
	if (fp != NULL) {
		if (gProgressMeter == PrStatBar) {
			(void) fprintf(fp, "%s", "2 (statbar)");
		} else if (gProgressMeter == PrPhilBar) {
			(void) fprintf(fp, "%s", "1 (philbar)");
		} else {
			(void) fprintf(fp, "%s", "0 (simple)");
		}
	} else {
		if ((val[0] == '0') || (ISTRNEQ(val, "simple", 6)))
			gProgressMeter = PrSizeAndRateMeter;
		else if ((val[0] == '1') || (ISTRNEQ(val, "phil", 4)))
			gProgressMeter = PrPhilBar;
		else
			gProgressMeter = PrStatBar;
		gConn.progress = gProgressMeter;
	}
}	/* SetProgressMeter */



void
SetSavePasswords(int t, const char *const val, FILE *fp)
{
	gUnused = t;
	if (fp != NULL) {
		if (gSavePasswords < 0)
			(void) fprintf(fp, "%s", "ask");
		else
			(void) fprintf(fp, "%s", YESNO(gSavePasswords));
	} else {
		if (ISTREQ(val, "ask"))
			gSavePasswords = -1;
		else
			gSavePasswords = StrToBool(val);
	}
}	/* SetSavePasswords */



void
SetXferTimeout(int t, const char *const val, FILE *fp)
{
	gUnused = t;
	if (fp != NULL) {
		(void) fprintf(fp, "%d", gXferTimeout);
	} else {
		gConn.xferTimeout = gXferTimeout = atoi(val);
	}
}	/* SetXferTimeout */




static void
Show1(int t)
{
	PrefOpt *p = &gPrefOpts[t];

	(void) printf("%-30s ", p->varname);
	if (p->proc != (PrefProc) 0)
		(*p->proc)(t, NULL, stdout);
	(void) printf("\n");
}	/* Show1 */




/* Modify or display the program's configuration. */
void
Set(const char *const tok1, const char *const tok2)
{
	int t;

	if ((tok1 == NULL) || (ISTREQ(tok1, "all"))) {
		/* Show all. */
		for (t=0; t<gNumPrefOpts; t++) {
			if (gPrefOpts[t].visible == kPrefOptVisible)
				Show1(t);
		}
	} else if (tok2 == NULL) {
		/* Show one. */
		for (t=0; t<gNumPrefOpts; t++) {
			if (ISTREQ(tok1, gPrefOpts[t].varname)) {
				if (gPrefOpts[t].visible == kPrefOptObselete) {
					(void) printf("The \"%s\" option is obselete or not implemented.\n", tok2);
				} else {
					Show1(t);
				}
				break;
			}
		}
		if (t >= gNumPrefOpts) {
			(void) printf("Unknown option \"%s\" -- try \"show all\" to list available options.\n", tok1);
		}
	} else {
		/* Set one. */
		for (t=0; t<gNumPrefOpts; t++) {
			if (ISTREQ(tok1, gPrefOpts[t].varname)) {
				if (gPrefOpts[t].visible == kPrefOptObselete) {
					(void) printf("The \"%s\" option is obselete or not implemented.\n", tok2);
				} else if (gPrefOpts[t].proc != (PrefProc) 0) {
					(*gPrefOpts[t].proc)(t, tok2, NULL);
					gPrefsDirty++;
				}
				break;
			}
		}
		if (t >= gNumPrefOpts) {
			(void) printf("Unknown option \"%s\" -- try \"show all\" to list available options.\n", tok1);
		}
	}
}	/* Set */




/* Read the saved configuration settings from a preferences file. */
void
LoadPrefs(void)
{
	FILE *fp;
	char pathName[256];
	char line[256];
	char *tok1, *tok2;
	int t;

	if (gOurDirectoryPath[0] == '\0')
		return;		/* Don't create in root directory. */
	(void) OurDirectoryPath(pathName, sizeof(pathName), kPrefFileName);

	fp = fopen(pathName, "r");
	if (fp == NULL) {
		/* Write a new one when we're done. */
		gPrefsDirty++;
	} else {
		/* Opened the preferences file. */
		line[sizeof(line) - 1] = '\0';
		while (fgets(line, sizeof(line) - 1, fp) != NULL) {
			tok1 = strtok(line, " =\t\r\n");
			if ((tok1 == NULL) || (tok1[0] == '#'))
				continue;
			tok2 = strtok(NULL, "\r\n");
			if (tok2 == NULL)
				continue;

			for (t=0; t<gNumPrefOpts; t++) {
				if (ISTREQ(tok1, gPrefOpts[t].varname)) {
					if (gPrefOpts[t].visible == kPrefOptObselete) {
						/* Probably converting an
						 * old 2.4.2 file.
						 */
						gPrefsDirty++;
					} else if (gPrefOpts[t].proc != (PrefProc) 0) {
						(*gPrefOpts[t].proc)(t, tok2, NULL);
					}
				}
			}
		}
		(void) fclose(fp);
	}
}	/* LoadPrefs */




/* Initialize the configuration settings, in case the user does not set them. */
void
InitPrefs(void)
{
	char *tok1;

	/* Set default values. */
	gPager[0] = '\0';
	gXferTimeout = 3600;
	gConnTimeout = 20;
	gCtrlTimeout = 135;
	gDataPortMode = kFallBackToSendPortMode;
	gConfirmClose = 1;
	gSavePasswords = -1;
	gProgressMeter = PrStatBar;

	tok1 = getenv("PAGER");
	if ((tok1 != NULL) && (tok1[0] != '\0')) {
#ifdef HAVE_STRSTR
		/* I prefer "less", but it doesn't work well here
		 * because it clears the screen after it finishes,
		 * and the default at EOF is to stay in less
		 * instead of exiting.
		 */
		if (strstr(gPager, "less") == NULL)
			(void) STRNCPY(gPager, tok1);
#else
		(void) STRNCPY(gPager, tok1);
#endif
	} else {
		(void) STRNCPY(gPager, "more");
	}
}	/* InitPrefs */




/* After reading the preferences, do some additional initialization. */
void
PostInitPrefs(void)
{
	if (gLib.defaultAnonPassword[0] == '\0') {
		FTPInitializeAnonPassword(&gLib);
		gPrefsDirty++;
	}
}	/* PostInitPrefs */




/* Write the configuration settings to a preferences file. */
void
SavePrefs(void)
{
	char pathName[256];
	char pathName2[256];
	char tName[32];
	int t;
	FILE *fp;

	if (gPrefsDirty == 0)
		return;		/* Don't need to save -- no changes made. */

	(void) OurDirectoryPath(pathName, sizeof(pathName), kPrefFileName);
	if (gNumProgramRuns == 1) {
		/* Save the 2.4.2 prefs file, if it exists. */
		(void) STRNCPY(pathName2, pathName);
		(void) STRNCAT(pathName2, ".v2");
		(void) unlink(pathName2);
		(void) rename(pathName, pathName2);
	}

	(void) sprintf(tName, "tpref%06d", (int) getpid());
	(void) OurDirectoryPath(pathName2, sizeof(pathName2), tName);

	fp = fopen(pathName2, "w");
	if (fp == NULL) {
		perror("could not save preferences file");
	} else {
		(void) fprintf(fp, "%s", "# NcFTP 3 preferences file\n# This file is loaded and overwritten each time NcFTP is run.\n#\n");
		for (t=0; t<gNumPrefOpts; t++) {
			if (gPrefOpts[t].visible == kPrefOptVisible) {
				(void) fprintf(fp, "%s=", gPrefOpts[t].varname);
				(*gPrefOpts[t].proc)(t, NULL, fp);
				(void) fprintf(fp, "\n");
			}
		}
		(void) fclose(fp);
		if (rename(pathName2, pathName) < 0) {
			perror("could not finish saving preferences file");
			(void) unlink(pathName2);
		};
	}
}	/* SavePrefs */




/* Save a sample configuration file for the firewall/proxy setup. */
void
WriteDefaultFirewallPrefs(FILE *fp)
{
	char *cp;

	FTPInitializeOurHostName(&gLib);
	cp = strchr(gLib.ourHostName, '.');

		(void) fprintf(fp, "%s", "\
# NcFTP firewall preferences\n\
# ==========================\n\
#\n\
");

		(void) fprintf(fp, "%s", "\
# If you need to use a proxy for FTP, you can configure it below.\n\
# If you do not need one, leave the ``firewall-type'' variable set\n\
# to 0.  Any line that does not begin with the ``#'' character is\n\
# considered a configuration command line.\n\
");
		(void) fprintf(fp, "%s", "\
#\n\
# Types of firewalls:\n\
# ------------------\n\
#\n\
#    type 1:  Connect to firewall host, but send \"USER user@real.host.name\"\n\
#\n\
");
		(void) fprintf(fp, "%s", "\
#    type 2:  Connect to firewall, login with \"USER fwuser\" and\n\
#             \"PASS fwpassword\", and then \"USER user@real.host.name\"\n\
#\n\
#    type 3:  Connect to and login to firewall, and then use\n\
#             \"SITE real.host.name\", followed by the regular USER and PASS.\n\
#\n\
");
		(void) fprintf(fp, "%s", "\
#    type 4:  Connect to and login to firewall, and then use\n\
#             \"OPEN real.host.name\", followed by the regular USER and PASS.\n\
#\n\
#    type 5:  Connect to firewall host, but send\n\
#             \"USER user@FWuser@real.host.name\" and\n\
#             \"PASS pass@FWpass\" to login.\n\
#\n\
#    type 0:  Do NOT use a firewall (most users will choose this).\n\
#\n\
firewall-type=0\n\
#\n\
#\n\
#\n\
");
		(void) fprintf(fp, "%s", "\
# The ``firewall-host'' variable should be the IP address or hostname of\n\
# your firewall server machine.\n\
#\n\
");

	if (cp == NULL) {
		(void) fprintf(fp, "firewall-host=firewall.domain.com\n");
	} else {
		(void) fprintf(fp, "firewall-host=firewall%s\n", cp);
	}

		(void) fprintf(fp, "%s", "\
#\n\
#\n\
#\n\
# The ``firewall-user'' variable tells NcFTP what to use as the user ID\n\
# when it logs in to the firewall before connecting to the outside world.\n\
#\n\
");
		(void) fprintf(fp, "firewall-user=%s\n", gUser);
		(void) fprintf(fp, "%s", "\
#\n\
#\n\
#\n\
# The ``firewall-password'' variable is the password associated with\n\
# the firewall-user ID.  If you set this here, be sure to change the\n\
# permissions on this file so that no one (except the superuser) can\n\
# see your password.  You may also leave this commented out, and then\n\
# NcFTP will prompt you each time for the password.\n\
");
		(void) fprintf(fp, "%s", "\
#\n\
firewall-password=fwpass\n\
#\n\
#\n\
#\n\
# Your firewall may require you to connect to a non-standard port for\n\
# outside FTP services, instead of the internet standard port number (21).\n\
#\n\
firewall-port=21\n\
");
		(void) fprintf(fp, "%s", "\
#\n\
#\n\
#\n\
# You probably do not want to FTP to the firewall for hosts on your own\n\
# domain.  You can set ``firewall-exception-list'' to a list of domains\n\
# or hosts where the firewall should not be used.  For example, if your\n\
# domain was ``probe.net'' you could set this to ``.probe.net''.\n\
#\n\
");
		(void) fprintf(fp, "%s", "\
# If you leave this commented out, the default behavior is to attempt to\n\
# lookup the current domain, and exclude hosts for it.  Otherwise, set it\n\
# to a list of comma-delimited domains or hostnames.  The special token\n\
# ``localdomain'' is used for unqualified hostnames, so if you want hosts\n\
# without explicit domain names to avoid the firewall, be sure to include\n\
# that in your list.\n\
#\n\
");

	if (cp != NULL) {
		(void) fprintf(fp, "firewall-exception-list=%s,localhost,localdomain\n", cp);
	} else {
		(void) fprintf(fp, "firewall-exception-list=.probe.net,localhost,foo.bar.com,localdomain\n");
	}
}	/* CreateDefaultFirewallPrefs */




/* Load those options specific to the firewall/proxy settings.  These are
 * kept in a different file so that other programs can read it and not
 * have to worry about the other junk in the prefs file.
 */
void
LoadFirewallPrefs(void)
{
	FILE *fp;
	char pathName[256];
	char line[256];
	char *tok1, *tok2;
	int n;
	char *cp;

	if (gOurDirectoryPath[0] == '\0')
		return;		/* Don't create in root directory. */
	(void) OurDirectoryPath(pathName, sizeof(pathName), kFirewallPrefFileName);

	/* Set default values. */
	gFirewallType = kFirewallNotInUse;
	gFirewallPort = 0;
	gFirewallHost[0] = '\0';
	gFirewallUser[0] = '\0';
	gFirewallPass[0] = '\0';
	gFirewallExceptionList[0] = '\0';

	fp = fopen(pathName, "r");
	if (fp == NULL) {
		/* Create a blank one. */
		fp = fopen(pathName, "w");
		if (fp == NULL)
			return;
		WriteDefaultFirewallPrefs(fp);
		(void) fclose(fp);
		gNumProgramRuns = 1;
		(void) chmod(pathName, 00600);
	} else {
		/* Opened the firewall preferences file. */
		line[sizeof(line) - 1] = '\0';
		while (fgets(line, sizeof(line) - 1, fp) != NULL) {
			tok1 = strtok(line, " =\t\r\n");
			if ((tok1 == NULL) || (tok1[0] == '#'))
				continue;
			tok2 = strtok(NULL, "\r\n");
			if (tok2 == NULL)
				continue;
			if (ISTREQ(tok1, "firewall-type")) {
				n = atoi(tok2);
				if ((n > 0) && (n <= kFirewallLastType))
					gFirewallType = n;
			} else if (ISTREQ(tok1, "firewall-host")) {
				(void) STRNCPY(gFirewallHost, tok2);
			} else if (ISTREQ(tok1, "firewall-port")) {
				n = atoi(tok2);
				if (n > 0)
					gFirewallPort = (unsigned int) n;
			} else if (ISTREQ(tok1, "firewall-user")) {
				(void) STRNCPY(gFirewallUser, tok2);
			} else if (ISTREQ(tok1, "firewall-pass")) {
				(void) STRNCPY(gFirewallPass, tok2);
			} else if (ISTREQ(tok1, "firewall-password")) {
				(void) STRNCPY(gFirewallPass, tok2);
			} else if (ISTREQ(tok1, "firewall-exception-list")) {
				(void) STRNCPY(gFirewallExceptionList, tok2);
			}
		}
		(void) fclose(fp);
	}

	if (gFirewallExceptionList[0] == '\0') {
		FTPInitializeOurHostName(&gLib);
		cp = strchr(gLib.ourHostName, '.');

		if (cp != NULL) {
			(void) STRNCPY(gFirewallExceptionList, cp);
			(void) STRNCAT(gFirewallExceptionList, ",localdomain");
		}
	}
}	/* LoadFirewallPrefs */




/* This maintains the little counter file that is used by version 3.0
 * to do things based on how many times the program was run.
 */
void
CheckForNewV3User(void)
{
	FILE *fp;
	struct stat st;
	char pathName[256];
	char line[256];

	gNumProgramRuns = 0;

	/* Don't create in root directory. */
	if (gOurDirectoryPath[0] != '\0') {
		(void) OurDirectoryPath(pathName, sizeof(pathName), kFirstFileName);

		if ((stat(pathName, &st) < 0) && (errno == ENOENT)) {
			gNumProgramRuns = 1;
			gPrefsDirty++;

			/* Create a blank one. */
			fp = fopen(pathName, "w");
			if (fp == NULL)
				return;
			(void) fprintf(fp, "# NcFTP uses this file to mark that you have run it before, and that you do not\n# need any special first-time instructions or setup.\n#\nruns=%d\n", gNumProgramRuns);
			(void) fclose(fp);
		} else {
			fp = fopen(pathName, "r");
			if (fp != NULL) {
				while (fgets(line, sizeof(line) - 1, fp) != NULL) {
					if (strncmp(line, "runs=", 5) == 0) {
						(void) sscanf(line + 5, "%d",
							&gNumProgramRuns);
						break;
					}
				}
				(void) fclose(fp);
			}

			/* Increment the count of program runs. */
			gNumProgramRuns++;
			if (gNumProgramRuns == 1)
				gPrefsDirty++;

			/* Race condition between other ncftp processes.
			 * This isn't a big deal because this counter isn't
			 * critical.
			 */

			fp = fopen(pathName, "w");
			if (fp != NULL) {
				(void) fprintf(fp, "# NcFTP uses this file to mark that you have run it before, and that you do not\n# need any special first-time instructions or setup.\n#\nruns=%d\n", gNumProgramRuns);
				(void) fclose(fp);
			}
		}
	}
}	/* CheckForNewV3User */
