#define INCL_DOS
#define INCL_DOSERRORS
#include <os2.h>
#include <stdlib.h>
#include <stdio.h>

#include "uniaud.h"

typedef struct DumpFileHeader {
    char ID[10]; // "UNIMIXDUMP"
    unsigned char driver[16];	/* Driver name */
    int controls; // number of controls
} DumpFileHeader;

char DUMPID[10] = "UNIMIXDUMP";

int pcms_num[MAX_CARDS] = {0}; // number of pcm instances
int cards_num = 0; // number of cards

POSS32_DEVCAPS pcmcaps[MAX_CARDS] = {0}; // pointer to pcm caps

UniaudCardInfo *CardInfo[MAX_CARDS] = {0};


int SaveDump(int card_id, char *fname)
{
    DumpFileHeader dfh;
    int ctls = 0;
    int i = 0;
    UniaudControlValue *ctl_val = NULL;
    FILE *f = NULL;
    UniaudControl *pCtl = NULL;

    if (!fname) return -1;
    f = fopen(fname, "wb");
    if (!f) return -1;
    memcpy(&dfh.ID,&DUMPID,sizeof(DUMPID));
    memcpy(&dfh.driver,&CardInfo[card_id]->driver,sizeof(CardInfo[card_id]->driver));

    ctls = uniaud_mixer_get_ctls_number(card_id);
    dfh.controls = ctls;

    fwrite(&dfh,sizeof(dfh),1,f); // writing header
    pCtl = (UniaudControl * )uniaud_mixer_get_ctl_list(card_id);
    if (pCtl)
    {
        ctl_val = (UniaudControlValue *)malloc(sizeof(UniaudControlValue));
        for(i=1; i<=ctls; i++)
        {
            memset(ctl_val,0,sizeof(UniaudControlValue));
            uniaud_mixer_get_ctl_val(card_id, pCtl->numid, ctl_val);
            fwrite(ctl_val,sizeof(UniaudControlValue),1,f); // writing values
            pCtl++;
        } // for
    }
    if (ctl_val) free(ctl_val);
    fclose(f);
    return 0;
}

int LoadDump(int card_id, char *fname)
{
    DumpFileHeader dfh;
    int ctls = 0;
    int i = 0;
    UniaudControlValue *ctl_val = NULL;
    FILE *f = NULL;
    UniaudControl *pCtl = NULL;

    if (!fname) return -1;
    f = fopen(fname, "rb");
    if (!f) return -1;

    fread(&dfh, sizeof(dfh),1,f);
    if ((memcmp(&dfh.ID,&DUMPID,sizeof(DUMPID)) != 0) ||
        (memcmp(&dfh.driver,&CardInfo[card_id]->driver,sizeof(CardInfo[card_id]->driver)) != 0))
    {
        fclose(f);
        return -2;
    }

    ctls = uniaud_mixer_get_ctls_number(card_id);

    if (dfh.controls!=ctls)
    {
        // invalid controls count
        fclose(f);
        return -3;
    }

    pCtl = (UniaudControl * )uniaud_mixer_get_ctl_list(card_id);

    if (pCtl)
    {
        ctl_val = (UniaudControlValue *)malloc(sizeof(UniaudControlValue));
        for(i=1; i<=ctls; i++)
        {
            fread(ctl_val,sizeof(UniaudControlValue),1,f); // writing values
            if (pCtl->numid != ctl_val->id.numid) continue;
            if (uniaud_mixer_put_ctl_val(card_id, pCtl->numid, ctl_val))
            {
                // invalid controls count
                fclose(f);
                return -4;
            }
            pCtl++;
        } // for
    }
    if (ctl_val) free(ctl_val);
    fclose(f);
    return 0;
}


void Usage(void)
{
    printf("Usage:\n");
    printf(" -dev            - card to work\n");
    printf(" -card           - show card info\n");
    printf(" -pcms           - show PCM instances info\n");
    printf(" -list           - full (id,name,bounds,value) control(s) list\n");
    printf(" -names          - just control(s) name(s) and ID\n");
    printf(" -id<num>        - use control num (for list or set value). Begins from 1\n");
    printf(" -val<num>       - set value for control\n");
    printf(" -cnt<num>       - set value for count number <num> in control. Begins from 0\n");
    printf(" -get            - get value switch. uses with -id and/or -cnt\n");
    printf(" -powerget       - get power state\n");
    printf(" -powerset<num>  - set power state\n");
    printf(" -save<file>     - save dump of all values of all controls to file\n");
    printf(" -load<file>     - load dump from file to all values of all controls\n");
}

int ShowCardInfo(UniaudCardInfo *cInfo)
{
    if (!cInfo) return -1;

    printf("Card info:\n");
    printf("  num: %i\n",cInfo->card);
    printf("  id: %s\n",cInfo->id);
    printf("  driver: %s\n",cInfo->driver);
    printf("  name: %s\n",cInfo->name);
    printf("  longname: %s\n",cInfo->longname);
    printf("  mixer: %s\n",cInfo->mixername);
    printf("  componenets: %s\n",cInfo->components);
    return 0;
}

int ShowPCMInstanceInfo(POSS32_DEVCAPS pcaps, int pcms)
{
    int i;
    POSS32_DEVCAPS pcaps1 = pcaps;

    if (!pcaps || pcms <= 0) return -1;

    for (i=0; i<pcms;i++)
    {
        printf("PCM No %i info:\n", i);
        printf("PCM dev name: [%s] mixer [%s]\n",pcaps1->szDeviceName, pcaps1->szMixerName);
        printf(" PLAY\n");
        printf("  streams: %i\n",pcaps1->waveOutCaps.nrStreams);
        printf("  channels: from %i to %i\n",pcaps1->waveOutCaps.ulMinChannels,
               pcaps1->waveOutCaps.ulMaxChannels);
        printf("  rates: from %i to %i\n",pcaps1->waveOutCaps.ulMinRate,
               pcaps1->waveOutCaps.ulMaxRate);
        printf(" RECORD\n");
        printf("  streams: %i\n",pcaps1->waveInCaps.nrStreams);
        printf("  channels: from %i to %i\n",pcaps1->waveInCaps.ulMinChannels,
               pcaps1->waveInCaps.ulMaxChannels);
        printf("  rates: from %i to %i\n",pcaps1->waveInCaps.ulMinRate,
               pcaps1->waveInCaps.ulMaxRate);
        printf("\n");
        pcaps1++;
    }
    return 0;
}

int PrintControl(UniaudControlInfo *ctl_info, UniaudControlValue *ctl_val)
{
    int i;

    if (ctl_info == NULL || ctl_val == NULL) return -1;

    printf("Control ID: %i\n",ctl_info->id.numid);
    printf("Interface: ");
    switch(ctl_info->id.iface)
    {
    case SNDRV_CTL_ELEM_IFACE_CARD:
        printf("global control\n"); break;
    case SNDRV_CTL_ELEM_IFACE_HWDEP:
        printf("hardware dependent device\n");break;
    case SNDRV_CTL_ELEM_IFACE_MIXER:
        printf("virtual mixer device\n"); break;
    case SNDRV_CTL_ELEM_IFACE_PCM:
        printf("PCM device\n"); break;
    case SNDRV_CTL_ELEM_IFACE_RAWMIDI:
        printf("RawMidi device\n"); break;
    case SNDRV_CTL_ELEM_IFACE_TIMER:
        printf("timer device\n"); break;
    case SNDRV_CTL_ELEM_IFACE_SEQUENCER:
        printf("sequencer client\n"); break;
    }
    printf("Device(client): %i, subdevice (substream) %i\n",
           ctl_info->id.device, ctl_info->id.subdevice);
    printf("Name: [%s]\n",ctl_info->id.name);
    printf("Index: %i\n", ctl_info->id.index);
    printf("Element type: ");

    switch(ctl_info->type)
    {
    case SNDRV_CTL_ELEM_TYPE_NONE:
        printf("invalid\n");break;
    case SNDRV_CTL_ELEM_TYPE_BOOLEAN:
        printf("boolean type\n");break;
    case SNDRV_CTL_ELEM_TYPE_INTEGER:
        printf("integer type\n");break;
    case SNDRV_CTL_ELEM_TYPE_ENUMERATED:
        printf("enumerated type\n");break;
    case SNDRV_CTL_ELEM_TYPE_BYTES:
        printf("byte array\n");break;
    case SNDRV_CTL_ELEM_TYPE_IEC958:
        printf("IEC958 (S/PDIF) setup\n");break;
    case SNDRV_CTL_ELEM_TYPE_INTEGER64:
        printf("64-bit integer type\n");break;
    }

    printf("Count of values: %i\n", ctl_info->count);
    printf("Value: \n");
    switch(ctl_info->type)
    {
    case SNDRV_CTL_ELEM_TYPE_BOOLEAN:
    case SNDRV_CTL_ELEM_TYPE_INTEGER:
        printf(" Bounds: min: %i, max: %i, step: %i\n",
               ctl_info->value.integer.min,
               ctl_info->value.integer.max,
               ctl_info->value.integer.step);
        for (i=0; i < ctl_info->count; i++)
            printf("  value %d: %i\n",i+1, ctl_val->value.integer.value[i]);
        break;
    case SNDRV_CTL_ELEM_TYPE_INTEGER64:
        printf(" Bounds: min: %i, max: %i, step: %i\n",
               ctl_info->value.integer64.min,
               ctl_info->value.integer64.max,
               ctl_info->value.integer64.step);
        for (i=0; i < ctl_info->count; i++)
            printf("  value %d: %i\n",i+1, ctl_val->value.integer64.value[i]);
        break;
    case SNDRV_CTL_ELEM_TYPE_ENUMERATED:
        printf(" Bounds: items: %i, item: %i, name: [%s]\n",
               ctl_info->value.enumerated.items,
               ctl_info->value.enumerated.item,
               ctl_info->value.enumerated.name);
        for (i=0; i < ctl_info->count; i++)
            printf("  value %d: %i\n",i+1, ctl_val->value.enumerated.item[i]);
#if 0
        if (ctl_val->value.enumerated.item_ptr)
            printf("item [%s]\n", ctl_val->value.enumerated.item_ptr);
#endif
        break;
    case SNDRV_CTL_ELEM_TYPE_BYTES:
        break;
    case SNDRV_CTL_ELEM_TYPE_IEC958:
        break;
    }

    printf("\n");
    return 0;
}

int unimix_show_controls(int card_id, int ctl_id, int just_names)
{
    int ctls, i;
    UniaudControlValue *ctl_val = NULL;
    UniaudControlInfo *ctl_info = NULL;
    UniaudControl *pCtl = NULL;

    ctls = uniaud_mixer_get_ctls_number(card_id);

    pCtl = (UniaudControl * )uniaud_mixer_get_ctl_list(card_id);

    if (pCtl)
    {
        ctl_val = (UniaudControlValue *)malloc(sizeof(UniaudControlValue));
        ctl_info = (UniaudControlInfo *)malloc(sizeof(UniaudControlInfo));
        for(i=1; i<=ctls; i++)
        {
            memset(ctl_val,0,sizeof(UniaudControlValue));
            memset(ctl_info,0,sizeof(UniaudControlInfo));
            uniaud_mixer_get_ctl_info(card_id, pCtl->numid, ctl_info);
            uniaud_mixer_get_ctl_val(card_id, pCtl->numid, ctl_val);
            if ((ctl_id && pCtl->numid == ctl_id) ||
                (ctl_id == 0))
            {
                if (just_names)
                    printf("ID: %i, name: [%s]\n",pCtl->numid, pCtl->name);
                else
                    PrintControl(ctl_info, ctl_val);
            }
            pCtl++;
        }
        free(ctl_info);
        free(ctl_val);
    }

}

#define ACTION_LIST      1
#define ACTION_PUT       2
#define ACTION_GET       3
#define ACTION_SAVEDUMP  4
#define ACTION_LOADDUMP  5
#define ACTION_SHOWCARD  6
#define ACTION_SHOWPCMS  7
#define ACTION_SETPCM    8
#define ACTION_GETPOWER  9
#define ACTION_SETPOWER  10

int main (int argc, char *argv[])
{
    int ArgNum;
    ULONG ctl_id = 0;
    int action = 0; // nothing
    int ctl_val = 0;
    int ctl_cnt = 0;
    int just_names = 0;
    char *fname; // dump file name
    int ver;
    int channels,i;
    int card_id = 0;

    printf("Command line mixer for UNIAUD. Version 0.05\n");
    printf("Copyright 2004,2005 by Netlabs.org\n");
    printf("Written by Vlad Stelmahosky aka Vladest\n");

    ver = uniaud_get_version();

    switch (ver)
    {
    case -2:
        printf("Error: uniaud not detected\n");
        return 1;
        break;
    case -1:
        printf("Error: uniaud error\n");
        return 1;
        break;
    default:
        if (ver != 113)
        {
            printf("Error: unsupported version of uniaud\n");
            return 1;
        } else
            printf("Detected UNIAUD version %1d.%02d\n", ver / 100, ver % 100 );
    }

    cards_num = uniaud_get_cards();
    if (!cards_num)
    {
        printf("No audio cards detected\n");
        return 1;
    }

    printf("Detected %i audio adapter(s)\n", cards_num);

    for(i=0;i<cards_num;i++)
    {
        CardInfo[i] = malloc(sizeof(UniaudCardInfo));
        if (uniaud_get_card_info(i, CardInfo[i]) < 0)
        {
            printf("Error: unable to get card info\n");
            return 1;
        }
//        ShowCardInfo(CardInfo[i]);
        pcms_num[i] = uniaud_get_pcm_instances(i);
        if (pcms_num <= 0)
        {
            printf("Error: invalid number of PCM instances\n");
            return 1;
        }
        pcmcaps[i] = (POSS32_DEVCAPS)malloc(sizeof(OSS32_DEVCAPS)*pcms_num[i]);

        if (uniaud_pcm_get_caps(i, pcmcaps[i]) < 0)
        {
            printf("Error: unable to get PCM instances\n");
            return 1;
        }

    }
//     printf("control %i changed\n", uniaud_mixer_wait());

    //    printf("sz = %i\n", sizeof(struct sndrv_ctl_elem_info));
    if (argc < 2)
    {
        Usage();
        return 1;
    }

    for(ArgNum = 1; ArgNum < argc; ArgNum++)
    {
        if (strncmp(argv[ArgNum], "-dev", 4) == 0)
            card_id = atoi(argv[ArgNum] + 4);
        else if (strncmp(argv[ArgNum], "-id", 3) == 0)
            ctl_id = atoi(argv[ArgNum] + 3);
        else if (strncmp(argv[ArgNum], "-save", 5) == 0)
        {
            action = ACTION_SAVEDUMP;
            fname = argv[ArgNum] + 5;
        }
        else if (strncmp(argv[ArgNum], "-load", 5) == 0)
        {
            action = ACTION_LOADDUMP;
            fname = argv[ArgNum] + 5;
        }
        else if (strcmp(argv[ArgNum], "-list") == 0)
            action = ACTION_LIST;
        else if (strcmp(argv[ArgNum], "-card") == 0)
            action = ACTION_SHOWCARD;
        else if (strcmp(argv[ArgNum], "-pcms") == 0)
            action = ACTION_SHOWPCMS;
        else if (strcmp(argv[ArgNum], "-names") == 0)
        {
            action = ACTION_LIST;
            just_names = 1;
        }
        else if (strncmp(argv[ArgNum], "-val", 4) == 0)
        {
            action = ACTION_PUT;
            ctl_val = atoi(argv[ArgNum] + 4);
        }
        else if (strncmp(argv[ArgNum], "-cnt", 4) == 0)
            ctl_cnt = atoi(argv[ArgNum] + 4);
        else if (strncmp(argv[ArgNum], "-get", 4) == 0)
            action = ACTION_GET;
        else if (strncmp(argv[ArgNum], "-powerset", 9) == 0)
        {
            action = ACTION_SETPOWER;
            ctl_val = atoi(argv[ArgNum] + 4);
        }
        else if (strncmp(argv[ArgNum], "-powerget", 9) == 0)
        {
            action = ACTION_GETPOWER;
        }

    }

    switch(action)
    {
    case ACTION_LIST:
        unimix_show_controls(card_id, ctl_id, just_names);
        break;
    case ACTION_PUT:
        printf("Set ID: %i, count value: %i, value: %i\n",ctl_id, ctl_cnt, ctl_val);
        uniaud_mixer_put_value(card_id, ctl_id, ctl_val, ctl_cnt);
        break;
    case ACTION_GET:
        ctl_val = uniaud_mixer_get_value(card_id, ctl_id, ctl_cnt);
        printf("%i\n",ctl_val);
        ctl_val = 0;
        break;
    case ACTION_SETPOWER:
        printf("Set power state: %x\n",ctl_val);
        uniaud_mixer_set_power_state(card_id, &ctl_val);
        break;
    case ACTION_GETPOWER:
        uniaud_mixer_get_power_state(card_id, &ctl_val);
        printf("power state: %x\n",ctl_val);
        ctl_val = 0;
        break;
    case ACTION_SAVEDUMP:
        if (SaveDump(0, fname) < 0)
            printf("Error saving dump\n");
        break;
    case ACTION_LOADDUMP:
        if (LoadDump(0, fname) < 0)
            printf("Error loading dump\n");
        break;
    case ACTION_SHOWCARD:
        ShowCardInfo(CardInfo[card_id]);
        break;
    case ACTION_SHOWPCMS:
        ShowPCMInstanceInfo(pcmcaps[card_id], pcms_num[card_id]);
        break;
    case ACTION_SETPCM:
        if (uniaud_pcm_find_pcm_for_chan(pcmcaps[card_id], pcms_num[card_id], 0 /*play for now*/, channels) < 0)
            printf("Cannot find PCM for %i channels\n",channels);
        break;
    }
    return 0;
}
