// ----------------------------------------------------------------------------
//  M88 - PC-8801 series emulator
//  Copyright (C) cisc 1999.
// ----------------------------------------------------------------------------
//  Main (ALU)̎
// ----------------------------------------------------------------------------
//  $Id: memory.cpp,v 1.24 1999/07/22 15:55:12 cisc Exp $

//  MemoryPage size should be equal to or less than 0x400.

#include "headers.h"
#include "file.h"
#include "device.h"
#include "device_i.h"
#include "pc88/memory.h"
#include "pc88/config.h"
#include "pc88/crtc.h"
#include "error.h"
#include "status.h"

#ifndef __OS2__
using namespace std;
#endif
using namespace PC8801;

// ----------------------------------------------------------------------------
//  Constructor / Destructor
//
Memory::Memory(const ID& id)
  : Device(id), rom(0), ram(0), eram(0), tvram(0), bus(0),
    dicrom(0), cdbios(0), n80rom(0)
{
    txtwnd = 0;
    erambanks = 16;
    waitmode = 0;
    waittype = 0;
    enablewait = false;
}

Memory::~Memory()
{
    delete[] rom;
    delete[] ram;
    delete[] eram;
    delete[] tvram;
    delete[] dicrom;
    delete[] cdbios;
    delete[] n80rom;
}

bool Memory::Init(Bus* _bus, CRTC* _crtc)
{
    bus = _bus, crtc = _crtc;
//    assert(Bus::pagebits <= 10);
    if (Bus::pagebits > 10)
        return false;

    if (!InitMemory())
        return false;

    port31 = port32 = port34 = port35 = 0;
    port71 = 0xff;
    port5x = 3;
    port99 = 0;
    n80mode = 0;
    seldic = false;

    Update00R();
    Update00W();
    Update60R();
    Update80();
    UpdateC0();
    UpdateF0();
    return true;
}

// ----------------------------------------------------------------------------
//  Reset
//
void Memory::Reset(uint, uint)
{
    sw31 = bus->In(0x31);
    bool high = !(bus->In(0x6e) & 0x80);

    waitmode = (sw31 & 0x40 ? 12 : 0) + (high ? 24 : 0);
    n80mode = n80rom && (newmode == Config::N802);
    selgvram = true;
    port5x = 0;

    bus->SetReadMemorys(0x8000, 0x8000, ram+0x8000);
    if (!n80mode)
    {
        Update00R();
        Update00W();
        Update60R();
        Update80();
        UpdateC0();
        UpdateF0();
    }
    else
    {
        porte3 = 0;
        UpdateN80W();
        UpdateN80R();
        UpdateN80G();
    }
}

// ----------------------------------------------------------------------------
//          31 32 34 35 5x 70 71 78 e2 e4
//  00-5f(r)*                       *  *
//  60-7f(r)*  *              *     *  *
//  00-7f(w)                        *  *
//  80-83   *              *     *
//  84-bf
//  c0-ef      *  *  *  *
//  f0-ff      *  *  *  *
//

//  ram     text    gvram   gvramh
//  1   2   1   2   1   42  3   4   4MHz v1
//  0   0   0.5 0.5 1   2   1   2        v2
//  1   2   1   2   7   72  4   8   8MHz
//  1   1   2   2   3.5 5   4   5

//  Wait bank   4-s     4-h     8-s     8-h
//  disp        on  off on  off on  off on  off
//  normal/dma
//  00-bf       2   1   0   0   1   1   1   1
//  c0-ef       2   1   0   0   1   1   1   1
//  f0-ff       2   1   1   0   2   1   2   1
//
//  normal/nodma
//  00-bf       1   1   0   0   1   1   1   1
//  c0-ef       1   1   0   0   1   1   1   1
//  f0-ff       1   1   0   0   1   1   1   1
//
//  gvram
//  00-bf       2   1   0   0   1   1   1   1
//  c0-ef       42  1   2   1   72  7   5   3
//  f0-ff       42  1   2   1   72  7   5   3
//
//  00-bf       1   1   0   0   1   1   1   1
//  c0-ef       42  1   2   1   72  7   5   3
//  f0-ff       42  1   2   1   72  7   5   3
//
//  gvram(h)
//  00-bf       2   1   0   0   1   1   1   1
//  c0-ef       4   3   2   1   8   4   5   3
//  f0-ff       4   3   2   1   8   4   5   3
//
//  00-bf       1   1   0   0   1   1   1   1
//  c0-ef       4   3   2   1   8   4   5   3
//  f0-ff       4   3   2   1   8   4   5   3

const Memory::WaitDesc Memory::waittable[48] =
{
    { 2, 2, 2},{ 1, 1, 1},  { 1, 1, 1},{ 1, 1, 1},
    { 2,42,42},{ 1, 1, 1},  { 1,42,42},{ 1, 1, 1},
    { 2, 4, 4},{ 1, 3, 3},  { 1, 4, 4},{ 1, 3, 3},

    { 0, 0, 1},{ 0, 0, 0},  { 0, 0, 0},{ 0, 0, 0},
    { 0, 2, 2},{ 0, 1, 1},  { 0, 2, 2},{ 0, 1, 1},
    { 0, 2, 2},{ 0, 1, 1},  { 0, 2, 2},{ 0, 1, 1},

    { 1, 1, 2},{ 1, 1, 1},  { 1, 1, 1},{ 1, 1, 1},
    { 1,72,72},{ 1, 7, 7},  { 1,72,72},{ 1, 7, 7},
    { 1, 8, 8},{ 1, 4, 4},  { 1, 8, 8},{ 1, 4, 4},

    { 1, 1, 2},{ 1, 1, 1},  { 1, 1, 1},{ 1, 1, 1},
    { 1, 5, 5},{ 1, 3, 3},  { 1, 5, 5},{ 1, 3, 3},
    { 1, 5, 5},{ 1, 3, 3},  { 1, 5, 5},{ 1, 3, 3},
};

// ----------------------------------------------------------------------------
//  Port31
//  b2 b1
//  0  0    N88
//  1  0    N80
//  x  1    RAM
//
void Memory::Out31(uint, uint data)
{
    if (!n80mode)
    {
        if ((data ^ port31) & 6)
        {
            port31 = data & 6;
            Update00R();
            Update60R();
            Update80();
        }
    }
    else
    {
        if ((data ^ port31) & 2)
        {
            port31 = data & 2;
            UpdateN80R();
            UpdateN80W();
        }
    }
}

// ----------------------------------------------------------------------------
//  Port32
//  b5      ALU Enable (port5x=RAM)
//  b4      RAM/~TVRAM (V1S )
//  b1 b0   N88 EROM bank select
//
void Memory::Out32(uint, uint data)
{
    if (!n80mode)
    {
        uint mod = data ^ port32;
        port32 = data;
        if (mod & 0x03)
            Update60R();
        if (mod & 0x40)
            UpdateC0();
        if (mod & 0x50)
            UpdateF0();
    }
}

// ----------------------------------------------------------------------------
//  Port34  ALU Operation
//
void Memory::Out34(uint, uint data)
{
    if (!n80mode)
    {
        port34 = data;
        for (uint i=0; i<3; i++)
        {
            switch (data & 0x11)
            {
            case 0x00: maskr.byte[i] = 0xff; masks.byte[i] = 0x00; maski.byte[i] = 0x00; break;
            case 0x01: maskr.byte[i] = 0x00; masks.byte[i] = 0xff; maski.byte[i] = 0x00; break;
            case 0x10: maskr.byte[i] = 0x00; masks.byte[i] = 0x00; maski.byte[i] = 0xff; break;
            case 0x11: maskr.byte[i] = 0x00; masks.byte[i] = 0x00; maski.byte[i] = 0x00; break;
            }
            data >>= 1;
        }
    }
}

// ----------------------------------------------------------------------------
//  Port35
//  b7      ALU Enable?
//  b5-b4   ALU Write Control
//  b2-b0   ALU Read MUX
//
void Memory::Out35(uint, uint data)
{
    if (!n80mode)
    {
        port35 = data;
        aluread.byte[0] = (data & 1) ? 0xff : 0x00;
        aluread.byte[1] = (data & 2) ? 0xff : 0x00;
        aluread.byte[2] = (data & 4) ? 0xff : 0x00;

        if (data & 0x80)
        {
            port5x = 3;
        }
        UpdateC0();
        UpdateF0();
    }
}

// ----------------------------------------------------------------------------
//  Port5c  GVRAM0
//  port5d  GVRAM1
//  port5e  GVRAM2
//  port5f  RAM
//
void Memory::Out5x(uint bank, uint)
{
    if (!n80mode)
    {
        port5x = bank & 3;
        UpdateC0();
        UpdateF0();
    }
    else
    {
        if (!(uint(bank-0x5d) < 2))
        {
            port5x = bank & 3;
            UpdateN80G();
        }
    }
}

// ----------------------------------------------------------------------------
//  Port70  TextWindow (N88 L)
//
void Memory::Out70(uint, uint data)
{
    txtwnd = data * 0x100;
    if (port31 == 0)
    {
        Update80();
    }
}

// ----------------------------------------------------------------------------
//  Port71
//  b0      N88ROM/~EROM
//
void Memory::Out71(uint, uint data)
{
    if (!n80mode)
    {
        port71 = data;
        if (port31 == 0)
        {
            Update60R();
        }
    }
}

// ----------------------------------------------------------------------------
//  Port78  ++Port70
//
void Memory::Out78(uint, uint)
{
    txtwnd = (txtwnd + 0x100) & 0xff00;
    if (port31 == 0)
    {
        Update80();
    }
}

// ----------------------------------------------------------------------------
//  Port99
//  b4      CD-BIOS/other
//  b0      CD-EROM
//
void Memory::Out99(uint, uint data)
{
    if (cdbios)
    {
        port99 = data & 0x11;
        Update00R();
        Update60R();
    }
    else
    {
        port99 = 0;
    }
}

// ----------------------------------------------------------------------------
//  Porte2
//  b4      erom write enable (pe4 < ERAM Pages)
//  b0      erom read enable
//
void Memory::Oute2(uint, uint data)
{
    porte2 = data;
    if (!n80mode)
    {
        Update00R();
        Update60R();
        Update00W();
    }
    else
    {
        UpdateN80R();
        UpdateN80W();
    }
}

// ----------------------------------------------------------------------------
//  Porte4  erom page select
//
void Memory::Oute3(uint, uint data)
{
    porte3 = data;
    if (!n80mode)
    {
        Update00R();
        Update60R();
        Update00W();
    }
    else
    {
        UpdateN80R();
        UpdateN80W();
    }
}

// ----------------------------------------------------------------------------
//  Port F0 ROMoNI
//
void Memory::Outf0(uint, uint data)
{
    portf0 = data;
    if (dicrom)
    {
        UpdateC0();
        UpdateF0();
    }
}

// ----------------------------------------------------------------------------
//  Port F0 ROMoNI
//
void Memory::Outf1(uint, uint data)
{
    if (dicrom)
    {
        seldic = !(data & 1);
        UpdateC0();
        UpdateF0();
    }
}

// ----------------------------------------------------------------------------
//  In xx
//
uint Memory::In32(uint)
{
    return port32;
}

uint Memory::In5c(uint)
{
    const static uint res[4] = { 0xf9, 0xfa, 0xfc, 0xf8 };
    return res[port5x & 3];
}

uint Memory::In70(uint)
{
    return txtwnd >> 8;
}

uint Memory::In71(uint)
{
    return port71;
}

uint Memory::Ine2(uint)
{
    return (porte3 & 0xfc) ? 0xff : (~porte2) | 0xee;
}

uint Memory::Ine3(uint)
{
    return (porte3 & 0xfc) ? 0xff : porte3;
}

// ----------------------------------------------------------------------------
//          31 32 34 35 5x 70 71 78 e2 e3
//  00-5f(r)*                       *  *  *  *
//
void Memory::Update00R()
{
    uint8* read;

    if ((porte2 & 0x01) && (porte3 < erambanks))
    {
        read = &eram[porte3 * 0x8000];
    }
    else
    {
        read = ram;

        if (!(port31 & 2))
        {
            // ROM
            if (port99 & 0x10)
                read = cdbios;
            else
                read = rom + (port31 & 4 ? n80 : n88);
        }
    }
    if (r00 != read)
    {
        r00 = read;
        bus->SetReadMemorys(0, 0x6000, read);
    }
}

// ----------------------------------------------------------------------------
//  00-7f(r)
//
void Memory::UpdateN80R()
{
    uint8* read;

    if ((porte2 & 0x01) && (porte3 < erambanks))
        read = &eram[porte3 * 0x8000];
    else
        read = n80rom;

    if (r00 != read)
    {
        r00 = read; r60 = read+0x6000;
        bus->SetReadMemorys(0, 0x8000, read);
    }
}

// ----------------------------------------------------------------------------
//          31 32 34 35 5x 70 71 78 e2 e3
//  60-7f(r)*  *              *     *  *  *  *
//
void Memory::Update60R()
{
    uint8* read;
    if ((porte2 & 0x01) && (porte3 < erambanks))
    {
        read = &eram[porte3 * 0x8000] + 0x6000;
    }
    else
    {
        read = ram + 0x6000;

        if (port31 == 0)
        {
            if (port99 & 0x10)
            {
                if (port99 & 0x01)
                    read = cdbios + 0x8000 + 0x2000 * (port32 & 3);
                else
                    read = cdbios + 0x6000;
            }
            else
            {
                if (port71 & 1)
                    read = rom + n88 + 0x6000;
                else
                    read = rom + n88e + 0x2000 * (port32 & 3);
            }
        }
        else if (port31 == 4)
        {
            read = rom + n80 + 0x6000;
        }
    }
    if (r60 != read)
    {
        r60 = read;
        bus->SetReadMemorys(0x6000, 0x2000, read);
    }
}

// ---------------------------------------------------------------------------
//          31 32 34 35 5x 70 71 78 e2 e4
//  00-7f(w)                        *  *
//
void Memory::Update00W()
{
    uint8* write;

    if ((porte2 & 0x10) && (porte3 < erambanks))
    {
        write = &eram[porte3 * 0x8000];
    }
    else
    {
        write = ram;
    }

    if (w00 != write)
    {
        w00 = write;
        bus->SetWriteMemorys(0, 0x8000, write);
    }
}

// ---------------------------------------------------------------------------
//          31 32 34 35 5x 70 71 78 e2 e4
//  00-7f(w)   *                    *  *
//
void Memory::UpdateN80W()
{
    uint8* write;

    if (((porte2 & 0x10) || (port31 & 0x02)) && (porte3 < erambanks))
    {
        write = &eram[porte3 * 0x8000];
    }
    else
    {
        write = ram;
    }

    if (w00 != write)
    {
        w00 = write;
        bus->SetWriteMemorys(0, 0x8000, write);
    }
}

// ---------------------------------------------------------------------------
//          31 32 34 35 5x 70 71 78 e2 e4
//  80-83   *              *     *
//
void Memory::Update80()
{
    if (port31 != 0)
    {
        bus->SetMemorys(0x8000, 0x400, ram + 0x8000);
    }
    else
    {
        if (txtwnd <= 0xfc00)
            bus->SetMemorys(0x8000, 0x400, ram + txtwnd);
        else
            bus->SetFuncs(0x8000, 0x400, this, RdWindow, WrWindow);
    }
}

// ----------------------------------------------------------------------------
//  eLXgEBhE(bvAEh)̃ANZX
//
void MEMCALL Memory::WrWindow(void* inst, uint addr, uint data)
{
    Memory* m = static_cast<Memory*>(inst);
    m->ram[(m->txtwnd + (addr & 0x3ff)) & 0xffff] = data;
}

uint MEMCALL Memory::RdWindow(void* inst, uint addr)
{
    Memory* m = static_cast<Memory*>(inst);
    return m->ram[(m->txtwnd + (addr & 0x3ff)) & 0xffff];
}


void Memory::SelectJisyo()
{
    if (seldic)
    {
        uint8* mem = dicrom + (portf0 & 0x1f) * 0x4000;
        if (mem != rc0)
        {
            rc0 = mem;
            bus->SetReadMemorys(0xc000, 0x4000, rc0);
        }
    }
}

// ----------------------------------------------------------------------------
//          31 32 34 35 5x 70 71 78 e2 e4
//  c0-ef      *  *  *  *
//
void Memory::UpdateC0()
{
    // is gvram selected ?

    if (port32 & 0x40)
    {
        port5x = 3;
        if (port35 & 0x80)
        {
            rc0 = 0;
            SelectALU();
            SelectJisyo();
            return;
        }
    }
    else
    {
        if (port5x < 3)
        {
            rc0 = 0;
            SelectGVRAM();
            SelectJisyo();
            return;
        }
    }

    // Normal RAM ?
    if (selgvram)
    {
        selgvram = false;
        rc0 = ram + 0xc000;
        bus->SetMemorys(0xc000, 0x3000, ram+0xc000);
        waittype &= 3;
        SetWait();
    }

    if (seldic)
    {
        SelectJisyo();
        return;
    }

    if (rc0 != ram + 0xc000)
    {
        rc0 = ram + 0xc000;
        bus->SetReadMemorys(0xc000, 0x3000, ram+0xc000);
    }
}

// ----------------------------------------------------------------------------
//          31 32 34 35 5x 70 71 78 e2 e4
//  f0-ff      *  *  *  *
//
void Memory::UpdateF0()
{
    if (!selgvram && !seldic)
    {
        uint8* mem;

        if (!(port32 & 0x10) && (sw31 & 0x40))
            mem = tvram;
        else
            mem = ram + 0xf000;

        bus->SetMemorys(0xf000, 0x1000, mem);
    }
}

// ----------------------------------------------------------------------------
//          31 32 34 35 5x 70 71 78 e2 e4
//  80-ff                *
//
void Memory::UpdateN80G()
{
    // is gvram selected ?

    if (port5x < 3)
    {
        rc0 = 0;
        bus->SetFuncs(0x8000, 0x4000, this, RdGVRAM0, WrGVRAM0);
        selgvram = true;
        return;
    }
    // Normal RAM ?
    if (selgvram)
    {
        selgvram = false;
        bus->SetMemorys(0x8000, 0x4000, ram+0x8000);
        waittype &= 3;
        SetWait();
    }
}

// ----------------------------------------------------------------------------
//  Select GVRAM
//  port 5c-5e
//
void Memory::SelectGVRAM()
{
    static const MemoryBus::ReadFuncPtr rf[3] =
    {
        RdGVRAM0, RdGVRAM1, RdGVRAM2
    };

    static const MemoryBus::WriteFuncPtr wf[3] =
    {
        WrGVRAM0, WrGVRAM1, WrGVRAM2
    };

    bus->SetFuncs(0xc000, 0x4000, this, rf[port5x & 3], wf[port5x & 3]);

    selgvram = true;
}

// ----------------------------------------------------------------------------
//  GVRAM ̓ǂݏ
//
#define SETDIRTY(addr)  \
    if (m->dirty[addr >> 4]) return; \
    else m->dirty[addr >> 4] = 1;

void MEMCALL Memory::WrGVRAM0(void* inst, uint addr, uint data)
{
    Memory* m = static_cast<Memory*>(inst);
    m->gvram[addr &= 0x3fff].byte[0] = data;
    SETDIRTY(addr);
}

void MEMCALL Memory::WrGVRAM1(void* inst, uint addr, uint data)
{
    Memory* m = static_cast<Memory*>(inst);
    m->gvram[addr &= 0x3fff].byte[1] = data;
    SETDIRTY(addr);
}

void MEMCALL Memory::WrGVRAM2(void* inst, uint addr, uint data)
{
    Memory* m = static_cast<Memory*>(inst);
    m->gvram[addr &= 0x3fff].byte[2] = data;
    SETDIRTY(addr);
}

uint MEMCALL Memory::RdGVRAM0(void* inst, uint addr)
{
    Memory* m = static_cast<Memory*>(inst);
    return m->gvram[addr & 0x3fff].byte[0];
}

uint MEMCALL Memory::RdGVRAM1(void* inst, uint addr)
{
    Memory* m = static_cast<Memory*>(inst);
    return m->gvram[addr & 0x3fff].byte[1];
}

uint MEMCALL Memory::RdGVRAM2(void* inst, uint addr)
{
    Memory* m = static_cast<Memory*>(inst);
    return m->gvram[addr & 0x3fff].byte[2];
}

// ----------------------------------------------------------------------------
//  Select ALU
//
void Memory::SelectALU()
{
    static const MemoryBus::WriteFuncPtr funcs[4] =
    {
        WrALUSet,   WrALURGB,   WrALUB,     WrALUR
    };

    bus->SetFuncs(0xc000, 0x4000, this, RdALU, funcs[(port35 >> 4) & 3]);
    if (!selgvram)
    {
        selgvram = true;
        waittype = (waittype & 3) | (port40 ? 8 : 4);
        SetWait();
    }
}

// ----------------------------------------------------------------------------
//  GVRAM R/W thru ALU
//
uint MEMCALL Memory::RdALU(void* inst, uint addr)
{
    Memory* m = static_cast<Memory*>(inst);
    quadbyte q;
    m->alureg = m->gvram[addr & 0x3fff];
    q.pack = m->alureg.pack ^ m->aluread.pack;
    return ~(q.byte[0] | q.byte[1] | q.byte[2]) & 0xff;
}

void MEMCALL Memory::WrALUSet(void* inst, uint addr, uint data)
{
    Memory* m = static_cast<Memory*>(inst);

    quadbyte q;
//  data &= 255;
    q.pack = (data << 16) | (data << 8) | data;
    addr &= 0x3fff;
    m->gvram[addr].pack =
        ((m->gvram[addr].pack
        & ~(q.pack & m->maskr.pack))
         | (q.pack & m->masks.pack))
         ^ (q.pack & m->maski.pack);
    SETDIRTY(addr);
}

void MEMCALL Memory::WrALURGB(void* inst, uint addr, uint)
{
    Memory* m = static_cast<Memory*>(inst);
    m->gvram[addr &= 0x3fff] = m->alureg;
    SETDIRTY(addr);
}

void MEMCALL Memory::WrALUR(void* inst, uint addr, uint)
{
    Memory* m = static_cast<Memory*>(inst);
    m->gvram[addr &= 0x3fff].byte[1] = m->alureg.byte[0];
    SETDIRTY(addr);
}

void MEMCALL Memory::WrALUB(void* inst, uint addr, uint)
{
    Memory* m = static_cast<Memory*>(inst);
    m->gvram[addr &= 0x3fff].byte[0] = m->alureg.byte[1];
    SETDIRTY(addr);
}

// ----------------------------------------------------------------------------
//  ̊蓖Ă ROM ̓ǂݍ݁B
//
bool Memory::InitMemory()
{
    delete rom;     rom   = new uint8[romsize];
    delete ram;     ram   = new uint8[0x10000];
    delete eram;    eram  = new uint8[0x8000 * erambanks];
    delete tvram;   tvram = new uint8[0x1000];

    if (!(rom && ram && eram && tvram))
    {
        Error::SetError(Error::OutOfMemory);
        return false;
    }
    SetRAMPattern(ram, 0x10000);
    SetRAMPattern(eram, 0x8000*erambanks);
    memset(gvram, 0, sizeof(quadbyte) * 0x4000);
    memset(tvram, 0, 0x1000);

    bus->SetMemorys(0, 0x10000, ram);

    if (!LoadROM())
    {
        Error::SetError(Error::NoROM);
        return false;
    }
    return true;
}

// ----------------------------------------------------------------------------
//  K{łȂ ROM ǂݍ
//
void Memory::LoadOptROM(const char* name, uint8*& rom, int size)
{
    FileIO file;
    if (file.Open(name, FileIO::readonly))
    {
        file.Seek(0, FileIO::begin);
        delete[] rom;
        rom = new uint8[size];
        if (rom)
            file.Read(rom, size);
    }
}

// ----------------------------------------------------------------------------
//  ROM ǂݍ
//
bool Memory::LoadROM()
{
    FileIO file;

    LoadOptROM("jisyo.rom", dicrom, 512*1024);
    LoadOptROM("cdbios.rom", cdbios, 0x10000);
    LoadOptROM("n80_2.rom", n80rom, 0x8000);

    if (file.Open("pc88.rom", FileIO::readonly))
    {
        file.Seek(0, FileIO::begin);
        file.Read(rom + n88, 0x8000);
        file.Read(rom + n80 + 0x6000, 0x2000);
        file.Seek(0x2000, FileIO::current);
        file.Read(rom + n88e, 0x8000);
        file.Seek(0x2000, FileIO::current);
        file.Read(rom + n80, 0x6000);
        return true;
    }

    if (!LoadROMImage(rom+n88, "n88.rom", 0x8000))
        return false;
    LoadROMImage(rom+n80,         "n80.rom",   0x8000);
    LoadROMImage(rom+n88e,        "n88_0.rom", 0x2000);
    LoadROMImage(rom+n88e+0x2000, "n88_1.rom", 0x2000);
    LoadROMImage(rom+n88e+0x4000, "n88_2.rom", 0x2000);
    LoadROMImage(rom+n88e+0x6000, "n88_3.rom", 0x2000);
    return true;
}

bool Memory::LoadROMImage(uint8* dest, const char* filename, int size)
{
    FileIO file;
    if (!file.Open(filename, FileIO::readonly))
        return false;
    file.Read(dest, size);
    return true;
}

// ----------------------------------------------------------------------------
//  Np^[쐬 (SR ^)
//  arg:    ram     RAM GA
//          length  RAM GA (0x80 ̔{)
//
void Memory::SetRAMPattern(uint8* ram, uint length)
{
    for (int i=0; i<length; i+=0x80, ram+=0x80)
    {
        uint8 b = ((i >> 7) ^ i) & 0x100 ? 0x00 : 0xff;
        memset(ram,       b, 0x40);
        memset(ram+0x40, ~b, 0x40);
        ram[0x7f] = b;
    }
    ram[-1] = 0;
}

// ----------------------------------------------------------------------------
//  EFCg̍XV
//
void Memory::SetWait()
{
    if (enablewait)
    {
        const WaitDesc& wait = waittable[waitmode + waittype];
        bus->SetWaits(0x0000, 0xc000, wait.b0);
        bus->SetWaits(0xc000, 0x3000, wait.bc);
        bus->SetWaits(0xf000, 0x1000, wait.bf);
    }
}

// ----------------------------------------------------------------------------
//  VRTC(EFCgύX)
//
void Memory::VRTC(uint, uint d)
{
    waittype = (waittype & ~3) | ((d << 1) & 2) | (crtc->GetStatus() & 0x10 ? 0 : 1);
    SetWait();
}

// ----------------------------------------------------------------------------
//  Out40
//
void Memory::Out40(uint, uint data)
{
    data &= 0x10;
    if (data ^ port40)
    {
        port40 = data;
        if (selgvram)
        {
            waittype = (waittype & 3) | (port40 ? 8 : 4);
            SetWait();
        }
    }
}

// ---------------------------------------------------------------------------
//  ݒ蔽f
//
void Memory::ApplyConfig(const Config* cfg)
{
    enablewait = (cfg->flags & Config::enablewait) != 0;
    newmode = cfg->basicmode;
    if (!enablewait)
    {
        bus->SetWaits(0, 0x10000, 0);
    }
//  statusdisplay.Show(10, 0, "Wait:%d", enablewait);
}

// ---------------------------------------------------------------------------
//  Device descriptor
//
const Device::Descriptor Memory::descriptor =
{
    Memory::indef, Memory::outdef
};

const Device::OutFuncPtr Memory::outdef[] =
{
#ifndef __OS2__
    static_cast<OutFuncPtr> (Reset),
    static_cast<OutFuncPtr> (Out31),
    static_cast<OutFuncPtr> (Out32),
    static_cast<OutFuncPtr> (Out34),
    static_cast<OutFuncPtr> (Out35),
    static_cast<OutFuncPtr> (Out40),
    static_cast<OutFuncPtr> (Out5x),
    static_cast<OutFuncPtr> (Out70),
    static_cast<OutFuncPtr> (Out71),
    static_cast<OutFuncPtr> (Out78),
    static_cast<OutFuncPtr> (Out99),
    static_cast<OutFuncPtr> (Oute2),
    static_cast<OutFuncPtr> (Oute3),
    static_cast<OutFuncPtr> (Outf0),
    static_cast<OutFuncPtr> (Outf1),
    static_cast<OutFuncPtr> (VRTC),
#else
    (Device::OutFuncPtr) (Reset),
    (Device::OutFuncPtr) (Out31),
    (Device::OutFuncPtr) (Out32),
    (Device::OutFuncPtr) (Out34),
    (Device::OutFuncPtr) (Out35),
    (Device::OutFuncPtr) (Out40),
    (Device::OutFuncPtr) (Out5x),
    (Device::OutFuncPtr) (Out70),
    (Device::OutFuncPtr) (Out71),
    (Device::OutFuncPtr) (Out78),
    (Device::OutFuncPtr) (Out99),
    (Device::OutFuncPtr) (Oute2),
    (Device::OutFuncPtr) (Oute3),
    (Device::OutFuncPtr) (Outf0),
    (Device::OutFuncPtr) (Outf1),
    (Device::OutFuncPtr) (VRTC),
#endif
};

const Device::InFuncPtr Memory::indef[] =
{
#ifndef __OS2__
    static_cast<InFuncPtr> (In32),
    static_cast<InFuncPtr> (In5c),
    static_cast<InFuncPtr> (In70),
    static_cast<InFuncPtr> (In71),
    static_cast<InFuncPtr> (Ine2),
    static_cast<InFuncPtr> (Ine3),
#else
    (Device::InFuncPtr) (In32),
    (Device::InFuncPtr) (In5c),
    (Device::InFuncPtr) (In70),
    (Device::InFuncPtr) (In71),
    (Device::InFuncPtr) (Ine2),
    (Device::InFuncPtr) (Ine3),
#endif
};
