// ---------------------------------------------------------------------------
//  M88 - PC-8801 Series Emulator
//  Copyright (C) cisc 1999.
// ---------------------------------------------------------------------------
//  Implementation of USART(uPD8251AF)
// ---------------------------------------------------------------------------
//  $Id: sio.cpp,v 1.1 1999/04/02 13:32:26 cisc Exp $

#include "headers.h"
#include "schedule.h"
#include "sio.h"

#define LOGNAME "sio"
#include "diag.h"

using namespace PC8801;

// ---------------------------------------------------------------------------
//  \zEj
//
SIO::SIO(const ID& id)
: Device(id)
{
}

SIO::~SIO()
{
}

// ---------------------------------------------------------------------------
//  
//
bool SIO::Init(Bus* _bus, uint _prxrdy, Scheduler* s)
{
    bus = _bus, prxrdy = _prxrdy, scheduler = s;

    return true;
}

// ---------------------------------------------------------------------------
//  肹
//
void SIO::Reset(uint, uint)
{
    mode = clear;
    status = TXRDY | TXE;
    baseclock = 1200 * 64;
}

// ---------------------------------------------------------------------------
//  Ƃ[ہ[
//
void SIO::SetControl(uint, uint d)
{
    LOG1("[%.2x] ", d);

    switch (mode)
    {
    case clear:
        // Mode Instruction
        if (d & 3)
        {
            // Asynchronus mode
            mode = async;
            // b7 b6 b5 b4 b3 b2 b1 b0
            // STOP  EP PE CHAR  RATE
            static const int clockdiv[] = { 1, 1, 16, 64 };
            clock = baseclock / clockdiv[d & 3];
            datalen = 5 + ((d >> 2) & 3);
            parity = d & 0x10 ? (d & 0x20 ? even : odd) : none;
            stop = (d >> 6) & 3;
            LOG4("Async: %d baud, Parity:%c Data:%d Stop:%s\n", clock, parity, datalen, stop==3 ? "2" : stop==2 ? "1.5" : "1");
        }
        else
        {
            // Synchronus mode
            mode = sync1;
            parity = d & 0x10 ? (d & 0x20 ? even : odd) : none;
            LOG2("Sync: %d baud, Parity:%c / ", clock, parity);
        }
        break;

    case sync1:
        mode = sync2;
        break;

    case sync2:
        mode = sync;
        LOG0("\n");
        break;

    case async:
    case sync:
        // Command Instruction
        // b7 - enter hunt mode
        // b6 - internal reset
        if (d & 0x40)
        {
            // Reset!
            LOG0(" Internal Reset!\n");
            mode = clear;
            break;
        }
        // b5 - request to send
        // b4 - error reset
        if (d & 0x10)
        {
            LOG0(" ERRCLR");
            status &= ~(PE | OE | FE);
        }
        // b3 - send break charactor
        if (d & 8)
        {
            LOG0(" SNDBRK");
        }
        // b2 - receive enable
        rxen = (d & 4) != 0;
        // b1 - data terminal ready
        // b0 - send enable
        txen = (d & 1) != 0;

        LOG2(" RxE:%d TxE:%d\n", rxen, txen);
        break;
    default:
        LOG1("internal error? <%d>\n", mode);
        break;
    }
}

// ---------------------------------------------------------------------------
//  Ł[
//
void SIO::SetData(uint, uint d)
{
    LOG1(">%.2x ", d);
}

// ---------------------------------------------------------------------------
//  傤Ƃ
//
uint SIO::GetStatus(uint)
{
    return status;
}

// ---------------------------------------------------------------------------
//  Ł[Ƃ
//
uint SIO::GetData(uint)
{
    return 0xff;
}

// ---------------------------------------------------------------------------
//  device description
//
const Device::Descriptor SIO::descriptor = { indef, outdef };

const Device::OutFuncPtr SIO::outdef[] =
{
#ifndef __OS2__
    static_cast<OutFuncPtr> (Reset),
    static_cast<OutFuncPtr> (SetControl),
    static_cast<OutFuncPtr> (SetData),
#else
    (Device::OutFuncPtr) (Reset),
    (Device::OutFuncPtr) (SetControl),
    (Device::OutFuncPtr) (SetData),
#endif
};

const Device::InFuncPtr SIO::indef[] =
{
#ifndef __OS2__
    static_cast<InFuncPtr> (GetStatus),
    static_cast<InFuncPtr> (GetData),
#else
    (Device::InFuncPtr) (GetStatus),
    (Device::InFuncPtr) (GetData),
#endif
};

