/*
    File:       DragView.m
 
    Contains:	Main application code
 
    Written by: Michelle Wyner
 
    Created:    June 1997

    Copyright:  (c)1997 by Apple Computer, Inc., all rights reserved.

    Change History (most recent first):
		 March 1998 - HI modifications - DG
                 December 1997 - fixed problem when trying to drag blank image.
                 October 8, 1997 - disabled buttons at start.
                 August 15 1997 - bugs fixed.
                 June 1997 -- File first created.

    You may incorporate this sample code into your applications without
    restriction, though the sample code has been provided "AS IS" and the
    responsibility for its operation is 100% yours.  However, what you are
    not permitted to do is to redistribute the source as "DSC Sample Code"
    after having made changes. If you're going to re-distribute the source,
    we require that you make it clear in the source that the code was
    descended from Apple Sample Code, but that you've made changes.
*/

#import "DragView.h"
@implementation DragView

// standard initialization stuff, plus we must register what type of files we can drag.  
- (id)initWithFrame:(NSRect)theRect
{
    [super initWithFrame:theRect];
    [self setDragTypes:[NSArray arrayWithObject:NSFilenamesPboardType]];
    [self registerForDraggedTypes:dragTypes];
    filename = [@"" retain];
    return self;
}

- (void)dealloc
{
    [self setDragTypes:nil];
    [self setImage:nil];
    [filename release];
    [filenames release];
    [desiredType release];
    return [super dealloc];

}

- (void)awakeFromNib
{
        // set the "on state image" for the menu items
    [cascadingOut setOnStateImage:[NSImage imageNamed:@"NSMenuRadio"]];
    [urlOut setOnStateImage:[NSImage imageNamed:@"NSMenuRadio"]];
    [ufsOut setOnStateImage:[NSImage imageNamed:@"NSMenuRadio"]];

// set the initial state of the Cascading menu item to be on.

    [cascadingOut setState:NSStateOn];

    //setting which item is currently on 
    itemOn = cascadingOut; 
    [[self window] makeKeyAndOrderFront: nil];
}

// this method is overridden to tell the app what to do if something is dragged onto the application icon in the Workspace Manager.

- (BOOL)application:(NSApplication *)theApplication openFile:(NSString *)someFile
{
    [self setImage:[[NSWorkspace sharedWorkspace] iconForFile:someFile]];
    [self display];

    [self getPathInfo:someFile];
    return YES;
}

- (BOOL)applicationShouldTerminateAfterLastWindowClosed:(NSApplication *)theApplication
{
    return YES;
}

- (void)drawRect:(NSRect)theRect
{
    NSRect bounds;
    NSSize imageSize;

    //setting up the file well, and if needbe, draw the file image which was dragged into the well.
    bounds = [self bounds];
    NSDrawWhiteBezel(theRect, theRect);
    if (image) {
        imageSize = [image size];
        imageOrigin.x = (bounds.size.width - imageSize.width) / 2.0;
        imageOrigin.y = (bounds.size.height - imageSize.height) / 2.0;
        [image compositeToPoint:imageOrigin operation:NSCompositeSourceOver];

    }

}

- (void)mouseDown:(NSEvent *)theEvent
{
    NSSize mouseLoc;
    
    mouseLoc.width = 0;
    mouseLoc.height = 0;
    if (!(image==nil)) // if there is no image, we don't want to do a drag.
      [self dragImage:image at:imageOrigin offset:mouseLoc 			event:theEvent pasteboard:[NSPasteboard 			pasteboardWithName:NSDragPboard]				source:self 							slideBack:YES];
}

// this gets called when the user presses the "open" button in the application's window.

- (void)openDoc:(id)sender
{
    [[NSWorkspace sharedWorkspace] openFile:filename withApplication:@"" andDeactivate:YES];
}

// this gets called when the user presses the "reveal" button in the application's window.

- (void)revealDoc:(id)sender
{
    if ( [[NSWorkspace sharedWorkspace] selectFile:filename inFileViewerRootedAtPath:@"/"])
    {
       [NSApp deactivate];
    }
}		

- (void)setDragTypes:(NSArray *)newDragTypes
{
    [dragTypes autorelease];
    dragTypes = [newDragTypes retain];
}

- (void)setImage:(NSImage *)newImage
{
    [image autorelease];
    image = [newImage retain];
    [self setNeedsDisplay:YES];
}


- (unsigned int)draggingEntered:(id <NSDraggingInfo>)sender
{
    NSPasteboard *pasteboard;
    unsigned int sourceDragMask;
    
           // get the mask that the source is using
    if ([sender draggingSource] !=self) {	    
        sourceDragMask  = [sender draggingSourceOperationMask];
         // make sure our type of data is available
            pasteboard = [sender draggingPasteboard];
            desiredType = [[pasteboard availableTypeFromArray:dragTypes] 	retain];

            if (desiredType) {
        // we handle generic drags
                if (sourceDragMask & NSDragOperationGeneric) {
                    [self setImage:[sender draggedImage]];
                    return NSDragOperationGeneric;            
                }
            }
            return NSDragOperationNone;
    }
    return NSDragOperationNone;
}

- (BOOL)performDragOperation:(id <NSDraggingInfo>)sender
{
    NSPasteboard *pasteboard;
       
    pasteboard = [sender draggingPasteboard];
    //releasing the previous files which were dragged.
    [filenames release];
    filenames = [[pasteboard propertyListForType:desiredType] retain];
    return YES;
}

- (void)draggingExited:(id <NSDraggingInfo>)sender
{
     // user left without dropping
    
    if (([sender draggingSource] !=self)) {
        [self setImage:nil];
        [fileField setString:@""];
        [self setNeedsDisplay:YES];
        
// if no file was dragged in (such as the drag was cancelled),
  // we need to disable the buttons.
  //if one is enabled, then the other is also.

        if ([openButton isEnabled]) {
          [openButton setEnabled:NO];
          [revealButton setEnabled:NO];
        }
    }
}

- (unsigned int)draggingUpdated:(id <NSDraggingInfo>)sender
{	
    return [self draggingEntered:sender];
}

- (void)concludeDragOperation:(id <NSDraggingInfo>)sender
{
    if (([sender draggingSource] !=self)) {
        [self getPathInfo:nil];
    }
           
         
}

// this method absolutely must be implemented.
- (unsigned int)draggingSourceOperationMaskForLocal:(BOOL)flag
{
    //allows all types of dragging operations.
    return NSDragOperationAll;
}

// this is used for the cascading filename thing.
- (NSString *)indentLine:(NSString *)theString indentLevel:(unsigned int)numTabs
{
    NSMutableString *result = [[NSMutableString alloc]initWithFormat:@"%@\n", theString];
    //spaces are used instead of the \t character because the final string comes out way uneven otherwise.
    while ((numTabs--) > 0) {
        [result insertString:@"  " atIndex:0];
    }
    return [result autorelease];
}

- (void)viewChoice:(id)sender
{
          
    //what is the state of the thing currently selected?  If
    //it's already on, leave it on, if it's off, we need to find
    //the one that's on, and turn it off, and turn the new one on.

    if (sender != itemOn)
      {
        [sender setState:NSStateOn];
        [itemOn setState:NSStateOff];
        itemOn = sender;
          
      }
    [self whatToDo];
}

- (void)whatToDo
{

    //figuring out based on which item is currently selected what we should do.
    if (itemOn == ufsOut) {
        [self doUFSThing];
    }

    else if (itemOn == cascadingOut) {
        [self doCascadingThing];
    }

    else if (itemOn == urlOut) {
        [self doURLThing];

    }
}

//this gets the full path from the item which was dragged. 
- (void)getPathInfo:(NSString *)aFile
{
  if (aFile == nil) {
        filename = [filenames objectAtIndex:0];
  }
  else {
    filename = aFile;
  }

// we need to enable the buttons if they need it
 // if one is disabled, then the other is also, so we only need to check one

  if (![openButton isEnabled]) {
    [openButton setEnabled:YES];
    [revealButton setEnabled:YES];
    [openButton display];
    [revealButton display];
  }

 //now that we have the path, how should it be written?
    [self whatToDo];   
}

- (void)doUFSThing
{
    // since the original filename is already in UFS format, we don't
    // have to do anything special to the string.

    [fileField setString:filename];
       
}

// for use when the "Cascading" menu item is selected.
- (void)doCascadingThing
{
    NSArray *pathStuff = [filename pathComponents];
    NSEnumerator *enumerator = [pathStuff objectEnumerator];
    id object;
   
    unsigned int numTabs = 0;
    NSString *finalString = @"";

    while ((object = [enumerator nextObject])) {
        if (![@"/" isEqualToString:object]) {
            finalString =  [finalString stringByAppendingString:[self 			indentLine:object indentLevel:numTabs]];
            numTabs++;
        }
    }

    [fileField setString:finalString];
}

- (void)doURLThing
{
    NSMutableString *buildingString = [[NSMutableString alloc]init];
    NSString *fileString = @"file://";
    NSString *stringToFind = @" ";
    int	stringLength=0;

   // As spaces in URLs are changed to the ASCII character equivalent of "%20", we must do the same thing here.  We need to escape the '%' however else the system will think it's a formatting character.

    NSString *replacementString = @"\%20";
    NSRange foundRange;
    
    [buildingString setString:filename];
    [buildingString insertString:fileString atIndex:0];

    stringLength = [buildingString length];
    // since some of the files might have spaces in them, we need to take care of that situation.

    foundRange = [buildingString rangeOfString:stringToFind];

    //if the range isn't 0, then a space was found.
    //if the string has more than 1 space, we need to loop through until all the spaces have been replaced.
    while (foundRange.length != 0) {
        [buildingString replaceCharactersInRange:foundRange withString:replacementString];
       foundRange.length = stringLength - foundRange.location; 
       foundRange.location++;
     
       foundRange = [buildingString rangeOfString:stringToFind options:NSCaseInsensitiveSearch range:foundRange];

    }
   [fileField setString:buildingString];
    
}

@end
